package com.elitesland.tw.tw5.server.common.funConfig.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.common.funConfig.payload.BusinessTableFieldsPayload;
import com.elitesland.tw.tw5.api.common.funConfig.query.BusinessTableFieldsQuery;
import com.elitesland.tw.tw5.api.common.funConfig.vo.BusinessTableFieldsVO;
import com.elitesland.tw.tw5.server.common.funConfig.entity.BusinessTableFieldsDO;
import com.elitesland.tw.tw5.server.common.funConfig.entity.QBusinessTableDO;
import com.elitesland.tw.tw5.server.common.funConfig.entity.QBusinessTableFieldsDO;
import com.elitesland.tw.tw5.server.common.funConfig.repo.BusinessTableFieldsRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPADeleteClause;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 业务对象字段明细
 *
 * @author duwh
 * @date 2023-06-14
 */
@Repository
@RequiredArgsConstructor
public class BusinessTableFieldsDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BusinessTableFieldsRepo repo;
    private final QBusinessTableFieldsDO qdo = QBusinessTableFieldsDO.businessTableFieldsDO;
    private final QBusinessTableDO qTableDO = QBusinessTableDO.businessTableDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessTableFieldsVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BusinessTableFieldsVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 字段
                qdo.field,
                qdo.tableId,
                // 字段类型(数据库)
                qdo.type,
                // 字段key
                qdo.fieldKey,
                // 字段名称
                qdo.fieldName,
                // 是否业务字段
                qdo.businessFlag,
                // 是否必填
                qdo.notNull,
                // 显示名称
                qdo.showName,
                // 业务类型 INPUT 文本输入 SELECT 选择器
                qdo.showType,
                // 业务属性 JSON格式
                qdo.showAttr,
                // 默认值
                qdo.defaultValue,
                // 正则验证
                qdo.regexVerify,
                // 排序号
                qdo.sortNo,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessTableFieldsVO> getJpaQueryWhere(BusinessTableFieldsQuery query) {
        JPAQuery<BusinessTableFieldsVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BusinessTableFieldsQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BusinessTableFieldsQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 字段 精确 */
        if (!ObjectUtils.isEmpty(query.getField())) {
            list.add(qdo.field.like(SqlUtil.toSqlLikeString(query.getField())));
        }
        /** 字段 精确 */
        if (!ObjectUtils.isEmpty(query.getTableId())) {
            list.add(qdo.tableId.eq(query.getTableId()));
        }
        /** 字段类型(数据库) 精确 */
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 字段key 精确 */
        if (!ObjectUtils.isEmpty(query.getFieldKey())) {
            list.add(qdo.fieldKey.like(SqlUtil.toSqlLikeString(query.getFieldKey())));
        }
        /** 字段名称 精确 */
        if (!ObjectUtils.isEmpty(query.getFieldName())) {
            list.add(qdo.fieldName.like(SqlUtil.toSqlLikeString(query.getFieldName())));
        }
        /** 是否业务字段 精确 */
        if (!ObjectUtils.isEmpty(query.getBusinessFlag())) {
            list.add(qdo.businessFlag.eq(query.getBusinessFlag()));
        }
        /** 是否必填 精确 */
        if (!ObjectUtils.isEmpty(query.getNotNull())) {
            list.add(qdo.notNull.eq(query.getNotNull()));
        }
        /** 显示名称 精确 */
        if (!ObjectUtils.isEmpty(query.getShowName())) {
            list.add(qdo.showName.like(SqlUtil.toSqlLikeString(query.getShowName())));
        }
        /** 业务类型 INPUT 文本输入 SELECT 选择器 精确 */
        if (!ObjectUtils.isEmpty(query.getShowType())) {
            list.add(qdo.showType.eq(query.getShowType()));
        }
        /** 业务属性 JSON格式 精确 */
        if (!ObjectUtils.isEmpty(query.getShowAttr())) {
            list.add(qdo.showAttr.eq(query.getShowAttr()));
        }
        /** 默认值 精确 */
        if (!ObjectUtils.isEmpty(query.getDefaultValue())) {
            list.add(qdo.defaultValue.eq(query.getDefaultValue()));
        }
        /** 正则验证 精确 */
        if (!ObjectUtils.isEmpty(query.getRegexVerify())) {
            list.add(qdo.regexVerify.like(SqlUtil.toSqlLikeString(query.getRegexVerify())));
        }
        /** 排序号 精确 */
        if (!ObjectUtils.isEmpty(query.getSortNo())) {
            list.add(qdo.sortNo.eq(query.getSortNo()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
       if (!ObjectUtils.isEmpty(query.getExt3())) {
           list.add(qdo.ext3.eq(query.getExt3()));
       }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BusinessTableFieldsVO queryByKey(Long id) {
        JPAQuery<BusinessTableFieldsVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BusinessTableFieldsVO> queryListDynamic(BusinessTableFieldsQuery query) {
        JPAQuery<BusinessTableFieldsVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BusinessTableFieldsVO> queryPaging(BusinessTableFieldsQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<BusinessTableFieldsVO> jpaQuery = getJpaQueryWhere(query);
        List<BusinessTableFieldsVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BusinessTableFieldsVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BusinessTableFieldsDO save(BusinessTableFieldsDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BusinessTableFieldsDO> saveAll(List<BusinessTableFieldsDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BusinessTableFieldsPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 字段
        if (payload.getField() != null) {
            update.set(qdo.field, payload.getField());
        }
        // 字段
        if (payload.getTableId() != null) {
            update.set(qdo.tableId, payload.getTableId());
        }
        // 字段类型(数据库)
        if (payload.getType() != null) {
            update.set(qdo.type, payload.getType());
        }
        // 字段key
        if (payload.getFieldKey() != null) {
            update.set(qdo.fieldKey, payload.getFieldKey());
        }
        // 字段名称
        if (payload.getFieldName() != null) {
            update.set(qdo.fieldName, payload.getFieldName());
        }
        // 是否业务字段
        if (payload.getBusinessFlag() != null) {
            update.set(qdo.businessFlag, payload.getBusinessFlag());
        }
        // 是否必填
        if (payload.getNotNull() != null) {
            update.set(qdo.notNull, payload.getNotNull());
        }
        // 显示名称
        if (payload.getShowName() != null) {
            update.set(qdo.showName, payload.getShowName());
        }
        // 业务类型 INPUT 文本输入 SELECT 选择器
        if (payload.getShowType() != null) {
            update.set(qdo.showType, payload.getShowType());
        }
        // 业务属性 JSON格式
        if (payload.getShowAttr() != null) {
            update.set(qdo.showAttr, payload.getShowAttr());
        }
        // 默认值
        if (payload.getDefaultValue() != null) {
            update.set(qdo.defaultValue, payload.getDefaultValue());
        }
        // 正则验证
        if (payload.getRegexVerify() != null) {
            update.set(qdo.regexVerify, payload.getRegexVerify());
        }
        // 排序号
        if (payload.getSortNo() != null) {
            update.set(qdo.sortNo, payload.getSortNo());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 字段
            if (nullFields.contains("field")) {
                update.setNull(qdo.field);
            }
            // 字段类型(数据库)
            if (nullFields.contains("type")) {
                update.setNull(qdo.type);
            }
            // 字段key
            if (nullFields.contains("fieldKey")) {
                update.setNull(qdo.fieldKey);
            }
            // 字段名称
            if (nullFields.contains("fieldName")) {
                update.setNull(qdo.fieldName);
            }
            // 是否业务字段
            if (nullFields.contains("businessFlag")) {
                update.setNull(qdo.businessFlag);
            }
            // 是否必填
            if (nullFields.contains("notNull")) {
                update.setNull(qdo.notNull);
            }
            // 显示名称
            if (nullFields.contains("showName")) {
                update.setNull(qdo.showName);
            }
            // 业务类型 INPUT 文本输入 SELECT 选择器
            if (nullFields.contains("showType")) {
                update.setNull(qdo.showType);
            }
            // 业务属性 JSON格式
            if (nullFields.contains("showAttr")) {
                update.setNull(qdo.showAttr);
            }
            // 默认值
            if (nullFields.contains("defaultValue")) {
                update.setNull(qdo.defaultValue);
            }
            // 正则验证
            if (nullFields.contains("regexVerify")) {
                update.setNull(qdo.regexVerify);
            }
            // 排序号
            if (nullFields.contains("sortNo")) {
                update.setNull(qdo.sortNo);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoftCopy(List<Long> keys) {
        JPADeleteClause deleteClause = jpaQueryFactory.delete(qdo)
                .where(qdo.id.in(keys));
        return deleteClause.execute();
    }

    /**
     * 通过tableids 查字段
     *
     * @param tableIdSet
     * @return
     */
    public List<Long> queryListByTableIds(List<Long> tableIdSet) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdo.id).from(qdo).where(qdo.tableId.longValue().in(tableIdSet));
        return jpaQuery.fetch();
    }

    /**
     * tableName fieldName 查字段
     *
     * @param tableName 表名
     * @param fieldName 字段名
     * @return 字段配置
     */
    public  BusinessTableFieldsVO queryByTableNameAndFieldName(String tableName, String fieldName) {
        JPAQuery<BusinessTableFieldsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.leftJoin(qTableDO).on(qdo.tableId.eq(qTableDO.id));
        jpaQuery.where(qdo.field.eq(fieldName));
        jpaQuery.where(qTableDO.tableName.eq(tableName));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.where(qTableDO.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }
}

