package com.elitesland.tw.tw5.server.prd.acc.controller;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimJdePayload;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimPayload;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimSourcePayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimQuery;
import com.elitesland.tw.tw5.api.prd.acc.service.AccReimService;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimFinancialExportVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimPrintVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.excel.ExcelUtil;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimDocStatusEnum;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimPayModeEnum;
import com.elitesland.tw.tw5.server.prd.acc.convert.AccReimConvert;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.List;

/**
 * 费用报销管理
 *
 * @author sunxw
 * @date 2023-11-22
 */
@Api(tags = "费用报销管理")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/acc/accReim")
@Slf4j
public class AccReimController {

    private final AccReimService accReimService;
    private final UdcUtil udcUtil;

    /**
     * api/acc/reim-新增
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping
    @ApiOperation("新增")
    public TwOutputUtil<AccReimVO> insert(@RequestBody AccReimPayload payload) {
        return TwOutputUtil.ok(accReimService.insert(payload));
    }

    /**
     * api/acc/reim-部分更新
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping
    @ApiOperation("部分更新-修改收款账号")
    public TwOutputUtil partialUpdate(@RequestBody AccReimPayload payload) {
        accReimService.updateBankAccount(payload);
        return TwOutputUtil.ok();
    }

    /**
     * api/acc/reim-更新,支持置空,置空的字段需需要借助 protected List<String> nullFields;
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("/dynamic")
    @ApiOperation("动态更新")
    public TwOutputUtil<Long> updateByKeyDynamic(@RequestBody AccReimPayload payload) {
        return TwOutputUtil.ok(accReimService.updateByKeyDynamic(payload));
    }

    /**
     * 详情
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/{key}")
    @UdcNameClass
    @ApiOperation("详情")
    public TwOutputUtil<AccReimVO> queryOneByKey(@PathVariable Long key) {
        return TwOutputUtil.ok(accReimService.queryByKey(key));
    }

    /**
     * 分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/paging")
    @UdcNameClass
    @ApiOperation("分页")
    public TwOutputUtil<PagingVO<AccReimVO>> paging(AccReimQuery query) {
        return TwOutputUtil.ok(accReimService.queryPaging(query));
    }

    /**
     * 查询列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/list")
    // @UdcNameClass
    @ApiOperation("查询列表")
    public TwOutputUtil<List<AccReimVO>> queryList(AccReimQuery query) {
        return TwOutputUtil.ok(accReimService.queryListDynamic(query));
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping()
    @ApiOperation("逻辑删除")
    public TwOutputUtil deleteSoft(Long[] keys) {
        accReimService.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 根据报销单号查询唯一
     *
     * @param reimNo 报销单号
     * @return result
     */
    @GetMapping("/queryOneByReimNo/{reimNo}")
    @ApiOperation("根据报销单号查询唯一")
    public TwOutputUtil<AccReimVO> queryOneByReimNo(@PathVariable("reimNo") String reimNo) {
        AccReimVO vo = accReimService.queryOneByReimNo(reimNo);
        return TwOutputUtil.ok(vo);
    }

    /**
     * 财务收单
     *
     * @param reimNo
     * @return
     */
    @PutMapping("/chargeUp/{reimNo}")
    @ApiOperation("财务收单")
    public TwOutputUtil<AccReimVO> finChargeUp(@PathVariable("reimNo") String reimNo) {
        AccReimVO vo = accReimService.finChargeUp(reimNo);
        return TwOutputUtil.ok(vo);
    }

    /**
     * 修改付款方式
     *
     * @param keys
     * @param payMode
     * @return
     */
    @PutMapping("/payMode/{keys}")
    @ApiOperation("修改付款方式")
    public TwOutputUtil updatePayMode(@PathVariable("keys") Long[] keys, String payMode) {
        accReimService.updatePayMode(keys, payMode);
        return TwOutputUtil.ok();
    }

    /**
     * 生成批次号
     *
     * @param keys 报销当主键数组
     * @return
     */
    @GetMapping("/addBatchNo")
    @ApiOperation("修改付款方式")
    public TwOutputUtil addBatchNo(Long[] keys) {
        accReimService.addBatchNo(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 批量待记账/付款处理
     *
     * @param payload
     * @return
     */
    @PutMapping("/batchChargeOrPay")
    @ApiOperation("批量待记账/付款处理")
    public TwOutputUtil batchChargeOrPay(@RequestBody AccReimPayload payload) {
//        String jdeReason = accReimService.batchChargeOrPay(payload.getKeys(), payload.getType(),payload.getExpenseProofDigest(),payload.getExpenseAccountDate());
//        if(StringUtils.hasText(jdeReason)){
//            return TwOutputUtil.ok("",jdeReason,null);
//        }
        // 改为异步调用jde 增加一个状态 成功/失败的时候更新状态 因此不需要返回
        AccReimJdePayload accReimJdePayload = accReimService.batchChargeOrPay(payload.getKeys(), payload.getType(), payload.getExpenseProofDigest(), payload.getExpenseAccountDate(), AccReimPayModeEnum.PAY_MODE_2.getCode());
        return TwOutputUtil.ok();
    }

    /**
     * 费用计算(目前仅餐费)
     *
     * @param busItemCode
     * @return
     */
    @PutMapping("/calcAmt/{busItemCode}")
    @ApiOperation("费用计算(目前仅餐费)")
    public TwOutputUtil calcAmt(@PathVariable("busItemCode") String busItemCode, AccReimSourcePayload calcSource) {
        AccReimSourcePayload payload = accReimService.calcAmt(busItemCode, calcSource);
        return TwOutputUtil.ok(payload);
    }

    /**
     * 记账导出
     *
     * @param batchId
     * @return
     */
    @PutMapping("/export/charge/{batchId}")
    @ApiOperation("记账导出")
    public void exportChargeDtl(HttpServletResponse response, @PathVariable("batchId") Long batchId) {
        accReimService.exportChargeDtl(response, batchId);
    }

    /**
     * 付款导出
     *
     * @param batchId
     * @return
     */
    @PutMapping("/export/pay/{batchId}")
    @ApiOperation("付款导出")
    public void exportPayDtl(HttpServletResponse response, @PathVariable("batchId") Long batchId) {
        accReimService.exportPayData(response, batchId);
    }


    /**
     * 项目详情报销导出
     *
     * @param query
     * @return
     */
    @GetMapping("/exportProjectAccReim")
    @ApiOperation("项目详情报销导出")
    public void exportProjectAccReim(HttpServletResponse response, AccReimQuery query) throws IOException {
        List<AccReimVO> accReimVOS = accReimService.queryListDynamic(query);
        if (!CollectionUtils.isEmpty(accReimVOS)) {
            accReimVOS = udcUtil.translateList(accReimVOS);
            for (AccReimVO accReimVO : accReimVOS) {
                if (accReimVO.getAdjustAmt() == null) {
                    accReimVO.setAdjustAmt(accReimVO.getReimTotalAmt());
                }
            }
            //定义文件名称
            String sheetName = "报销导出";
            //对文件名进行固定格式编码
            String fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
            //设置请求响应内容类型
            //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            //设置请求响应内容编码方式
            response.setCharacterEncoding("utf-8");
            //文件下载，指定默认名
            response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

            final ExcelWriterSheetBuilder sheet = EasyExcel.write(response.getOutputStream(), AccReimVO.class)
                    .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                    .sheet(sheetName);
            // 列
            ExcelUtil.excelHelper(sheet, AccReimVO.class, null);
            //写入
            sheet.doWrite(accReimVOS);
        }

    }

    /**
     * 财务报销导出
     *
     * @param query
     * @return
     */
    @GetMapping("/exportFinancialAccReim")
    @ApiOperation("财务报销导出")
    public void exportFinancialAccReim(HttpServletResponse response, AccReimQuery query) throws IOException {
        List<AccReimVO> accReimVOS = accReimService.queryListDynamic(query);
        if (!CollectionUtils.isEmpty(accReimVOS)) {
            accReimVOS = udcUtil.translateList(accReimVOS);
            List<AccReimFinancialExportVO> accReimFinancialExportVOS = AccReimConvert.INSTANCE.voListToFinVoList(accReimVOS);
            //定义文件名称
            String sheetName = "报销导出";
            //对文件名进行固定格式编码
            String fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
            //设置请求响应内容类型
            //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            //设置请求响应内容编码方式
            response.setCharacterEncoding("utf-8");
            //文件下载，指定默认名
            response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

            final ExcelWriterSheetBuilder sheet = EasyExcel.write(response.getOutputStream(), AccReimFinancialExportVO.class)
                    .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                    .sheet(sheetName);
            // 列
            ExcelUtil.excelHelper(sheet, AccReimFinancialExportVO.class, null);
            //写入
            sheet.doWrite(accReimFinancialExportVOS);
        }

    }

    /**
     * 网银提交
     *
     * @param ids
     */
    @PostMapping("/payBankSubmit/{ids}")
    @ApiOperation("网银提交")
    public TwOutputUtil payBankSubmit(@PathVariable("ids") Long[] ids) {
        accReimService.payBankSubmit(ids);
        return TwOutputUtil.ok();
    }

    /**
     * 预算-费用占用分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/paging2Budget")
    @UdcNameClass
    @ApiOperation("预算-费用占用分页")
    public TwOutputUtil<PagingVO<AccReimVO>> paging2Budget(@RequestBody AccReimQuery query) {
        return TwOutputUtil.ok(accReimService.paging2Budget(query));
    }

    /**
     * 付款待记账分页查询
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */

    @GetMapping("/queryPendingPayChargePaging")
    @UdcNameClass
    @ApiOperation("付款待记账分页查询")
    public TwOutputUtil<PagingVO<AccReimVO>> queryPendingPayChargePaging(AccReimQuery query) {
        query.setReimStatus(AccReimDocStatusEnum.PENDING_PAYMENT_CHARGE.getCode());
        return TwOutputUtil.ok(accReimService.queryPaging(query));
    }

    /**
     * 付款待记账审批
     *
     * @param payload 单据
     * @return {@link TwOutputUtil}
     */

    @PostMapping("/batchPayCharge")
    @ApiOperation("付款待记账审批")
    public TwOutputUtil batchPayCharge(@RequestBody AccReimPayload payload) {
        if (CollectionUtils.isEmpty(payload.getKeys())) {
            throw TwException.error("", "请勾选数据后再提交");
        }
//        String jdeReason = accReimService.batchPayCharge(payload.getKeys(),payload.getPayProofDigest());
//        if(StringUtils.hasText(jdeReason)){
//            return TwOutputUtil.ok("",jdeReason,null);
//        }
        // 改为异步调用jde 增加一个状态 成功/失败的时候更新状态 因此不需要返回
        accReimService.batchPayCharge(payload.getKeys(), payload.getPayProofDigest());
        return TwOutputUtil.ok();
    }

    /**
     * 付款导出
     *
     * @param batchIdList
     * @return
     */
    @GetMapping("/print/pay")
    @ApiOperation("付款打印查询")
    public TwOutputUtil<AccReimPrintVO> exportPayDtl(Long[] batchIdList) {
        if (batchIdList == null || batchIdList.length == 0) {
            throw TwException.error("", "请勾选数据后再提交");
        }
        return TwOutputUtil.ok(accReimService.queryPrintPayData(Arrays.asList(batchIdList)));
    }

}

