package com.elitesland.tw.tw5.server.prd.acc.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimItemConfigRefPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimItemConfigRefQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimItemConfigRefVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimItemConfigRefDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimItemConfigRefDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimItemConfigRefRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 报销科目配置与核算项目管理
 *
 * @author sunxw
 * @date 2023-12-15
 */
@Repository
@RequiredArgsConstructor
public class AccReimItemConfigRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimItemConfigRefRepo repo;
    private final QAccReimItemConfigRefDO qdo = QAccReimItemConfigRefDO.accReimItemConfigRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimItemConfigRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimItemConfigRefVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 报销科目配置ID
                qdo.reimItemConfigId,
                // 核算项目
                qdo.busAccItemId,
                // 扩展字段1
                qdo.extString1,
                // 扩展字段2
                qdo.extString2,
                // 扩展字段3
                qdo.extString3
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimItemConfigRefVO> getJpaQueryWhere(AccReimItemConfigRefQuery query) {
        JPAQuery<AccReimItemConfigRefVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimItemConfigRefQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimItemConfigRefQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 报销科目配置ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimItemConfigId())) {
            list.add(qdo.reimItemConfigId.eq(query.getReimItemConfigId()));
        }
        /** 核算项目 精确 */
        if (!ObjectUtils.isEmpty(query.getBusAccItemId())) {
            list.add(qdo.busAccItemId.eq(query.getBusAccItemId()));
        }
        /** 扩展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString1())) {
            list.add(qdo.extString1.eq(query.getExtString1()));
        }
        /** 扩展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString2())) {
            list.add(qdo.extString2.eq(query.getExtString2()));
        }
        /** 扩展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString3())) {
            list.add(qdo.extString3.eq(query.getExtString3()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimItemConfigRefVO queryByKey(Long id) {
        JPAQuery<AccReimItemConfigRefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimItemConfigRefVO> queryListDynamic(AccReimItemConfigRefQuery query) {
        JPAQuery<AccReimItemConfigRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimItemConfigRefVO> queryPaging(AccReimItemConfigRefQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimItemConfigRefVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimItemConfigRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimItemConfigRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimItemConfigRefDO save(AccReimItemConfigRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimItemConfigRefDO> saveAll(List<AccReimItemConfigRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimItemConfigRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 报销科目配置ID
        if (payload.getReimItemConfigId() != null) {
            update.set(qdo.reimItemConfigId, payload.getReimItemConfigId());
        }
        // 核算项目
        if (payload.getBusAccItemId() != null) {
            update.set(qdo.busAccItemId, payload.getBusAccItemId());
        }
        // 扩展字段1
        if (payload.getExtString1() != null) {
            update.set(qdo.extString1, payload.getExtString1());
        }
        // 扩展字段2
        if (payload.getExtString2() != null) {
            update.set(qdo.extString2, payload.getExtString2());
        }
        // 扩展字段3
        if (payload.getExtString3() != null) {
            update.set(qdo.extString3, payload.getExtString3());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 报销科目配置ID
            if (nullFields.contains("reimItemConfigId")) {
                update.setNull(qdo.reimItemConfigId);
            }
            // 核算项目
            if (nullFields.contains("busAccItemId")) {
                update.setNull(qdo.busAccItemId);
            }
            // 扩展字段1
            if (nullFields.contains("extString1")) {
                update.setNull(qdo.extString1);
            }
            // 扩展字段2
            if (nullFields.contains("extString2")) {
                update.setNull(qdo.extString2);
            }
            // 扩展字段3
            if (nullFields.contains("extString3")) {
                update.setNull(qdo.extString3);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据报销科目配置ID逻辑删除
     *
     * @param reimConfigId 配置id
     * @return 删除的行数
     */
    public long delByReimConfigId(Long reimConfigId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.reimItemConfigId.eq(reimConfigId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

