package com.elitesland.tw.tw5.server.prd.acc.service;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimDetailPayload;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimPayload;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimRulePayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimQuery;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimRuleQuery;
import com.elitesland.tw.tw5.api.prd.acc.service.AccBusinessItemService;
import com.elitesland.tw.tw5.api.prd.acc.service.AccReimRuleService;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimDetailVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimRuleVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimVO;
import com.elitesland.tw.tw5.api.prd.inv.query.InvInvoiceQuery;
import com.elitesland.tw.tw5.api.prd.inv.service.InvInvoiceService;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvInvoiceVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimDocTypeEnum;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimRuleNoticeLevelEnum;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimRuleTypeEnum;
import com.elitesland.tw.tw5.server.prd.acc.convert.AccReimRuleConvert;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccReimDAO;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccReimDetailDAO;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccReimRuleDAO;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimRuleDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimRuleRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 报销规则管理
 *
 * @author sunxw
 * @date 2023-12-03
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AccReimRuleServiceImpl extends BaseServiceImpl implements AccReimRuleService {

    enum RuleCodeEnum {

        RULE001("RULE001"),
        RULE002("RULE002"),
        RULE003("RULE003"),
        RULE004("RULE004"),
        RULE005("RULE005"),
        RULE006("RULE006"),
        RULE007("RULE007"),
        RULE008("RULE008"),
        RULE009("RULE009"),
        RULE010("RULE010"),
        RULE011("RULE011"),
        RULE012("RULE012"),
        RULE013("RULE013"),
        RULE014("RULE014"),
        RULE015("RULE015"),
        ;
        private String ruleDesc;

        RuleCodeEnum(String ruleDesc) {
            this.ruleDesc = ruleDesc;
        }

        static RuleCodeEnum find(String ruleDesc) {
            for (RuleCodeEnum instance : RuleCodeEnum.values()) {
                if (ruleDesc.equals(instance.ruleDesc)) {
                    return instance;
                }
            }
            return null;
        }
    }

    private final AccReimRuleRepo repo;
    private final AccReimRuleDAO dao;
    private final AccBusinessItemService accBusinessItemService;
    private final AccReimDAO reimDAO;
    private final AccReimDetailDAO reimDetailDAO;
    private final InvInvoiceService invInvoiceService;
    private final CacheUtil cacheUtil;

    @Override
    public PagingVO<AccReimRuleVO> queryPaging(AccReimRuleQuery query) {
        return dao.queryPaging(query);
    }

    @Override
    public List<AccReimRuleVO> queryListDynamic(AccReimRuleQuery query) {
        return dao.queryListDynamic(query);
    }

    @Override
    public AccReimRuleVO queryByKey(Long key) {
        return dao.queryByKey(key);
    }

    @Override
    @Transactional
    public AccReimRuleVO insert(AccReimRulePayload payload) {
        AccReimRuleDO entityDo = AccReimRuleConvert.INSTANCE.toDo(payload);
        //生成规则编号
        /*String ruleNo = generateSeqNum("");
        entityDo.setRuleNo(ruleNo);*/
        return AccReimRuleConvert.INSTANCE.toVo(repo.save(entityDo));
    }

    @Override
    @Transactional
    public AccReimRuleVO update(AccReimRulePayload payload) {
        AccReimRuleDO entity = repo.findById(payload.getId()).orElseGet(AccReimRuleDO::new);
        Assert.notNull(entity.getId(), "不存在");
        AccReimRuleDO entityDo = AccReimRuleConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return AccReimRuleConvert.INSTANCE.toVo(repo.save(entity));
    }

    @Override
    @Transactional
    public long updateByKeyDynamic(AccReimRulePayload payload) {
        AccReimRuleDO entity = repo.findById(payload.getId()).orElseGet(AccReimRuleDO::new);
        Assert.notNull(entity.getId(), "不存在");
        long result = dao.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            dao.deleteSoft(keys);
        }
    }

    @Override
    public List<Map<String, String>> reimRuleCheck(AccReimPayload payload) {
        List<AccReimDetailPayload> details = payload.getDetails();
        if (ObjectUtils.isEmpty(details)) {
            return Collections.emptyList();
        }
        List<Map<String, String>> checkResult = new ArrayList<>();
        //获取所有开启的规则
        AccReimRuleQuery query = new AccReimRuleQuery();
        query.setRuleFlag(1);
        List<AccReimRuleVO> accReimRuleVOS = dao.queryListDynamic(query);
        //筛选出通用的、当前类型的与科目专属的
        List<AccReimRuleVO> normalAndSelfRuleVOS = new ArrayList<>();
        List<AccReimRuleVO> item4RuleVOS = new ArrayList<>();
        accReimRuleVOS.forEach(
                ruleVO -> {
                    if (AccReimRuleTypeEnum.NORMAL.getCode().equals(ruleVO.getRuleType()) || payload.getReimDocType().equals(ruleVO.getRuleType())) {
                        normalAndSelfRuleVOS.add(ruleVO);
                    } else {
                        item4RuleVOS.add(ruleVO);
                    }
                }
        );

        //获取明细所有的发票信息
        List<Long> invIdList = new ArrayList<>();
        for (AccReimDetailPayload detailPayload : details) {
            if (!ObjectUtils.isEmpty(detailPayload.getInvIdList())) {
                invIdList.addAll(detailPayload.getInvIdList());
            }
        }
        List<InvInvoiceVO> invInvoiceVOS;
        if (!ObjectUtils.isEmpty(invIdList)) {
            InvInvoiceQuery invoiceQuery = new InvInvoiceQuery();
            invoiceQuery.setIdList(invIdList);
            invInvoiceVOS = invInvoiceService.queryListDynamic(invoiceQuery);
        } else {
            invInvoiceVOS = null;
        }

        //获取当前报销人信息
        PrdOrgEmployeeVO employeeVO = cacheUtil.getEmployee(payload.getReimUserId());

        details.forEach(
                dtl -> {
                    //获取当前科目对应的所有规则
                    List<AccReimRuleVO> busItem4Rules = item4RuleVOS.stream()
                            .filter(vo -> !ObjectUtils.isEmpty(vo.getBusAccItemCodes())
                                    && vo.getBusAccItemCodes().contains(dtl.getBusAccItemCode()))
                            .collect(Collectors.toList());
                    busItem4Rules.addAll(normalAndSelfRuleVOS);

                    //获取明细对应的发票信息
                    if (!ObjectUtils.isEmpty(invInvoiceVOS) && !ObjectUtils.isEmpty(dtl.getInvIdList())) {
                        List<InvInvoiceVO> selfInvVO = invInvoiceVOS.stream()
                                .filter(invInvoiceVO -> dtl.getInvIdList().contains(invInvoiceVO.getId()))
                                .collect(Collectors.toList());
                        if (!ObjectUtils.isEmpty(selfInvVO)) {
                            dtl.setInvInvoiceVOS(selfInvVO);
                        }
                    }

                    //获取当前用户费用发生日期此月内的所有报销及明细
                    List<AccReimVO> currentMonthReim = this.getCurrentMonthReim(payload.getReimUserId(), dtl.getExpenseDate());
                    List<AccReimDetailVO> currentMonthDtl = new ArrayList<>();
                    if (!ObjectUtils.isEmpty(currentMonthReim)) {
                        currentMonthDtl = this.getCurrentMonthDtl(currentMonthReim);
                    }

                    //遍历科目对应的所有规则
                    for (AccReimRuleVO ruleVO : busItem4Rules) {
                        this.ruleHandler(ruleVO, payload, dtl, checkResult, currentMonthDtl, employeeVO, currentMonthReim);
                    }
                }
        );
        return checkResult;
    }

    /**
     * 获取当前用户费用发生日期此月内的所有报销单明细
     *
     * @param voList
     * @return
     */
    private List<AccReimDetailVO> getCurrentMonthDtl(List<AccReimVO> voList) {
        Set<Long> reimIdList = voList.stream().map(AccReimVO::getId).collect(Collectors.toSet());
        return reimDetailDAO.queryByMasIds(new ArrayList<>(reimIdList));
        /*voList.forEach(
                vo -> {
                    List<AccReimDetailVO> detailVOS4Self = detailVOS.stream()
                            .filter(dtl -> dtl.getMasId().equals(vo.getId()))
                            .collect(Collectors.toList());
                    if (!ObjectUtils.isEmpty(detailVOS4Self)) {
                        vo.setDetails(detailVOS4Self);
                    }
                }
        );*/
    }

    /**
     * 获取当前用户费用发生日期此月内的所有报销单
     *
     * @param reimUserId
     * @param expenseDate
     * @return
     */
    private List<AccReimVO> getCurrentMonthReim(Long reimUserId, LocalDate expenseDate) {
        AccReimQuery reimQuery = new AccReimQuery();
        reimQuery.setReimUserId(reimUserId);
        LocalDate startOfMonth = expenseDate.withDayOfMonth(1);
        LocalDate endOfMonth = expenseDate.withDayOfMonth(expenseDate.lengthOfMonth());
        //加一天，不然30号时间会变成 2023-12-30 00:00:00 ，当天的无法控制
        endOfMonth = endOfMonth.plusDays(1);
        reimQuery.setExpenseDateStart(startOfMonth);
        reimQuery.setExpenseDateEnd(endOfMonth);
        List<AccReimVO> voList = reimDAO.queryList4Rule(reimQuery);
        if (ObjectUtils.isEmpty(voList)) {
            return Collections.emptyList();
        }
        return voList;
    }

    /**
     * 对规则进行枚举
     *
     * @param ruleVO
     * @param payload
     */
    private void ruleHandler(AccReimRuleVO ruleVO, AccReimPayload payload, AccReimDetailPayload detailPayload,
                             List<Map<String, String>> checkResult, List<AccReimDetailVO> currentMonthDtl,
                             PrdOrgEmployeeVO employeeVO, List<AccReimVO> currentMonthReim) {
        switch (RuleCodeEnum.find(ruleVO.getRuleNo())) {
            case RULE001:
                //
                break;
            case RULE002:
                //
                break;
            case RULE003:
                //
                break;
            case RULE004:
                //
                break;
            case RULE005:
                //
                break;
            case RULE006:
                //
                break;
            case RULE007:
                //
                break;
            case RULE008:
                //餐费、业务招待费重复处理
                this.mealAndEntertainRepeatHandler(ruleVO, checkResult, detailPayload, currentMonthDtl, currentMonthReim, payload);
                break;
            case RULE009:
                //必须关联发票或填写无发票原因 行政订票特殊 2023/12/20 需求定
                this.dtlInvFillHandler(ruleVO, payload, detailPayload, checkResult);
                break;
            case RULE010:
                //
                break;
            case RULE011:
                //发票开票日期校验
                this.invAndExpenseDateHandler(ruleVO, checkResult, detailPayload);
                break;
            case RULE012:
                //
                break;
            case RULE013:
                //报销重复校验 -> 筛选出专项报销主数据
                Set<Long> collect = currentMonthReim
                        .stream()
                        .filter(vo -> AccReimDocTypeEnum.ACC_SPECIAL.getCode().equals(vo.getReimDocType()))
                        .map(AccReimVO::getId)
                        .collect(Collectors.toSet());
                if (ObjectUtils.isEmpty(collect)) {
                    break;
                }
                List<AccReimDetailVO> filterDtl = currentMonthDtl.stream().filter(dtl -> collect.contains(dtl.getMasId())).collect(Collectors.toList());
                this.reimRepeatHandler(ruleVO, detailPayload, checkResult, filterDtl, payload.getId());
                break;
            case RULE014:
                //业务招待费校验
                this.busEntertainFeeHandler(ruleVO, detailPayload, checkResult, payload);
                break;
            case RULE015:
                //专项报销额度校验
                this.limitAmtHandler(ruleVO, detailPayload, employeeVO, checkResult, payload);
                break;
            default:
                break;
        }

    }

    /**
     * 发票填充校验
     *
     * @param payload
     * @param detailPayload
     * @param checkResult   规则校验结果
     */
    private void dtlInvFillHandler(AccReimRuleVO ruleVO, AccReimPayload payload, AccReimDetailPayload detailPayload, List<Map<String, String>> checkResult) {
        if (!AccReimDocTypeEnum.ACC_TRIP_TICKET.getCode().equals(payload.getReimDocType())
                && ObjectUtils.isEmpty(detailPayload.getNoinvReason()) && ObjectUtils.isEmpty(detailPayload.getInvIdList())) {
            if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                throw TwException.error("", "必须关联发票或者填写无发票原因！！！");
            } else if (AccReimRuleNoticeLevelEnum.WARN.getCode().equals(ruleVO.getNoticeLevel())) {

            } else {

            }
        }
    }

    /**
     * 报销重复校验 -> 目前仅对专项做了配置
     *
     * @param detailPayload
     * @param checkResult
     */
    private void reimRepeatHandler(AccReimRuleVO ruleVO, AccReimDetailPayload detailPayload, List<Map<String, String>> checkResult,
                                   List<AccReimDetailVO> currentMonthDtl, Long reimId) {
        //可能有时接收的ID为负值，进行下处理
        if (!ObjectUtils.isEmpty(reimId) && reimId <= 0L) {
            reimId = null;
        }
        //判断专项报销是否已存在当月的数据，若已提过报销，则不允许再次提交
        if (ObjectUtils.isEmpty(currentMonthDtl)) {
            return;
        }
        List<Long> busItemIds = currentMonthDtl.stream().map(AccReimDetailVO::getBusAccItemId).collect(Collectors.toList());
        if (busItemIds.contains(detailPayload.getBusAccItemId())) {
            //筛选出对应的masId
            Set<Long> masIds = currentMonthDtl.stream()
                    .filter(dtl -> dtl.getBusAccItemId().equals(detailPayload.getBusAccItemId()))
                    .map(AccReimDetailVO::getMasId)
                    .collect(Collectors.toSet());
            //判断是否与当前的单据id是否一致
            final Long useReimId = reimId;
            masIds.forEach(
                    masId -> {
                        //不是同一个报销单，进行校验处理
                        if (!Objects.equals(masId, useReimId)) {
                            if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                                //不合法则直接报错
                                throw TwException.error("", "专项报销重复!每一种专项报销科目类型当月仅允许提交一次！");
                            } else if (AccReimRuleNoticeLevelEnum.WARN.getCode().equals(ruleVO.getNoticeLevel())) {
                                //存在警告目前暂不做处理
                            } else {

                            }
                        }
                    }
            );
        }
    }

    /**
     * 业务招待费处理
     *
     * @param ruleVO
     * @param detailPayload
     * @param checkResult
     */
    private void busEntertainFeeHandler(AccReimRuleVO ruleVO, AccReimDetailPayload detailPayload,
                                        List<Map<String, String>> checkResult, AccReimPayload payload) {
        //目前仅校验差旅
        if (!AccReimDocTypeEnum.ACC_TRIP.getCode().equals(payload.getReimDocType())) {
            return;
        }
        //报销金额超出额度
        if (ruleVO.getBusAccItemCodes().contains(detailPayload.getBusAccItemCode())) {
            if (detailPayload.getReimAmt().compareTo(ruleVO.getLimitAmt()) > 0) {
                if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                    //不合法则直接报错
                    throw TwException.error("", "业务招待费超过" + ruleVO.getLimitAmt() + "请走特殊费用报销！");
                } else if (AccReimRuleNoticeLevelEnum.WARN.getCode().equals(ruleVO.getNoticeLevel())) {
                    //存在警告目前暂不做处理
                } else {

                }
            }
        }
    }

    /**
     * 餐费、业务招待费重复处理
     *
     * @param ruleVO
     * @param checkResult
     * @param currentMonthDtl
     */
    private void mealAndEntertainRepeatHandler(AccReimRuleVO ruleVO, List<Map<String, String>> checkResult,
                                               AccReimDetailPayload detailPayload, List<AccReimDetailVO> currentMonthDtl,
                                               List<AccReimVO> currentMonthReim, AccReimPayload payload) {

        String[] split = ruleVO.getBusAccItemCodes().split(",");
        List<String> ruleItemCodes = List.of(split);

        //当前报销单的所有核算项目
        List<String> currentReimCode = payload.getDetails().stream().map(AccReimDetailPayload::getBusAccItemCode).collect(Collectors.toList());
        //一笔报销单同时包含两个需要校验的科目
        if (currentReimCode.containsAll(ruleItemCodes)) {
            if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                throw TwException.error("", "存在餐费与业务招待费重复报销！");
            }
        }

        if (!ruleItemCodes.contains(detailPayload.getBusAccItemCode())) {
            return;
        }

        if (ObjectUtils.isEmpty(currentMonthReim) || ObjectUtils.isEmpty(currentMonthDtl)) {
            return;
        }

        //明细的费用发生日期
        List<LocalDate> dtlExpenseDateList = payload.getDetails().stream().map(AccReimDetailPayload::getExpenseDate).collect(Collectors.toList());
        //费用发生日期对应的报销单id
        Set<Long> masIdSet = currentMonthDtl.stream()
                .filter(dtl -> dtlExpenseDateList.contains(dtl.getExpenseDate()))
                .map(AccReimDetailVO::getMasId)
                .collect(Collectors.toSet());
        //多条明细对应的报销单的关联单号
        Set<Long> relateDocIds = currentMonthReim.stream()
                .filter(reim -> !ObjectUtils.isEmpty(reim.getRelatedDocId()) && masIdSet.contains(reim.getId()))
                .map(AccReimVO::getRelatedDocId)
                .collect(Collectors.toSet());
        //不是一个报销单，直接放行
        if (ObjectUtils.isEmpty(relateDocIds) || !relateDocIds.contains(payload.getRelatedDocId())) {
            return;
        }

        //筛选费用发生日期的是否报销单重复
        //查询费用发生日期的报销核算科目
        //排除编辑的情况，因为前面已经做过同一个单子的两种科目校验
        if (!ObjectUtils.isEmpty(payload.getId())) {
            return;
        }
        List<String> busAccItemCodes = currentMonthDtl.stream()
                .filter(dtl -> detailPayload.getExpenseDate().equals(dtl.getExpenseDate()))
                .map(AccReimDetailVO::getBusAccItemCode)
                .collect(Collectors.toList());
        for (String ruleItemCode : ruleItemCodes) {
            if (busAccItemCodes.contains(ruleItemCode)) {
                if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                    throw TwException.error("", "存在餐费与业务招待费重复报销！");
                } else if (AccReimRuleNoticeLevelEnum.WARN.getCode().equals(ruleVO.getNoticeLevel())) {

                } else {

                }
            }
        }
    }

    /**
     * 发票开票日期校验
     *
     * @param ruleVO
     * @param checkResult
     * @param detailPayload
     */
    private void invAndExpenseDateHandler(AccReimRuleVO ruleVO, List<Map<String, String>> checkResult, AccReimDetailPayload detailPayload) {
        if (ObjectUtils.isEmpty(detailPayload.getInvInvoiceVOS())) {
            log.info("发票信息为空，无需校验");
            return;
        }
        if (ruleVO.getBusAccItemCodes().contains(detailPayload.getBusAccItemCode())) {
            log.info("当前科目不需做此项检查，科目编码 is {}", detailPayload.getBusAccItemCode());
            return;
        }
        for (InvInvoiceVO invInvoiceVO : detailPayload.getInvInvoiceVOS()) {
            if (ObjectUtils.isEmpty(invInvoiceVO.getInvoiceDate())) {
                log.info("发票开票日期为空，调过校验，发票号为：{}", invInvoiceVO.getInvoiceNo());
                continue;
            }
            LocalDate invDate = LocalDate.parse(invInvoiceVO.getInvoiceDate());
            if (invDate.compareTo(detailPayload.getExpenseDate()) < 0) {
                if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                    //不合法则直接报错
                    throw TwException.error("", "发票的开票日期不能早于费用发生日期");
                } else if (AccReimRuleNoticeLevelEnum.WARN.getCode().equals(ruleVO.getNoticeLevel())) {
                    //存在警告目前暂不做处理
                } else {

                }
            }
        }

    }

    /**
     * 报销额度校验 -> 目前针对于专项生效
     *
     * @param ruleVO        报销额度配置，第一个科目编码为交通费，第二个为通讯费
     * @param detailPayload
     * @param employeeVO
     * @param checkResult
     */
    private void limitAmtHandler(AccReimRuleVO ruleVO, AccReimDetailPayload detailPayload, PrdOrgEmployeeVO employeeVO,
                                 List<Map<String, String>> checkResult, AccReimPayload payload) {
        //因为根据code匹配，其他类型报销单可能匹配到，因为非专项直接return
        if (!AccReimDocTypeEnum.ACC_SPECIAL.getCode().equals(payload.getReimDocType())) {
            return;
        }
        String[] split = ruleVO.getBusAccItemCodes().split(",");
        if (ObjectUtils.isEmpty(split) || split.length < 2) {
            log.error("交通、通讯费用code未在配置表进行维护，请在acc_reim_rule表中维护当前规则信息，rule info is {}", JSON.toJSONString(ruleVO));
            throw TwException.error("", "交通、通讯费用code未在配置表进行维护，请在acc_reim_rule表中维护当前规则信息,规则编号为：" + ruleVO.getRuleNo());
        }
        String extString7 = employeeVO.getExtString7();
        String extString8 = employeeVO.getExtString8();
        if (ObjectUtils.isEmpty(extString7) || ObjectUtils.isEmpty(extString8)) {
            throw TwException.error("", "员工信息的交通费额度或通讯费额度未维护！请联系管理员进行维护！");
        }
        //交通费校验
        if (detailPayload.getBusAccItemCode().equals(split[0])) {
            BigDecimal portLimitAmt = new BigDecimal(extString8);
            if (detailPayload.getReimAmt().compareTo(portLimitAmt) > 0) {
                if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                    throw TwException.error("", "专项报销交通费已超过个人额度！");
                }
            }
        }
        //通讯费校验
        if (detailPayload.getBusAccItemCode().equals(split[1])) {
            BigDecimal phoneLimitAmt = new BigDecimal(extString7);
            if (detailPayload.getReimAmt().compareTo(phoneLimitAmt) > 0) {
                if (AccReimRuleNoticeLevelEnum.ILLEGAL.getCode().equals(ruleVO.getNoticeLevel())) {
                    throw TwException.error("", "专项报销通讯费已超过个人额度！");
                }
            }
        }
    }

}

