package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitesland.tw.tw5.api.prd.cal.payload.CalEqvaIncomeLogPayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalEqvaIncomeLogQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalEqvaIncomeLogVO;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalEqvaIncomeLogDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalEqvaIncomeLogDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalEqvaIncomeLogRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 当量收入配置变更记录
 *
 * @author wangly
 * @date 2024-02-22
 */
@Repository
@RequiredArgsConstructor
public class CalEqvaIncomeLogDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalEqvaIncomeLogRepo repo;
    private final QCalEqvaIncomeLogDO qdo = QCalEqvaIncomeLogDO.calEqvaIncomeLogDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaIncomeLogVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalEqvaIncomeLogVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 当量收入配置主键
                qdo.relateId,
                // 单位当量收入
                qdo.preeqvaAmt,
                // 状态
                qdo.lineStatus,
                // 项目id
                qdo.projId,
                // 项目名称
                qdo.projName,
                // 资源id
                qdo.resId,
                // 资源类型1内部，0外部
                qdo.resType,
                // 版本号
                qdo.version,
                // 起始日期
                qdo.startDate,
                // 失效日期
                qdo.endDate,
                // 结算方式
                qdo.settleType
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaIncomeLogVO> getJpaQueryWhere(CalEqvaIncomeLogQuery query) {
        JPAQuery<CalEqvaIncomeLogVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalEqvaIncomeLogQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalEqvaIncomeLogQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 当量收入配置主键 精确 */
        if (!ObjectUtils.isEmpty(query.getRelateId())) {
            list.add(qdo.relateId.eq(query.getRelateId()));
        }
        /** 单位当量收入 精确 */
        if (!ObjectUtils.isEmpty(query.getPreeqvaAmt())) {
            list.add(qdo.preeqvaAmt.eq(query.getPreeqvaAmt()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getLineStatus())) {
            list.add(qdo.lineStatus.eq(query.getLineStatus()));
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        }
        /** 项目名称 精确 */
        if (!ObjectUtils.isEmpty(query.getProjName())) {
            list.add(qdo.projName.eq(query.getProjName()));
        }
        /** 资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()));
        }
        /** 资源类型1内部，0外部 精确 */
        if (!ObjectUtils.isEmpty(query.getResType())) {
            list.add(qdo.resType.eq(query.getResType()));
        }
        /** 版本号 精确 */
        if (!ObjectUtils.isEmpty(query.getVersion())) {
            list.add(qdo.version.eq(query.getVersion()));
        }
        /** 起始日期 精确 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.goe(query.getStartDate()));
        }

        /** 失效日期 精确 */
        if (!ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.endDate.loe(query.getEndDate()));
        }
        /** 结算方式 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleType())) {
            list.add(qdo.settleType.eq(query.getSettleType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalEqvaIncomeLogVO queryByKey(Long id) {
        JPAQuery<CalEqvaIncomeLogVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据当量收入配置主键查询最近的一条
     *
     * @param key 主键
     * @return 结果
     */
    public CalEqvaIncomeLogVO queryFirstByRelateId(Long key) {
        JPAQuery<CalEqvaIncomeLogVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.relateId.eq(key));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.orderBy(qdo.version.desc()).orderBy(qdo.createTime.desc());
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalEqvaIncomeLogVO> queryListDynamic(CalEqvaIncomeLogQuery query) {
        JPAQuery<CalEqvaIncomeLogVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalEqvaIncomeLogVO> queryPaging(CalEqvaIncomeLogQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<CalEqvaIncomeLogVO> jpaQuery = getJpaQueryWhere(query);
        List<CalEqvaIncomeLogVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalEqvaIncomeLogVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalEqvaIncomeLogDO save(CalEqvaIncomeLogDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalEqvaIncomeLogDO> saveAll(List<CalEqvaIncomeLogDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalEqvaIncomeLogPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 当量收入配置主键
        if (payload.getRelateId() != null) {
            update.set(qdo.relateId, payload.getRelateId());
        }
        // 单位当量收入
        if (payload.getPreeqvaAmt() != null) {
            update.set(qdo.preeqvaAmt, payload.getPreeqvaAmt());
        }
        // 状态
        if (payload.getLineStatus() != null) {
            update.set(qdo.lineStatus, payload.getLineStatus());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 项目名称
        if (payload.getProjName() != null) {
            update.set(qdo.projName, payload.getProjName());
        }
        // 资源id
        if (payload.getResId() != null) {
            update.set(qdo.resId, payload.getResId());
        }
        // 资源类型1内部，0外部
        if (payload.getResType() != null) {
            update.set(qdo.resType, payload.getResType());
        }
        // 版本号
        if (payload.getVersion() != null) {
            update.set(qdo.version, payload.getVersion());
        }
        // 起始日期
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 失效日期
        if (payload.getEndDate() != null) {
            update.set(qdo.endDate, payload.getEndDate());
        }
        // 结算方式
        if (payload.getSettleType() != null) {
            update.set(qdo.settleType, payload.getSettleType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 当量收入配置主键
            if (nullFields.contains("relateId")) {
                update.setNull(qdo.relateId);
            }
            // 单位当量收入
            if (nullFields.contains("preeqvaAmt")) {
                update.setNull(qdo.preeqvaAmt);
            }
            // 状态
            if (nullFields.contains("lineStatus")) {
                update.setNull(qdo.lineStatus);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 项目名称
            if (nullFields.contains("projName")) {
                update.setNull(qdo.projName);
            }
            // 资源id
            if (nullFields.contains("resId")) {
                update.setNull(qdo.resId);
            }
            // 资源类型1内部，0外部
            if (nullFields.contains("resType")) {
                update.setNull(qdo.resType);
            }
            // 版本号
            if (nullFields.contains("version")) {
                update.setNull(qdo.version);
            }
            // 起始日期
            if (nullFields.contains("startDate")) {
                update.setNull(qdo.startDate);
            }
            // 失效日期
            if (nullFields.contains("endDate")) {
                update.setNull(qdo.endDate);
            }
            // 结算方式
            if (nullFields.contains("settleType")) {
                update.setNull(qdo.settleType);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

