package com.elitesland.tw.tw5.server.prd.partner.business.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.partner.business.payload.BusinessOperationSitesPayload;
import com.elitesland.tw.tw5.api.prd.partner.business.query.BusinessOperationSitesQuery;
import com.elitesland.tw.tw5.api.prd.partner.business.vo.BusinessOperationSitesVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.partner.business.entity.BusinessOperationSitesDO;
import com.elitesland.tw.tw5.server.prd.partner.business.entity.QBusinessOperationSitesDO;
import com.elitesland.tw.tw5.server.prd.partner.business.repo.BusinessOperationSitesRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 业务合作伙伴-年报网址
 *
 * @author wangly
 * @date 2023-05-18
 */
@Repository
@RequiredArgsConstructor
public class BusinessOperationSitesDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BusinessOperationSitesRepo repo;
    private final QBusinessOperationSitesDO qdo = QBusinessOperationSitesDO.businessOperationSitesDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessOperationSitesVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BusinessOperationSitesVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 地址簿id
                qdo.bookId,
                // 审核时间
                qdo.date,
                // 网址
                qdo.url,
                // 企业网址名称
                qdo.name,
                // 网址类型
                qdo.type,
                // 业务伙伴主键 business_partner.id
                qdo.partnerId,
                // 业务伙伴名称
                qdo.partnerName
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessOperationSitesVO> getJpaQueryWhere(BusinessOperationSitesQuery query) {
        JPAQuery<BusinessOperationSitesVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BusinessOperationSitesQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BusinessOperationSitesQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 地址簿id 精确 */
        if (!ObjectUtils.isEmpty(query.getBookId())) {
            list.add(qdo.bookId.eq(query.getBookId()));
        }
        /** 审核时间 精确 */
        if (!ObjectUtils.isEmpty(query.getDate())) {
            list.add(qdo.date.eq(query.getDate()));
        }
        /** 网址 精确 */
        if (!ObjectUtils.isEmpty(query.getUrl())) {
            list.add(qdo.url.eq(query.getUrl()));
        }
        /** 企业网址名称 精确 */
        if (!ObjectUtils.isEmpty(query.getName())) {
            list.add(qdo.name.eq(query.getName()));
        }
        /** 网址类型 精确 */
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 业务伙伴主键 business_partner.id 精确 */
        if (!ObjectUtils.isEmpty(query.getPartnerId())) {
            list.add(qdo.partnerId.eq(query.getPartnerId()));
        }
        /** 业务伙伴名称 精确 */
        if (!ObjectUtils.isEmpty(query.getPartnerName())) {
            list.add(qdo.partnerName.eq(query.getPartnerName()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BusinessOperationSitesVO queryByKey(Long id) {
        JPAQuery<BusinessOperationSitesVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BusinessOperationSitesVO> queryListDynamic(BusinessOperationSitesQuery query) {
        JPAQuery<BusinessOperationSitesVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BusinessOperationSitesVO> queryPaging(BusinessOperationSitesQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<BusinessOperationSitesVO> jpaQuery = getJpaQueryWhere(query);
        List<BusinessOperationSitesVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BusinessOperationSitesVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BusinessOperationSitesDO save(BusinessOperationSitesDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BusinessOperationSitesDO> saveAll(List<BusinessOperationSitesDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BusinessOperationSitesPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 地址簿id
        if (payload.getBookId() != null) {
            update.set(qdo.bookId, payload.getBookId());
        }
        // 审核时间
        if (payload.getDate() != null) {
            update.set(qdo.date, payload.getDate());
        }
        // 网址
        if (payload.getUrl() != null) {
            update.set(qdo.url, payload.getUrl());
        }
        // 企业网址名称
        if (payload.getName() != null) {
            update.set(qdo.name, payload.getName());
        }
        // 网址类型
        if (payload.getType() != null) {
            update.set(qdo.type, payload.getType());
        }
        // 业务伙伴主键 business_partner.id
        if (payload.getPartnerId() != null) {
            update.set(qdo.partnerId, payload.getPartnerId());
        }
        // 业务伙伴名称
        if (payload.getPartnerName() != null) {
            update.set(qdo.partnerName, payload.getPartnerName());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 地址簿id
            if (nullFields.contains("bookId")) {
                update.setNull(qdo.bookId);
            }
            // 审核时间
            if (nullFields.contains("date")) {
                update.setNull(qdo.date);
            }
            // 网址
            if (nullFields.contains("url")) {
                update.setNull(qdo.url);
            }
            // 企业网址名称
            if (nullFields.contains("name")) {
                update.setNull(qdo.name);
            }
            // 网址类型
            if (nullFields.contains("type")) {
                update.setNull(qdo.type);
            }
            // 业务伙伴主键 business_partner.id
            if (nullFields.contains("partnerId")) {
                update.setNull(qdo.partnerId);
            }
            // 业务伙伴名称
            if (nullFields.contains("partnerName")) {
                update.setNull(qdo.partnerName);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 根据partnerId删除
     *
     * @param partnerId 业务伙伴id
     * @return 删除的行数
     */
    public void deleteByPartnerId(Long partnerId){
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.partnerId.eq(partnerId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }
}

