package com.elitesland.tw.tw5.server.prd.pms.dao;


import com.elitesland.tw.tw5.api.prd.pms.payload.PmsDistBroadcastPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsDistBroadcastQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsDistBroadcastVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsDistBroadcastDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsDistBroadcastDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsDistributeRespondDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsDistBroadcastRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 广播
 *
 * @author xxb
 * @date 2023-10-07
 */
@Repository
@RequiredArgsConstructor
public class PmsDistBroadcastDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsDistBroadcastRepo repo;
    private final QPmsDistBroadcastDO qdo = QPmsDistBroadcastDO.pmsDistBroadcastDO;

    private final QPmsDistributeRespondDO respondQdo = QPmsDistributeRespondDO.pmsDistributeRespondDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsDistBroadcastVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsDistBroadcastVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 派发ID
                qdo.distId,
                // 应答人数上限
                qdo.respNumber,
                // 广播天数
                qdo.broadcastDays,
                // 剩余天数
                qdo.remainingDays
        )).from(qdo);
    }


    /**
     * 登录人未响应过的派发数据（感兴趣和不感兴趣都算响应）
     * @param respondentResId
     * @return
     */
    public List<PmsDistBroadcastVO> findListByRespondentResId(Long respondentResId){
        JPAQuery<PmsDistBroadcastVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PmsDistBroadcastVO.class,
                qdo.id
        )).from(qdo).innerJoin(respondQdo).on(qdo.distId.eq(respondQdo.distId));
        List<Predicate> list = new ArrayList<>();
        List<String> status = Arrays.asList("NO RESPONDING","INTERESTED");
        list.add(respondQdo.respStatus.isNotNull().and(respondQdo.respStatus.notIn(status)));
        list.add(respondQdo.disterResId.eq(respondentResId));
        list.add(respondQdo.deleteFlag.eq(0));
        // 条件封装
        jpaQuery.where(ExpressionUtils.allOf(list));
        return jpaQuery.fetch();
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsDistBroadcastVO> getJpaQueryWhere(PmsDistBroadcastQuery query) {
        JPAQuery<PmsDistBroadcastVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsDistBroadcastQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsDistBroadcastQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 派发ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDistId())) {
            list.add(qdo.distId.eq(query.getDistId()));
        }
        /** 应答人数上限 精确 */
        if (!ObjectUtils.isEmpty(query.getRespNumber())) {
            list.add(qdo.respNumber.eq(query.getRespNumber()));
        }
        /** 广播天数 精确 */
        if (!ObjectUtils.isEmpty(query.getBroadcastDays())) {
            list.add(qdo.broadcastDays.eq(query.getBroadcastDays()));
        }
        /** 剩余天数 精确 */
        if (!ObjectUtils.isEmpty(query.getRemainingDays())) {
            list.add(qdo.remainingDays.eq(query.getRemainingDays()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsDistBroadcastVO queryByKey(Long id) {
        JPAQuery<PmsDistBroadcastVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsDistBroadcastVO> queryListDynamic(PmsDistBroadcastQuery query) {
        JPAQuery<PmsDistBroadcastVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsDistBroadcastVO> queryPaging(PmsDistBroadcastQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PmsDistBroadcastVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsDistBroadcastVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsDistBroadcastVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsDistBroadcastDO save(PmsDistBroadcastDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsDistBroadcastDO> saveAll(List<PmsDistBroadcastDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsDistBroadcastPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 派发ID
        if (payload.getDistId() != null) {
            update.set(qdo.distId, payload.getDistId());
        }
        // 应答人数上限
        if (payload.getRespNumber() != null) {
            update.set(qdo.respNumber, payload.getRespNumber());
        }
        // 广播天数
        if (payload.getBroadcastDays() != null) {
            update.set(qdo.broadcastDays, payload.getBroadcastDays());
        }
        // 剩余天数
        if (payload.getRemainingDays() != null) {
            update.set(qdo.remainingDays, payload.getRemainingDays());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 派发ID
            if (nullFields.contains("distId")) {
                update.setNull(qdo.distId);
            }
            // 应答人数上限
            if (nullFields.contains("respNumber")) {
                update.setNull(qdo.respNumber);
            }
            // 广播天数
            if (nullFields.contains("broadcastDays")) {
                update.setNull(qdo.broadcastDays);
            }
            // 剩余天数
            if (nullFields.contains("remainingDays")) {
                update.setNull(qdo.remainingDays);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据派发ID删除
     *
     * @param distIds 派发ID
     * @return 删除的行数
     */
    public long deleteByDistIds(List<Long> distIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.distId.in(distIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

