package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitesland.tw.tw5.api.prd.purchase.payload.PaymentPlanReferPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PaymentPlanReferQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PaymentPlanReferVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchasePaymentPlanVO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PaymentPlanReferDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPaymentPlanReferDO;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PaymentPlanReferRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 付款计划参考
 *
 * @author likunpeng
 * @date 2023-11-22
 */
@Repository
@RequiredArgsConstructor
public class PaymentPlanReferDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PaymentPlanReferRepo repo;
    private final QPaymentPlanReferDO qdo = QPaymentPlanReferDO.paymentPlanReferDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PaymentPlanReferVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PaymentPlanReferVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 预付款ID（付款申请单表主键）
                qdo.paymentApplyId,
                // 预付款编号
                qdo.paymentNo,
                // 付款阶段
                qdo.paymentStage,
                // 付款金额（预付款金额）
                qdo.paymentAmt,
                // 本次付款金额
                qdo.currentPaymentAmt,
                // 付款比例
                qdo.paymentProportion,
                // 预计付款日期
                qdo.estimatedPaymentDate,
                // 采购合同id
                qdo.contractId,
                // 采购合同编号
                qdo.contractNo,
                // 付款状态
                qdo.paymentStatus,
                // 付款时间
                qdo.paymentDate
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PaymentPlanReferVO> getJpaQueryWhere(PaymentPlanReferQuery query) {
        JPAQuery<PaymentPlanReferVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PaymentPlanReferQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PaymentPlanReferQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 预付款ID（付款申请单表主键） 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplyId())) {
            list.add(qdo.paymentApplyId.eq(query.getPaymentApplyId()));
        }
        /** 预付款编号 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentNo())) {
            list.add(qdo.paymentNo.eq(query.getPaymentNo()));
        }
        /** 付款阶段 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentStage())) {
            list.add(qdo.paymentStage.eq(query.getPaymentStage()));
        }
        /** 付款金额（预付款金额） 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentAmt())) {
            list.add(qdo.paymentAmt.eq(query.getPaymentAmt()));
        }
        /** 本次付款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrentPaymentAmt())) {
            list.add(qdo.currentPaymentAmt.eq(query.getCurrentPaymentAmt()));
        }
        /** 付款比例 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentProportion())) {
            list.add(qdo.paymentProportion.eq(query.getPaymentProportion()));
        }
        /** 预计付款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getEstimatedPaymentDate())) {
            list.add(qdo.estimatedPaymentDate.eq(query.getEstimatedPaymentDate()));
        }
        /** 采购合同id 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 采购合同编号 精确 */
        if (!ObjectUtils.isEmpty(query.getContractNo())) {
            list.add(qdo.contractNo.eq(query.getContractNo()));
        }
        /** 付款状态 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentStatus())) {
            list.add(qdo.paymentStatus.eq(query.getPaymentStatus()));
        }
        /** 付款时间 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentDate())) {
            list.add(qdo.paymentDate.eq(query.getPaymentDate()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PaymentPlanReferVO queryByKey(Long id) {
        JPAQuery<PaymentPlanReferVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PaymentPlanReferVO> queryListDynamic(PaymentPlanReferQuery query) {
        JPAQuery<PaymentPlanReferVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PaymentPlanReferVO> queryPaging(PaymentPlanReferQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PaymentPlanReferVO> jpaQuery = getJpaQueryWhere(query);
        List<PaymentPlanReferVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PaymentPlanReferVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PaymentPlanReferDO save(PaymentPlanReferDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PaymentPlanReferDO> saveAll(List<PaymentPlanReferDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PaymentPlanReferPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 预付款ID（付款申请单表主键）
        if (payload.getPaymentApplyId() != null) {
            update.set(qdo.paymentApplyId, payload.getPaymentApplyId());
        }
        // 预付款编号
        if (payload.getPaymentNo() != null) {
            update.set(qdo.paymentNo, payload.getPaymentNo());
        }
        // 付款阶段
        if (payload.getPaymentStage() != null) {
            update.set(qdo.paymentStage, payload.getPaymentStage());
        }
        // 付款金额（预付款金额）
        if (payload.getPaymentAmt() != null) {
            update.set(qdo.paymentAmt, payload.getPaymentAmt());
        }
        // 本次付款金额
        if (payload.getCurrentPaymentAmt() != null) {
            update.set(qdo.currentPaymentAmt, payload.getCurrentPaymentAmt());
        }
        // 付款比例
        if (payload.getPaymentProportion() != null) {
            update.set(qdo.paymentProportion, payload.getPaymentProportion());
        }
        // 预计付款日期
        if (payload.getEstimatedPaymentDate() != null) {
            update.set(qdo.estimatedPaymentDate, payload.getEstimatedPaymentDate());
        }
        // 采购合同id
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        // 采购合同编号
        if (payload.getContractNo() != null) {
            update.set(qdo.contractNo, payload.getContractNo());
        }
        // 付款状态
        if (payload.getPaymentStatus() != null) {
            update.set(qdo.paymentStatus, payload.getPaymentStatus());
        }
        // 付款时间
        if (payload.getPaymentDate() != null) {
            update.set(qdo.paymentDate, payload.getPaymentDate());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 预付款ID（付款申请单表主键）
            if (nullFields.contains("paymentApplyId")) {
                update.setNull(qdo.paymentApplyId);
            }
            // 预付款编号
            if (nullFields.contains("paymentNo")) {
                update.setNull(qdo.paymentNo);
            }
            // 付款阶段
            if (nullFields.contains("paymentStage")) {
                update.setNull(qdo.paymentStage);
            }
            // 付款金额（预付款金额）
            if (nullFields.contains("paymentAmt")) {
                update.setNull(qdo.paymentAmt);
            }
            // 本次付款金额
            if (nullFields.contains("currentPaymentAmt")) {
                update.setNull(qdo.currentPaymentAmt);
            }
            // 付款比例
            if (nullFields.contains("paymentProportion")) {
                update.setNull(qdo.paymentProportion);
            }
            // 预计付款日期
            if (nullFields.contains("estimatedPaymentDate")) {
                update.setNull(qdo.estimatedPaymentDate);
            }
            // 采购合同id
            if (nullFields.contains("contractId")) {
                update.setNull(qdo.contractId);
            }
            // 采购合同编号
            if (nullFields.contains("contractNo")) {
                update.setNull(qdo.contractNo);
            }
            // 付款状态
            if (nullFields.contains("paymentStatus")) {
                update.setNull(qdo.paymentStatus);
            }
            // 付款时间
            if (nullFields.contains("paymentDate")) {
                update.setNull(qdo.paymentDate);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据付款申请单Id删除付款计划参考
     * @param id 付款申请单Id
     */
    public void deleteByPaymentId(Long id) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.paymentApplyId.eq(id));
        update.execute();
    }

    /**
     * 根据付款申请单ID查询列表
     *
     * @param paymentApplyId 付款申请单ID
     * @return {@link List}<{@link PurchasePaymentPlanVO}>
     */
    public List<PaymentPlanReferVO> queryListByPaymentApplyId(Long paymentApplyId) {
        JPAQuery<PaymentPlanReferVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.paymentApplyId.eq(paymentApplyId));
        return jpaQuerySelect.fetch();
    }
}

