package com.elitesland.tw.tw5.server.prd.schedule.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.schedule.payload.PrdCalendarPayload;
import com.elitesland.tw.tw5.api.prd.schedule.query.PrdCalendarQuery;
import com.elitesland.tw.tw5.api.prd.schedule.vo.PrdCalendarVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.schedule.entity.PrdCalendarDO;
import com.elitesland.tw.tw5.server.prd.schedule.entity.QPrdCalendarDO;
import com.elitesland.tw.tw5.server.prd.schedule.repo.PrdCalendarRepo;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * @author zoey
 * @Description:日历的dao层
 * @date 2022/4/19 - 23:25
 */
@Repository
@RequiredArgsConstructor
public class PrdCalendarDAO {
    private final JPAQueryFactory jpaQueryFactory;
    private final PrdCalendarRepo repo;
    private final QPrdCalendarDO qdo = QPrdCalendarDO.prdCalendarDO;


    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdCalendarDO save(PrdCalendarDO ado) {
        return repo.save(ado);
    }


    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdCalendarDO> saveAll(List<PrdCalendarDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdCalendarVO queryByKey(Long id) {
        JPAQuery<PrdCalendarVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        return jpaQuery.fetchFirst();
    }

    public List<PrdCalendarVO> queryByCreateUserId(Long userId) {
        JPAQuery<PrdCalendarVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.createUserId.eq(userId));
        return jpaQuery.fetch();
    }

    public List<PrdCalendarVO> queryDefaultByUserId(Long userId) {
        JPAQuery<PrdCalendarVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.createUserId.eq(userId));
        jpaQuery.where(qdo.defaultFlag.eq(1));
        return jpaQuery.fetch();
    }

    /**
     * 根据企业微信日历id查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdCalendarVO queryByQyWxCalId(String id) {
        JPAQuery<PrdCalendarVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.qyWxCalId.eq(id));
        return jpaQuery.fetchFirst();
    }


    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdCalendarVO> getJpaQuerySelect() {
        JPAQuery<PrdCalendarVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PrdCalendarVO.class,
                qdo.id,
                qdo.title,
                qdo.color,
                qdo.description,
                qdo.readonly,
                qdo.defaultFlag,
                qdo.createUserId,
                qdo.qyWxCalId
        )).from(qdo);
        return jpaQuery;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdCalendarVO> getJpaQueryWhere(PrdCalendarQuery query) {
        JPAQuery<PrdCalendarVO> jpaQuery = getJpaQuerySelect();

        if (!ObjectUtils.isEmpty(query.getTitle())) {
            jpaQuery.where(qdo.title.like(SqlUtil.toSqlLikeString(query.getTitle())));
        }
        if (!ObjectUtils.isEmpty(query.getDescription())) {
            jpaQuery.where(qdo.description.like(SqlUtil.toSqlLikeString(query.getDescription())));
        }
        if (!ObjectUtils.isEmpty(query.getDefaultFlag())) {
            jpaQuery.where(qdo.defaultFlag.eq(query.getDefaultFlag()));
        }
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            jpaQuery.where(qdo.createUserId.eq(query.getCreateUserId()));
        }
        if (!ObjectUtils.isEmpty(query.getCalIds())) {
            jpaQuery.where(qdo.id.in(query.getCalIds()));
        } else {
            jpaQuery.where(qdo.id.in(Collections.singletonList(-1L)));
        }
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 动态查询集合(不分页查询)
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdCalendarVO> queryListDynamic(PrdCalendarQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.asc("createTime");
        query.setOrders(Arrays.asList(orderItem));
        JPAQuery<PrdCalendarVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }


    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdCalendarVO> queryPaging(PrdCalendarQuery query) {
        JPAQuery<PrdCalendarVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<PrdCalendarVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        System.out.println(result.getTotal());
        return PagingVO.<PrdCalendarVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(PrdCalendarPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        if (payload.getTitle() != null) {
            update.set(qdo.title, payload.getTitle());
        }
        if (payload.getDescription() != null) {
            update.set(qdo.description, payload.getDescription());
        }
        if (payload.getColor() != null) {
            update.set(qdo.color, payload.getColor());
        }
        if (payload.getReadonly() != null) {
            update.set(qdo.readonly, payload.getReadonly());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
//            if (nullFields.contains("description")) {
//                update.setNull(qdo.description);
//            }
        }
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        return update.execute();
    }


}
