package com.elitesland.tw.tw5.server.prd.system.service;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemPermissionRuleDetailPayload;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemPermissionRulePayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemPermissionFieldQuery;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemPermissionRuleQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemPermissonRuleService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemPermissionFieldObjRoleFunctionVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemPermissionFieldVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemPermissionRuleDetailVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemPermissionRuleVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionFieldType;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionRuleSubTypeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionRuleType;
import com.elitesland.tw.tw5.server.common.util.RedisUtils;
import com.elitesland.tw.tw5.server.prd.org.dao.PrdOrgEmployeeDAO;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemPermissionRuleConvert;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemPermissionRuleDetailConvert;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemPermissionRuleDAO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemBusinessObjectDO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemNewFunctionDO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemPermissionRuleDO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemPermissionRuleDetailDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemBusinessObjectRepo;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemNewFunctionRepo;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemPermissionRuleDetailRepo;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemPermissionRuleRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 权限规则
 *
 * @Author Quruiqizz
 * @Date 2023/9/25 15:49
 **/
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdSystemPermissionRuleServiceImpl implements PrdSystemPermissonRuleService {

    private final PrdSystemPermissionRuleDAO dao;
    private final PrdSystemPermissionRuleRepo repo;
    private final PrdSystemPermissionRuleDetailRepo detailRepo;
    private final RedisUtils redisUtils;
    private final PrdSystemBusinessObjectRepo objectRepo;
    private final PrdSystemNewFunctionRepo functionRepo;
    private static final String RULE_CACHE_PREFIX = "sys:permission:rule:";
    private static final String FIELD_RULE_CACHE_PREFIX = "sys:permission:field:rule:";

    /**
     * 绑定数据规则
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void insert(PrdSystemPermissionRulePayload payload) {
        //校验数据
        checkBindDataRule(payload);
        Map<String, String> map = new HashMap<>();
        //根据code查询数据
        PrdSystemPermissionRuleDO entity = dao.queryPermissionRuleByCode(payload.getRuleCode());
        Assert.isNull(entity, "ruleCode不能重复");
        PrdSystemPermissionRuleDO entityDO = PrdSystemPermissionRuleConvert.INSTANCE.toDo(payload);
        entityDO.setRuleTypeName(Objects.requireNonNull(PermissionRuleType.find(entityDO.getRuleType())).getDesc());
        final PrdSystemPermissionRuleDO entitySave = dao.saveBusinessObjectDataRule(entityDO);
        List<PrdSystemPermissionRuleDetailPayload> detailList = payload.getDetailList();
        if (!CollectionUtils.isEmpty(detailList)) {
            for (PrdSystemPermissionRuleDetailPayload detailPayload : detailList) {
                //根据code查询数据
                if (!map.containsKey(detailPayload.getRuleDetailCode())){
                    PrdSystemPermissionRuleDetailDO detailEntityDO = PrdSystemPermissionRuleDetailConvert.INSTANCE.toDo(detailPayload);
                    detailEntityDO.setRuleId(entitySave.getId());
                    dao.saveBusinessObjectDataRuleDetail(detailEntityDO);
                    map.put(detailPayload.getRuleDetailCode(), detailPayload.getRuleDetailCode());
                }else{
                    Assert.isNull(map, "ruleDetailCode不能重复");
                }
            }
        }
        //清楚缓存
        PrdSystemNewFunctionDO functionDO = functionRepo.findById(payload.getFunctionId()).orElseGet(PrdSystemNewFunctionDO::new);
        removeRuleCache(functionDO.getFunctionCode());
    }

    /**
     * 绑定数据删除
     * @param keys
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(Long[] keys) {
        List<PrdSystemPermissionRuleDO> ruleDOS = repo.findAllById(Arrays.asList(keys));
        dao.deletePermissonRuleByIds(keys);
        for (PrdSystemPermissionRuleDO ruleDO : ruleDOS) {
            PrdSystemNewFunctionDO newFunctionDO = functionRepo.findById(ruleDO.getFunctionId()).orElseGet(PrdSystemNewFunctionDO::new);
            removeRuleCache(newFunctionDO.getFunctionCode());
            List<Long> detailIds = dao.queryPermissionRuleDetailIdsByRuleId(ruleDO.getId());
            dao.deletePermissionRuleDetailByIds(detailIds);
        }
    }

    /**
     * 权限规则模糊分页
     * @param query
     * @return
     */
    @Override
    public PagingVO<PrdSystemPermissionRuleVO> paging(PrdSystemPermissionRuleQuery query) {
        //校验
        checkQuery(query);
        return dao.queryPermissionRule(query);
    }

    /**
     * 绑定数据修改
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(PrdSystemPermissionRulePayload payload) {
        //校验修改数据
        checkUpdate(payload);
        PrdSystemPermissionRuleDO entity = repo.findById(payload.getId()).orElseGet(PrdSystemPermissionRuleDO::new);
        Assert.notNull(entity.getId(), "不存在");
        //校验编号是不是重复
        if (StringUtils.hasText(payload.getRuleCode()) && !entity.getRuleCode().equals(payload.getRuleCode())){
            Assert.isNull(dao.queryPermissionRuleByCode(payload.getRuleCode()), "ruleCode不能重复");
        }
        PrdSystemPermissionRuleDO entityDO = PrdSystemPermissionRuleConvert.INSTANCE.toDo(payload);
        entity.copy(entityDO);
        changeEntity(entityDO, entity);
        entity.setRuleTypeName(Objects.requireNonNull(PermissionRuleType.find(entityDO.getRuleType())).getDesc());
        repo.save(entity);
        //全删
        List<Long> ids = dao.queryPermissionRuleDetailIdsByRuleId(entity.getId());
        detailRepo.deleteAllById(ids);
        Map<String, String> map = new HashMap<>();
        List<PrdSystemPermissionRuleDetailDO> list = new ArrayList<>();
        if (!CollectionUtils.isEmpty(payload.getDetailList())){
            for (PrdSystemPermissionRuleDetailPayload detailPayload : payload.getDetailList()) {
                PrdSystemPermissionRuleDetailDO detailEntityDO = insertEntity(detailPayload, entity);
                if (!map.containsKey(detailEntityDO.getRuleDetailCode())){
                    list.add(detailEntityDO);
                    map.put(detailEntityDO.getRuleDetailCode(), detailEntityDO.getRuleDetailCode());
                }else{
                    Assert.isNull(map, "ruleDetailCode不能重复");
                }
            }
        }
        //全插
        detailRepo.saveAll(list);
        //删除缓存
        PrdSystemNewFunctionDO functionDO = functionRepo.findById(payload.getFunctionId()).orElseGet(PrdSystemNewFunctionDO::new);
        removeRuleCache(functionDO.getFunctionCode());
    }

    /**
     * 如果规则类型是用户，清空rule_sub_type,level_scope（包括name）
     * 如果规则类型是组织，清空rule_sub_type（包括name）
     * @param entityDO
     * @param entity
     */
    private void changeEntity(PrdSystemPermissionRuleDO entityDO, PrdSystemPermissionRuleDO entity) {
        if (PermissionRuleType.USER.getName().equals(entityDO.getRuleType())){
            //如果用户类型是用户，则将rule_sub_type、level_scope置空
            entity.setRuleSubType(null);
            entity.setRuleSubTypeName(null);
            entity.setLevelScope(null);
            entity.setLevelScopeName(null);
        } else if(PermissionRuleType.ORG.getName().equals(entityDO.getRuleType())){
            //如果用户类型是组织，则将rule_sub_type置空
            entity.setRuleSubType(null);
            entity.setRuleSubTypeName(null);
        } else if(PermissionRuleType.ROLE.getName().equals(entityDO.getRuleType()) && PermissionRuleSubTypeEnum.BUSINESS_ROLE.getName().equals(entityDO.getRuleSubType())){
            //如果用户类型是 角色 并且是 业务角色，将level_scope置空
            entity.setLevelScope(null);
            entity.setLevelScopeName(null);
        }
    }

    /**
     * 获取插入实体类
     * @param detailPayload
     * @param payload
     * @return
     */
    private PrdSystemPermissionRuleDetailDO insertEntity(PrdSystemPermissionRuleDetailPayload detailPayload, PrdSystemPermissionRuleDO payload) {
        //校验数据完整性
        checkInsertEntity(detailPayload, payload);
        PrdSystemPermissionRuleDetailDO detailEntityDO = PrdSystemPermissionRuleDetailConvert.INSTANCE.toDo(detailPayload);
        return detailEntityDO;
    }

    /**
     * 校验插入数据
     * @param detailPayload
     * @param payload
     */
    private void checkInsertEntity(PrdSystemPermissionRuleDetailPayload detailPayload, PrdSystemPermissionRuleDO payload) {
        if (!StringUtils.hasText(detailPayload.getRuleDetailCode())){
            throw TwException.error("", "detailCode不能为空");
        }
    }

    /**
     * 主键查询
     * @param key
     * @return
     */
    @Override
    public PrdSystemPermissionRuleVO get(Long key) {
        PrdSystemPermissionRuleDO ruleDO = repo.findById(key).orElseGet(PrdSystemPermissionRuleDO::new);
        PrdSystemPermissionRuleVO ruleVO = PrdSystemPermissionRuleConvert.INSTANCE.toVo(ruleDO);
        //处理或和且的关系
        List<PrdSystemPermissionRuleDetailVO> ruleDetailVOS = dao.queryPermissionRuleDetailByRuleId(ruleVO.getId());
        String groupExpr = ruleVO.getGroupExpr().trim();
        for (int i = 0; i < ruleDetailVOS.size(); i++) {
            if (i != 0) {
                ruleDetailVOS.get(i).setExpr(groupExpr.substring(0, 1));
                groupExpr = groupExpr.substring(ruleDetailVOS.get(i).getRuleDetailCode().length() + 1);
            }else {
                groupExpr = groupExpr.substring(ruleDetailVOS.get(i).getRuleDetailCode().length());
            }
        }
        ruleVO.setDetailList(ruleDetailVOS);
        return ruleVO;
    }

    /**
     * 检查修改信息
     * @param payload
     */
    private void checkUpdate(PrdSystemPermissionRulePayload payload) {
        if (ObjectUtils.isEmpty(payload.getId())){
            throw TwException.error("", "权限规则主键不能为空");
        }
    }

    /**
     * 校验查询
     * @param query
     */
    private void checkQuery(PrdSystemPermissionRuleQuery query) {
        if (ObjectUtils.isEmpty(query.getFunctionId())){
            throw TwException.error("", "functionId不能为空");
        }
    }

    /**
     * 校验规则数据绑定入参
     * @param payload
     */
    private void checkBindDataRule(PrdSystemPermissionRulePayload payload) {
        if (ObjectUtils.isEmpty(payload.getFunctionId())){
            throw TwException.error("", "functionId不能为空");
        }
        if (!StringUtils.hasText(payload.getRuleCode())){
            throw TwException.error("", "ruleCode不能为空");
        }
    }

    @Override
    public List<PrdSystemPermissionRuleVO> getAllByFunctionCode(String functionCode) {
        String redisKey = RULE_CACHE_PREFIX + functionCode;
        // 优先获取缓存
        if(redisUtils.hasKey(redisKey)){
            return (List<PrdSystemPermissionRuleVO>)redisUtils.get(redisKey);
        }
        List<PrdSystemPermissionRuleVO> allByObjectCode = dao.getAllByFunctionCode(functionCode);
        if(CollectionUtils.isEmpty(allByObjectCode)){
            return Collections.emptyList();
        }
        // 查询明细
        allByObjectCode.forEach(rule -> rule.setDetailList(dao.queryPermissionRuleDetailByRuleId(rule.getId())));

        // 设置缓存
        redisUtils.set(redisKey, allByObjectCode);
        return allByObjectCode;
    }

    /**
     * 根据用户查询数据权限
     *
     * @param query
     * @return
     */
    @Override
    public PagingVO<PrdSystemPermissionRuleVO> pagingByUser(PrdSystemPermissionRuleQuery query) {
        //检查参数
        checkQueryByUser(query);
        Map<Long, String> map = new HashMap<>();
        //获取用户所有组织
        List<Long> orgLongs = dao.getAllOrgByUserId(query);
        query.setOrgIds(orgLongs.stream().map(String::valueOf).collect(Collectors.toList()));
        PagingVO<PrdSystemPermissionRuleVO> pagingVO = dao.queryPermissionRuleByUser(query);
        List<PrdSystemPermissionRuleVO> records = pagingVO.getRecords();
        for (PrdSystemPermissionRuleVO record : records) {
            if (map.containsKey(record.getObjectId())) {
                record.setObjectName(map.get(record.getObjectId()));
                continue;
            }
            PrdSystemBusinessObjectDO objectDO = objectRepo.findById(record.getObjectId()).orElseGet(PrdSystemBusinessObjectDO::new);
            record.setObjectName(objectDO.getObjectName());
            map.put(objectDO.getId(), objectDO.getObjectName());
        }
        return pagingVO;
    }

    /**
     * 查询权限字段
     *
     * @param query
     * @return
     */
    @Override
    public List<PrdSystemPermissionFieldVO> listField(PrdSystemPermissionFieldQuery query) {
        //检查查询参数
        checkListField(query);
        Map<String, List<PrdSystemPermissionFieldVO>> map = new HashMap<>();
        List<PrdSystemPermissionFieldVO> result = new ArrayList<>();
        List<PrdSystemPermissionFieldVO> list = dao.queryPermissionFieldByFunctionId(query);
        for (PrdSystemPermissionFieldVO fieldVO : list) {
            if (map.containsKey(fieldVO.getFieldType())) {
                //如果存在的话
                List<PrdSystemPermissionFieldVO> fieldVOS = map.get(fieldVO.getFieldType());
                fieldVOS.add(fieldVO);
            }else {
                //如果不存在的话
                PrdSystemPermissionFieldVO fieldVO1 = new PrdSystemPermissionFieldVO();
                fieldVO1.setFieldType(fieldVO.getFieldType());
                List<PrdSystemPermissionFieldVO> fieldVOS = new ArrayList<>();
                fieldVOS.add(fieldVO);
                fieldVO1.setList(fieldVOS);
                fieldVO1.setField(fieldVO.getFieldType());
                fieldVO1.setFieldName(PermissionFieldType.find(fieldVO.getFieldType()).getDesc());
                result.add(fieldVO1);
                map.put(fieldVO.getFieldType(), fieldVOS);
            }
        }
        return result;
    }

    /**
     * 检查字段查询参数
     * @param query
     */
    private void checkListField(PrdSystemPermissionFieldQuery query) {
        if (ObjectUtils.isEmpty(query.getFunctionId())) {
            throw TwException.error("", "functionId不能为空");
        }
    }

    /**
     * 检查查询用户数据
     * @param query
     */
    private void checkQueryByUser(PrdSystemPermissionRuleQuery query) {
        if (ObjectUtils.isEmpty(query.getUserId())) {
            throw TwException.error("", "userId不能为空");
        }
        if (ObjectUtils.isEmpty(query.getRoleId())) {
            throw TwException.error("", "roleId不能为空");
        }
    }

    /**
     * 清除权限规则缓存
     * @param functionCode 业务对象编码
     */
    private void removeRuleCache(String functionCode){
        String redisKey = RULE_CACHE_PREFIX + functionCode;
        if(redisUtils.hasKey(redisKey)){
            redisUtils.del(redisKey);
        }
    }

    @Override
    public List<PrdSystemPermissionFieldObjRoleFunctionVO> getPermissionFieldRule(String objectCode, String className, String functionCode) {
        String redisKey = FIELD_RULE_CACHE_PREFIX + objectCode;
        // 优先获取缓存
        if(redisUtils.hasKey(redisKey)){
            var result =  redisUtils.hget(redisKey, functionCode + "_" + className);
            if(Objects.nonNull(result)) {
                return (List<PrdSystemPermissionFieldObjRoleFunctionVO>) result;
            }
        }

        List<PrdSystemPermissionFieldObjRoleFunctionVO> permissionFieldRule = dao.getPermissionFieldRule(objectCode, className, functionCode);
        if(CollectionUtils.isEmpty(permissionFieldRule)){
            return Collections.emptyList();
        }

        // 设置缓存
        redisUtils.hset(redisKey, functionCode + "_" + className, permissionFieldRule);

        return permissionFieldRule;
    }

    /**
     * 清除字段权限缓存
     * @param objectCode 业务对象编码
     */
    private void removeFieldRuleCache(String objectCode){
        String redisKey = FIELD_RULE_CACHE_PREFIX + objectCode;
        if(redisUtils.hasKey(redisKey)){
            redisUtils.del(redisKey);
        }
    }
}
