package com.elitesland.tw.tw5.server.provider.event.support;

import com.elitesland.tw.tw5.server.provider.event.BaseEvent;
import com.elitesland.tw.tw5.server.provider.event.BaseEventHandler;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.context.ApplicationListener;
import org.springframework.util.StopWatch;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;

/**
 * 事件监听器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/14
 */
@Log4j2
public class EventListener<E extends BaseEvent> implements ApplicationListener<E> {

    private final ObjectMapper objectMapper;
    private final EventProperties eventProperties;
    private final Map<Class<E>, List<BaseEventHandler<E>>> handlerMap = new HashMap<>();

    public EventListener(ObjectProvider<BaseEventHandler<?>> eventHandlers, EventProperties eventProperties,
                         ObjectMapper objectMapper) {
        this.eventProperties = eventProperties;
        this.objectMapper = objectMapper;

        init(eventHandlers);
    }

    @Override
    public void onApplicationEvent(E event) {
        log.info("发生事件：{}，事件详细内容：{}", event::getEvent, () -> {
            try {
                return objectMapper.writeValueAsString(event);
            } catch (JsonProcessingException e) {
                log.error("序列化事件【" + event.getEvent() + "】信息异常：", e);
            }
            return "";
        });
        if (!Boolean.TRUE.equals(eventProperties.getEnabled())) {
            log.warn("未启用事件处理！");
            return;
        }

        StopWatch watch = new StopWatch();
        watch.start();

        handleEvent(event);

        watch.stop();
        log.info("事件{}处理完毕，用时{}s！", event.getEvent(), watch.getTotalTimeMillis() / 1000);
    }

    private void init(ObjectProvider<BaseEventHandler<?>> eventHandlers) {
        eventHandlers.orderedStream()
                .forEach(e -> {
                    Class<E> eventType = obtainParameterizedType(e);
                    handlerMap.computeIfAbsent(eventType, t -> new ArrayList<>()).add((BaseEventHandler<E>) e);
                });
    }

    private Class<E> obtainParameterizedType(BaseEventHandler<?> handler) {
        var interfaces = handler.getClass().getGenericInterfaces();
        if (interfaces.length > 0) {
            for (Type inter : interfaces) {
                if (inter instanceof ParameterizedType t) {
                    var typeArguments = t.getActualTypeArguments();
                    if (typeArguments.length > 0) {
                        for (Type typeArgument : typeArguments) {
                            if (BaseEvent.class.isAssignableFrom((Class<?>) typeArgument)) {
                                return (Class<E>) typeArgument;
                            }
                        }
                    }
                }
            }
        }

        throw new IllegalStateException(handler.getClass().getName() + "缺少必要的泛型参数");
    }

    private void handleEvent(E event) {
        var handlers = handlerMap.getOrDefault(event.getClass(), Collections.emptyList());
        if (handlers.isEmpty()) {
            log.warn("未发现事件{}的有效处理器！", event.getEvent());
            return;
        }

        for (BaseEventHandler<E> handler : handlers) {
            handler.handle(event);
        }
    }
}
