package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsInspectionItemConfigPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsInspectionItemConfigQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsInspectionItemConfigVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsInspectionItemConfigDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsInspectionItemConfigDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsInspectionItemConfigRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 结项检查项配置
 *
 * @author xxb
 * @date 2023-11-27
 */
@Repository
@RequiredArgsConstructor
public class PmsInspectionItemConfigDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsInspectionItemConfigRepo repo;
    private final QPmsInspectionItemConfigDO qdo = QPmsInspectionItemConfigDO.pmsInspectionItemConfigDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsInspectionItemConfigVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsInspectionItemConfigVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 工作类型编号
                qdo.workTypeCodes,
                // 工作类型名称
                qdo.workTypeNames,
                // 检查事项
                qdo.checkItem,
                // 检查方式
                qdo.checkMethod,
                // 检查说明
                qdo.checkDescription,
                // 结项审批节点
                qdo.approvalNodes,
                qdo.approvalNodeNames,
                qdo.sortNo
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsInspectionItemConfigVO> getJpaQueryWhere(PmsInspectionItemConfigQuery query) {
        JPAQuery<PmsInspectionItemConfigVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        List<OrderItem> orderItemList = new ArrayList<>();
        OrderItem orderItem = OrderItem.asc("sortNo");
        orderItemList.add(orderItem);
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderItemList));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsInspectionItemConfigQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsInspectionItemConfigQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 工作类型编号 精确 */
        if (!ObjectUtils.isEmpty(query.getWorkTypeCodes())) {
            list.add(qdo.workTypeCodes.like(SqlUtil.toSqlLikeString(query.getWorkTypeCodes())));
        }
        /** 工作类型名称 精确 */
        if (!ObjectUtils.isEmpty(query.getWorkTypeNames())) {
            list.add(qdo.workTypeNames.eq(query.getWorkTypeNames()));
        }
        /** 检查事项 精确 */
        if (!ObjectUtils.isEmpty(query.getCheckItem())) {
            list.add(qdo.checkItem.eq(query.getCheckItem()));
        }
        /** 检查方式 精确 */
        if (!ObjectUtils.isEmpty(query.getCheckMethod())) {
            list.add(qdo.checkMethod.eq(query.getCheckMethod()));
        }
        /** 检查说明 精确 */
        if (!ObjectUtils.isEmpty(query.getCheckDescription())) {
            list.add(qdo.checkDescription.eq(query.getCheckDescription()));
        }
        /** 结项审批节点 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovalNodes())) {
            list.add(qdo.approvalNodes.like(SqlUtil.toSqlLikeString(query.getApprovalNodes())));
        }
        if (!ObjectUtils.isEmpty(query.getApprovalNodeNames())) {
            list.add(qdo.approvalNodeNames.like(SqlUtil.toSqlLikeString(query.getApprovalNodeNames())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsInspectionItemConfigVO queryByKey(Long id) {
        JPAQuery<PmsInspectionItemConfigVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }


    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsInspectionItemConfigVO> queryListDynamic(PmsInspectionItemConfigQuery query) {
        JPAQuery<PmsInspectionItemConfigVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsInspectionItemConfigVO> queryPaging(PmsInspectionItemConfigQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsInspectionItemConfigVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsInspectionItemConfigVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsInspectionItemConfigVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsInspectionItemConfigDO save(PmsInspectionItemConfigDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsInspectionItemConfigDO> saveAll(List<PmsInspectionItemConfigDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsInspectionItemConfigPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 工作类型编号
        if (payload.getWorkTypeCodes() != null) {
            update.set(qdo.workTypeCodes, payload.getWorkTypeCodes());
        }
        // 工作类型名称
        if (payload.getWorkTypeNames() != null) {
            update.set(qdo.workTypeNames, payload.getWorkTypeNames());
        }
        // 检查事项
        if (payload.getCheckItem() != null) {
            update.set(qdo.checkItem, payload.getCheckItem());
        }
        // 检查方式
        if (payload.getCheckMethod() != null) {
            update.set(qdo.checkMethod, payload.getCheckMethod());
        }
        // 检查说明
        if (payload.getCheckDescription() != null) {
            update.set(qdo.checkDescription, payload.getCheckDescription());
        }
        // 结项审批节点
        if (payload.getApprovalNodes() != null) {
            update.set(qdo.approvalNodes, payload.getApprovalNodes());
        }
        if (payload.getApprovalNodeNames() != null) {
            update.set(qdo.approvalNodeNames, payload.getApprovalNodeNames());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 工作类型编号
            if (nullFields.contains("workTypeCodes")) {
                update.setNull(qdo.workTypeCodes);
            }
            // 工作类型名称
            if (nullFields.contains("workTypeNames")) {
                update.setNull(qdo.workTypeNames);
            }
            // 检查事项
            if (nullFields.contains("checkItem")) {
                update.setNull(qdo.checkItem);
            }
            // 检查方式
            if (nullFields.contains("checkMethod")) {
                update.setNull(qdo.checkMethod);
            }
            // 检查说明
            if (nullFields.contains("checkDescription")) {
                update.setNull(qdo.checkDescription);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

