package com.elitesland.tw.tw5.server.prd.pms.dao;


import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectTemplateActPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectTemplateActQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectTemplateActVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectTemplateActDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectTemplateActDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectTemplateActRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目模板活动
 *
 * @author xxb
 * @date 2023-08-03
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectTemplateActDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectTemplateActRepo repo;
    private final QPmsProjectTemplateActDO qdo = QPmsProjectTemplateActDO.pmsProjectTemplateActDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectTemplateActVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectTemplateActVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 模板Id
                qdo.templateId,
                // 活动编号
                qdo.actCode,
                // 活动名称
                qdo.actName,
                // 规划天数
                qdo.plannedDays,
                // 规划当量
                qdo.planningEquivalent,
                // 里程碑标记 1是 0不是
                qdo.milestoneFlag,
                // 阶段标记 1是 0不是
                qdo.actStageFlag,
                // 序号
                qdo.sortNo
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectTemplateActVO> getJpaQueryWhere(PmsProjectTemplateActQuery query) {
        JPAQuery<PmsProjectTemplateActVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, getOrders(query.getOrders())));
            return jpaQuery;
    }


    /**
     * 排序
     * @param orders
     * @return
     */
    private List<OrderItem> getOrders(List<OrderItem> orders){
        if (ObjectUtils.isEmpty(orders)){
            // 序号 升序
            orders = new ArrayList<>();
            OrderItem orderItem = new OrderItem();
            orderItem.setColumn("sortNo");
            orders.add(orderItem);
        }
        return orders;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectTemplateActQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectTemplateActQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 模板Id 精确 */
        if (!ObjectUtils.isEmpty(query.getTemplateId())) {
            list.add(qdo.templateId.eq(query.getTemplateId()));
        }
        /** 活动编号 精确 */
        if (!ObjectUtils.isEmpty(query.getActCode())) {
            list.add(qdo.actCode.eq(query.getActCode()));
        }
        /** 活动名称 精确 */
        if (!ObjectUtils.isEmpty(query.getActName())) {
            list.add(qdo.actName.eq(query.getActName()));
        }
        /** 规划天数 精确 */
        if (!ObjectUtils.isEmpty(query.getPlannedDays())) {
            list.add(qdo.plannedDays.eq(query.getPlannedDays()));
        }
        /** 规划当量 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanningEquivalent())) {
            list.add(qdo.planningEquivalent.eq(query.getPlanningEquivalent()));
        }
        /** 里程碑标记 1是 0不是 精确 */
        if (!ObjectUtils.isEmpty(query.getMilestoneFlag())) {
            list.add(qdo.milestoneFlag.eq(query.getMilestoneFlag()));
        }
        /** 阶段标记 1是 0不是 精确 */
        if (!ObjectUtils.isEmpty(query.getActStageFlag())) {
            list.add(qdo.actStageFlag.eq(query.getActStageFlag()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectTemplateActVO queryByKey(Long id) {
        JPAQuery<PmsProjectTemplateActVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param templateId 主键
     * @return 结果
     */
    public List<PmsProjectTemplateActVO> queryByTemplateId(Long templateId) {
        JPAQuery<PmsProjectTemplateActVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.templateId.eq(templateId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, getOrders(null)));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectTemplateActVO> queryListDynamic(PmsProjectTemplateActQuery query) {
        JPAQuery<PmsProjectTemplateActVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectTemplateActVO> queryPaging(PmsProjectTemplateActQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PmsProjectTemplateActVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectTemplateActVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectTemplateActVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectTemplateActDO save(PmsProjectTemplateActDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectTemplateActDO> saveAll(List<PmsProjectTemplateActDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectTemplateActPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 模板Id
        if (payload.getTemplateId() != null) {
            update.set(qdo.templateId, payload.getTemplateId());
        }
        // 活动编号
        if (payload.getActCode() != null) {
            update.set(qdo.actCode, payload.getActCode());
        }
        // 活动名称
        if (payload.getActName() != null) {
            update.set(qdo.actName, payload.getActName());
        }
        // 规划天数
        if (payload.getPlannedDays() != null) {
            update.set(qdo.plannedDays, payload.getPlannedDays());
        }
        // 规划当量
        if (payload.getPlanningEquivalent() != null) {
            update.set(qdo.planningEquivalent, payload.getPlanningEquivalent());
        }
        // 里程碑标记 1是 0不是
        if (payload.getMilestoneFlag() != null) {
            update.set(qdo.milestoneFlag, payload.getMilestoneFlag());
        }
        // 阶段标记 1是 0不是
        if (payload.getActStageFlag() != null) {
            update.set(qdo.actStageFlag, payload.getActStageFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 模板Id
            if (nullFields.contains("templateId")) {
                update.setNull(qdo.templateId);
            }
            // 活动编号
            if (nullFields.contains("actCode")) {
                update.setNull(qdo.actCode);
            }
            // 活动名称
            if (nullFields.contains("actName")) {
                update.setNull(qdo.actName);
            }
            // 规划天数
            if (nullFields.contains("plannedDays")) {
                update.setNull(qdo.plannedDays);
            }
            // 规划当量
            if (nullFields.contains("planningEquivalent")) {
                update.setNull(qdo.planningEquivalent);
            }
            // 里程碑标记 1是 0不是
            if (nullFields.contains("milestoneFlag")) {
                update.setNull(qdo.milestoneFlag);
            }
            // 阶段标记 1是 0不是
            if (nullFields.contains("actStageFlag")) {
                update.setNull(qdo.actStageFlag);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

