package com.elitesland.tw.tw5.server.prd.salecon.controller;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseContractManagerVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.SaleConContractPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.SaleConContractQuery;
import com.elitesland.tw.tw5.api.prd.salecon.service.SaleConContractService;
import com.elitesland.tw.tw5.api.prd.salecon.vo.SaleConContractVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemLogVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.permission.annotation.FunctionDetail;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionDomain;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionFunction;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionCodeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionTypeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionDomainEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionTypeEnum;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;

/**
 * 销售合同
 *
 * @author danting
 * @date 2023-03-23
 */
@Api(tags = "销售合同")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/contract/sale")
@Slf4j
@PermissionDomain(domain = PermissionDomainEnum.SALE_CON_CONTRACT)
public class SaleConContractController {

    private final SaleConContractService saleConContractService;

    /**
     * 新增
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping
    @ApiOperation("新增")
    public TwOutputUtil insert(@RequestBody SaleConContractPayload payload) {
        return TwOutputUtil.ok(saleConContractService.insert(payload));
    }

    /**
     * 更新
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping
    @ApiOperation("更新")
    public TwOutputUtil update(@RequestBody SaleConContractPayload payload) {
        return TwOutputUtil.ok(saleConContractService.update(payload));
    }

    /**
     * 动态更新
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("updateByKeyDynamic")
    @ApiOperation("更新")
    public TwOutputUtil updateByKeyDynamic(@RequestBody SaleConContractPayload payload) {
        return TwOutputUtil.ok(saleConContractService.updateByKeyDynamic(payload));
    }

    /**
     * 更新信息（额定当量、额定费用等，无状态校验）
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("updateInfoByKey")
    @ApiOperation("更新信息（额定当量、额定费用等，无状态校验）")
    public TwOutputUtil updateInfoByKeyDynamic(@RequestBody SaleConContractPayload payload) {
        return TwOutputUtil.ok(saleConContractService.updateInfoByKeyDynamic(payload));
    }

    /**
     * 主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/{key}")
    @UdcNameClass
    @ApiOperation("主键查询")
    public TwOutputUtil queryOneByKey(@PathVariable Long key) {
        return TwOutputUtil.ok(saleConContractService.queryByKey(key, true));
    }

    /**
     * 分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/paging")
    @UdcNameClass
    @ApiOperation("分页")
    @PermissionFunction(functions = {
            @FunctionDetail(functionCode = FunctionCodeEnum.SALE_CON_CONTRACT_PAGE, type = FunctionTypeEnum.TAB)
    }, fieldPermission = true, permissionType = PermissionTypeEnum.DATA_PERMISSION)
    public TwOutputUtil<PagingVO<SaleConContractVO>> paging(@RequestBody SaleConContractQuery query) {
        log.info("销售合同分页api层入参合同名称 is {},合同状态 is {}", query.getName(), query.getStatusList());
        return TwOutputUtil.ok(saleConContractService.paging(query));
    }

    /**
     * 批量导出活动excel
     *
     * @param query 查询条件
     * @return result
     */
    @PostMapping("/downloadSaleCon")
    public void downloadSaleCon(HttpServletResponse response, @RequestBody SaleConContractQuery query) {
        saleConContractService.downloadSaleCon(response, query);
    }

    /**
     * 查询列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/list")
    @UdcNameClass
    @ApiOperation("查询列表")
    public TwOutputUtil queryList(SaleConContractQuery query) {
        return TwOutputUtil.ok(saleConContractService.queryListDynamic(query));
    }

    /**
     * 根据父ID查询子合同列表
     *
     * @param parentId
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/list/{parentId}")
    @UdcNameClass
    @ApiOperation("根据父ID查询子合同列表")
    public TwOutputUtil queryListByConId(@PathVariable Long parentId) {
        List<SaleConContractVO> voList = saleConContractService.queryListByConId(parentId);
        return TwOutputUtil.ok(voList);
    }

    /**
     * 删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping("/deleteSoft")
    @ApiOperation("删除")
    public TwOutputUtil deleteSoft(Long[] keys) {
        saleConContractService.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 激活
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/active")
//    //@GlobalTransactional
    @ApiOperation("激活")
    public TwOutputUtil active(Long[] keys) {
        saleConContractService.active(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 暂挂
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/pending")
    @ApiOperation("暂挂")
    public TwOutputUtil pending(Long[] keys) {
        saleConContractService.pending(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 关闭合同-带流程
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/closeWithWorkFlow")
//    //@GlobalTransactional
    @ApiOperation("关闭")
    public TwOutputUtil closeWithWorkFlow(@RequestParam Long[] keys, @RequestParam(required = false) String closeCacheKey, @RequestParam(required = false) String authToken) {
        saleConContractService.closeWithWorkFlow(Arrays.asList(keys), closeCacheKey, authToken);
        return TwOutputUtil.ok();
    }
    /**
     * 关闭
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/close")
//    //@GlobalTransactional
    @ApiOperation("关闭")
    public TwOutputUtil close(@RequestParam Long[] keys, @RequestParam(required = false) String closeCacheKey, @RequestParam(required = false) String authToken) {
        saleConContractService.close(Arrays.asList(keys), closeCacheKey, authToken);
        return TwOutputUtil.ok();
    }

    /**
     * 作废带工作流
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/invalidWithWorkFlow")
//    //@GlobalTransactional
    @ApiOperation("作废带流程")
    public TwOutputUtil invalidWithWorkFlow(Long[] keys) {
        saleConContractService.invalidWithWorkFlow(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 作废
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/invalid")
//    //@GlobalTransactional
    @ApiOperation("作废")
    public TwOutputUtil invalid(Long[] keys) {
        saleConContractService.invalid(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }
    /**
     * 变更
     *
     * @param payload 单据
     * @return result
     */
    @PostMapping("/change")
//    //@GlobalTransactional
    @ApiOperation("子合同变更")
    public TwOutputUtil change(@RequestBody SaleConContractPayload payload) {
        saleConContractService.change(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 变更
     *
     * @param payload 单据
     * @return result
     */
    @PostMapping("/updateChange")
//    //@GlobalTransactional
    @ApiOperation("子合同变更-更新")
    public TwOutputUtil updateChange(@RequestBody SaleConContractPayload payload) {
        saleConContractService.updateChange(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 变更-查询详情
     *
     * @param businessKey 单据
     * @return result
     */
    @GetMapping("/queryChangeDetail")
    @ApiOperation("子合同变更-查询详情")
    public TwOutputUtil queryChangeDetail(Long businessKey) {
        return TwOutputUtil.ok(saleConContractService.queryChangeDetail(businessKey));
    }

    /**
     * 操作记录日志列表
     *
     * @param key 合同主键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/log/list/{key}")
    @ApiOperation("操作记录日志列表")
    public TwOutputUtil logList(@PathVariable Long key) {
        final List<PrdSystemLogVO> prdSystemLogVOS = saleConContractService.queryLogList(key);
        return TwOutputUtil.ok(prdSystemLogVOS);
    }

    /**
     * 生成子合同的默认名称 --用来向前端展示子合同的默认名称
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/subCode/{parentId}")
    @ApiOperation("生成子合同的默认名称")
    public TwOutputUtil subCode(@PathVariable Long parentId) {
        String subCode = saleConContractService.subCode(parentId);
        return TwOutputUtil.ok(subCode);
    }


    /**
     * 合同归档
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/conFiling")
    @ApiOperation("归档")
    public TwOutputUtil conFiling(Long[] keys) {
        saleConContractService.conFiling(keys);
        return TwOutputUtil.ok();
    }

    /**
     * 提交虚拟合同
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/fictitiousConSubmit")
//    //@GlobalTransactional
    @ApiOperation("提交虚拟合同")
    public TwOutputUtil fictitiousConSubmit(@RequestParam Long[] keys) {
        saleConContractService.fictitiousConSubmit(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }


    /**
     * 采购合同查询
     *
     * @param saleconId 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/purchase/con/{saleconId}")
    @UdcNameClass
    @ApiOperation("查询采购合同列表")
    public TwOutputUtil<List<PurchaseContractManagerVO>> queryPurchConList(@PathVariable Long saleconId) {
        return TwOutputUtil.ok(saleConContractService.queryPurchConList(saleconId));
    }
}
