package com.elitesland.tw.tw5.server.prd.acc.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimDetailPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimDetailQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimDetailVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.*;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 费用报销管理明细
 *
 * @author sunxw
 * @date 2023-11-23
 */
@Repository
@RequiredArgsConstructor
public class AccReimDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimDetailRepo repo;
    private final QAccReimDetailDO qdo = QAccReimDetailDO.accReimDetailDO;
    private final QAccBudgetItemDO qBudgetItemDO = QAccBudgetItemDO.accBudgetItemDO;
    private final QAccBusinessItemDO qBusinessItemDO = QAccBusinessItemDO.accBusinessItemDO;
    private final QAccFinancialSubjectDO qFinancialSubjectDO = QAccFinancialSubjectDO.accFinancialSubjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimDetailVO.class,
                        qdo.id,
                        //qdo.remark,
                        //qdo.createUserId,
                        //qdo.creator,
                        //qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 报销单ID
                        qdo.masId,
                        // 费用日期
                        qdo.expenseDate,
                        qdo.expensePlace,
                        // 核算项目
                        qdo.busAccItemId,
                        qBusinessItemDO.businessCode.as("busAccItemCode"),
                        qBusinessItemDO.businessName.as("busAccItemName"),
                        // 预算项目
                        qdo.budgetItemId,
                        qBudgetItemDO.budgetName.as("budgetItemName"),
                        // 会计科目
                        qdo.finAccSubjId,
                        qFinancialSubjectDO.accName.as("finAccSubjName"),
                        // 报销说明
                        qdo.reimRemark,
                        // 报销金额(含税)
                        qdo.reimAmt,
                        // 税额
                        qdo.taxAmt,
                        //支付币种
                        qdo.currCode,
                        //税率
                        qdo.taxRate,
                        //发票金额
                        qdo.invAmt,
                        // 折扣
                        qdo.invLimit,
                        // 调整后金额
                        qdo.adjustAmt,
                        // 连号发票列表
                        qdo.continuousInvoiceNos,
                        // 提醒文本
                        qdo.remindText,
                        // 报错警告文本
                        qdo.errorText,
                        // 城市
                        qdo.expensePlaceGrade,
                        //报销额度
                        qdo.reimLimit,
                        //是否超额
                        qdo.excessLimit,
                        //不等于
                        qdo.notEqualFlag,
                        // 逾期报销配置id
                        qdo.reimSettingOverdueId,
                        // 连号报销配置id
                        qdo.reimSettingContinousId,
                        // 报销住宿额度配置id
                        qdo.reimSettingLimitId,
                        // 发票抬头配置id
                        qdo.reimSettingTitleId,
                        // 发票张数
                        qdo.invoiceNum,
                        //无发票原因
                        qdo.noinvReason,
                        // 计算金额来源ID
                        qdo.calcAmtSourceId,
                        qdo.checkResult,
                        // 费用承担人
                        qdo.costPayers,
                        qdo.reasonDetailType,
                        qdo.reasonDetailId,
                        qdo.reasonDetailName
                )).from(qdo)
                .leftJoin(qBudgetItemDO).on(qdo.budgetItemId.eq(qBudgetItemDO.id).and(qBudgetItemDO.deleteFlag.eq(0)))
                .leftJoin(qBusinessItemDO).on(qdo.busAccItemId.eq(qBusinessItemDO.id).and(qBusinessItemDO.deleteFlag.eq(0)))
                .leftJoin(qFinancialSubjectDO).on(qdo.finAccSubjId.eq(qFinancialSubjectDO.id).and(qFinancialSubjectDO.deleteFlag.eq(0)))
                ;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimDetailVO> getJpaQueryWhere(AccReimDetailQuery query) {
        JPAQuery<AccReimDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimDetailQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 报销单ID 精确 */
        if (!ObjectUtils.isEmpty(query.getMasId())) {
            list.add(qdo.masId.eq(query.getMasId()));
        }
        /** 费用日期 精确 */
        if (!ObjectUtils.isEmpty(query.getExpenseDate())) {
            list.add(qdo.expenseDate.eq(query.getExpenseDate()));
        }
        /** 核算项目 精确 */
        if (!ObjectUtils.isEmpty(query.getBusAccItemId())) {
            list.add(qdo.busAccItemId.eq(query.getBusAccItemId()));
        }
        /** 预算项目 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetItemId())) {
            list.add(qdo.budgetItemId.eq(query.getBudgetItemId()));
        }
        /** 会计科目 精确 */
        if (!ObjectUtils.isEmpty(query.getFinAccSubjId())) {
            list.add(qdo.finAccSubjId.eq(query.getFinAccSubjId()));
        }
        /** 报销说明 精确 */
        if (!ObjectUtils.isEmpty(query.getReimRemark())) {
            list.add(qdo.reimRemark.eq(query.getReimRemark()));
        }
        /** 报销金额 精确 */
        if (!ObjectUtils.isEmpty(query.getReimAmt())) {
            list.add(qdo.reimAmt.eq(query.getReimAmt()));
        }
        /** 税额 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxAmt())) {
            list.add(qdo.taxAmt.eq(query.getTaxAmt()));
        }
        /** 发票张数 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceNum())) {
            list.add(qdo.invoiceNum.eq(query.getInvoiceNum()));
        }
        /** 计算金额来源ID 精确 */
        if (!ObjectUtils.isEmpty(query.getCalcAmtSourceId())) {
            list.add(qdo.calcAmtSourceId.eq(query.getCalcAmtSourceId()));
        }
        if (!ObjectUtils.isEmpty(query.getCostPayers())) {
            list.add(qdo.costPayers.like(SqlUtil.toSqlLikeString(query.getCostPayers())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimDetailVO queryByKey(Long id) {
        JPAQuery<AccReimDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主表id查询
     *
     * @param masId
     * @return
     */
    public List<AccReimDetailVO> queryByMasId(Long masId) {
        JPAQuery<AccReimDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.masId.eq(masId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据主表id集合查询
     *
     * @param masIds
     * @return
     */
    public List<AccReimDetailVO> queryByMasIds(List<Long> masIds) {
        JPAQuery<AccReimDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.masId.in(masIds));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimDetailVO> queryListDynamic(AccReimDetailQuery query) {
        JPAQuery<AccReimDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimDetailVO> queryPaging(AccReimDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimDetailDO save(AccReimDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimDetailDO> saveAll(List<AccReimDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 报销单ID
        if (payload.getMasId() != null) {
            update.set(qdo.masId, payload.getMasId());
        }
        // 费用日期
        if (payload.getExpenseDate() != null) {
            update.set(qdo.expenseDate, payload.getExpenseDate());
        }
        // 核算项目
        if (payload.getBusAccItemId() != null) {
            update.set(qdo.busAccItemId, payload.getBusAccItemId());
        }
        // 预算项目
        if (payload.getBudgetItemId() != null) {
            update.set(qdo.budgetItemId, payload.getBudgetItemId());
        }
        // 会计科目
        if (payload.getFinAccSubjId() != null) {
            update.set(qdo.finAccSubjId, payload.getFinAccSubjId());
        }
        // 报销说明
        if (payload.getReimRemark() != null) {
            update.set(qdo.reimRemark, payload.getReimRemark());
        }
        // 报销金额
        if (payload.getReimAmt() != null) {
            update.set(qdo.reimAmt, payload.getReimAmt());
        }
        // 税额
        if (payload.getTaxAmt() != null) {
            update.set(qdo.taxAmt, payload.getTaxAmt());
        }
        //税率
        if (payload.getTaxRate() != null) {
            update.set(qdo.taxRate, payload.getTaxRate());
        }
        //费用发生地
        if (payload.getExpensePlace() != null) {
            update.set(qdo.expensePlace, payload.getExpensePlace());
        }
        //发票金额
        if (payload.getInvAmt() != null) {
            update.set(qdo.invAmt, payload.getInvAmt());
        }
        //发票金额
        if (payload.getInvLimit() != null) {
            update.set(qdo.invLimit, payload.getInvLimit());
        }
        //调整后金额
        if (payload.getAdjustAmt() != null) {
            update.set(qdo.adjustAmt, payload.getAdjustAmt());
        }
        // 连号发票列表
        if (payload.getContinuousInvoiceNos() != null) {
            update.set(qdo.continuousInvoiceNos, payload.getContinuousInvoiceNos());
        }
        // 提醒文本
        if (payload.getRemindText() != null) {
            update.set(qdo.remindText, payload.getRemindText());
        }
        // 报错警告文本
        if (payload.getErrorText() != null) {
            update.set(qdo.errorText, payload.getErrorText());
        }
        // 城市
        if (payload.getExpensePlaceGrade() != null) {
            update.set(qdo.expensePlaceGrade, payload.getExpensePlaceGrade());
        }
        //报销额度
        if (payload.getReimLimit() != null) {
            update.set(qdo.reimLimit, payload.getReimLimit());
        }
        //是否超额
        if (payload.getExcessLimit() != null) {
            update.set(qdo.excessLimit, payload.getExcessLimit());
        }
        //不等于
        if (payload.getNotEqualFlag() != null) {
            update.set(qdo.notEqualFlag, payload.getNotEqualFlag());
        }
        // 逾期报销配置id
        if (payload.getReimSettingOverdueId() != null) {
            update.set(qdo.reimSettingOverdueId, payload.getReimSettingOverdueId());
        }
        // 连号报销配置id
        if (payload.getReimSettingContinousId() != null) {
            update.set(qdo.reimSettingContinousId, payload.getReimSettingContinousId());
        }
        // 报销住宿额度配置id
        if (payload.getReimSettingLimitId() != null) {
            update.set(qdo.reimSettingLimitId, payload.getReimSettingLimitId());
        }
        // 发票抬头配置id
        if (payload.getReimSettingTitleId() != null) {
            update.set(qdo.reimSettingTitleId, payload.getReimSettingTitleId());
        }
        // 发票张数
        if (payload.getInvoiceNum() != null) {
            update.set(qdo.invoiceNum, payload.getInvoiceNum());
        }
        //无发票原因
        if (payload.getNoinvReason() != null) {
            update.set(qdo.noinvReason, payload.getNoinvReason());
        }
        //规则检查结果
        if (payload.getCheckResult() != null) {
            update.set(qdo.checkResult, payload.getCheckResult());
        }
        //支付币种
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        //费用承担人
        if (payload.getCostPayers() != null) {
            update.set(qdo.costPayers, payload.getCostPayers());
        }
        if (payload.getReasonDetailType() != null) {
            update.set(qdo.reasonDetailType, payload.getReasonDetailType());
        }
        if (payload.getReasonDetailId() != null) {
            update.set(qdo.reasonDetailId, payload.getReasonDetailId());
        }
        if (payload.getReasonDetailName() != null) {
            update.set(qdo.reasonDetailName, payload.getReasonDetailName());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 报销单ID
            if (nullFields.contains("masId")) {
                update.setNull(qdo.masId);
            }
            // 费用日期
            if (nullFields.contains("expenseDate")) {
                update.setNull(qdo.expenseDate);
            }
            // 核算项目
            if (nullFields.contains("busAccItemId")) {
                update.setNull(qdo.busAccItemId);
            }
            // 预算项目
            if (nullFields.contains("budgetItemId")) {
                update.setNull(qdo.budgetItemId);
            }
            // 会计科目
            if (nullFields.contains("finAccSubjId")) {
                update.setNull(qdo.finAccSubjId);
            }
            // 报销说明
            if (nullFields.contains("reimRemark")) {
                update.setNull(qdo.reimRemark);
            }
            // 报销金额
            if (nullFields.contains("reimAmt")) {
                update.setNull(qdo.reimAmt);
            }// 税额
            if (nullFields.contains("taxAmt")) {
                update.setNull(qdo.taxAmt);
            }
            // 发票张数
            if (nullFields.contains("invoiceNum")) {
                update.setNull(qdo.invoiceNum);
            }
            // 计算金额来源ID
            if (nullFields.contains("calcAmtSourceId")) {
                update.setNull(qdo.calcAmtSourceId);
            }
            // 费用承担人
            if (nullFields.contains("costPayers")) {
                update.setNull(qdo.costPayers);
            }
            if (nullFields.contains("reasonDetailType")) {
                update.setNull(qdo.reasonDetailType);
            }
            if (nullFields.contains("reasonDetailId")) {
                update.setNull(qdo.reasonDetailId);
            }
            if (nullFields.contains("reasonDetailName")) {
                update.setNull(qdo.reasonDetailName);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据主表id进行删除
     *
     * @param masId 主表id
     * @return 删除的行数
     */
    public long deleteSoftByMasId(Long masId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.masId.eq(masId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

