package com.elitesland.tw.tw5.server.prd.adm.service;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.adm.payload.AdmTripTicketDataPayload;
import com.elitesland.tw.tw5.api.prd.adm.payload.AdmTripTicketPayload;
import com.elitesland.tw.tw5.api.prd.adm.query.AdmTripTicketQuery;
import com.elitesland.tw.tw5.api.prd.adm.service.AdmTripTicketService;
import com.elitesland.tw.tw5.api.prd.adm.vo.AdmTripTicketVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.excel.ExcelUtil;
import com.elitesland.tw.tw5.server.prd.adm.convert.AdmTripTicketConvert;
import com.elitesland.tw.tw5.server.prd.adm.dao.AdmTripTicketDAO;
import com.elitesland.tw.tw5.server.prd.adm.entity.AdmTripTicketDO;
import com.elitesland.tw.tw5.server.prd.adm.repo.AdmTripTicketRepo;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 行政订票管理
 *
 * @author carl
 * @date 2023-10-30
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AdmTripTicketServiceImpl extends BaseServiceImpl implements AdmTripTicketService {

    private final AdmTripTicketRepo admTripTicketRepo;
    private final AdmTripTicketDAO admTripTicketDAO;
    private final UdcUtil udcUtil;

    @Override
    public PagingVO<AdmTripTicketVO> queryPaging(AdmTripTicketQuery query) {
        return admTripTicketDAO.queryPaging(query);
    }

    @Override
    public List<AdmTripTicketVO> queryListDynamic(AdmTripTicketQuery query) {
        return admTripTicketDAO.queryListDynamic(query);
    }

    @Override
    public AdmTripTicketVO queryByKey(Long key) {
        AdmTripTicketDO entity = admTripTicketRepo.findById(key).orElseGet(AdmTripTicketDO::new);
        Assert.notNull(entity.getId(), "不存在");
        AdmTripTicketVO vo = AdmTripTicketConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AdmTripTicketVO insertOrUpdate(AdmTripTicketPayload payload) {
        if (payload.getApplyId() == null) {
            throw TwException.error("", "出差申请id不可为空");
        }
        AdmTripTicketDO entityDo = AdmTripTicketConvert.INSTANCE.toDo(payload);
        return AdmTripTicketConvert.INSTANCE.toVo(admTripTicketRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AdmTripTicketVO update(AdmTripTicketPayload payload) {
        AdmTripTicketDO entity = admTripTicketRepo.findById(payload.getId()).orElseGet(AdmTripTicketDO::new);
        Assert.notNull(entity.getId(), "不存在");
        AdmTripTicketDO entityDo = AdmTripTicketConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return AdmTripTicketConvert.INSTANCE.toVo(admTripTicketRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(AdmTripTicketPayload payload) {
        AdmTripTicketDO entity = admTripTicketRepo.findById(payload.getId()).orElseGet(AdmTripTicketDO::new);
        Assert.notNull(entity.getId(), "不存在");
        long result = admTripTicketDAO.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            admTripTicketDAO.deleteSoft(keys);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<AdmTripTicketVO> bacthInsert(AdmTripTicketDataPayload payload) {
        if (payload == null) {
            throw TwException.error("", "订票数据不可为空");
        }
        if (payload.getApplyId() == null) {
            throw TwException.error("", "出差申请数据不可为空");
        }
        //全删全插
        //  admTripTicketDAO.deleteSoftByApplyId(payload.getApplyId());
        List<AdmTripTicketVO> admTripTicketVOS = admTripTicketDAO.queryListByApplyId(payload.getApplyId());

        if (!ObjectUtils.isEmpty(payload.getPayloads())) {
//            List<Long> collect = payload.getPayloads().stream().filter(data -> data.getId() != null).map(AdmTripTicketPayload::getId).collect(Collectors.toList());

            List<Long> collect = new ArrayList<>();
            List<AdmTripTicketDO> entityDos = new ArrayList<>();
            payload.getPayloads().forEach(payload0 -> {
                if (payload0.getId() != null) {
                    collect.add(payload0.getId());
                }
                payload0.setApplyId(payload.getApplyId());
                entityDos.add(AdmTripTicketConvert.INSTANCE.toDo(payload0));
            });
            //删除不操作的数据
            List<Long> delIds = admTripTicketVOS.stream().filter(vo -> !collect.contains(vo.getId())).map(AdmTripTicketVO::getId).collect(Collectors.toList());
            if (!ObjectUtils.isEmpty(delIds)) {
                admTripTicketDAO.deleteSoft(delIds);
            }

            List<AdmTripTicketDO> admTripTicketDOS = admTripTicketRepo.saveAll(entityDos);
            return AdmTripTicketConvert.INSTANCE.toVos(admTripTicketDOS);
        }
        return null;
    }

    @Override
    public void exportList(AdmTripTicketQuery query, HttpServletResponse response) throws IOException {
        //定义文件名称
        String sheetName = "行政订票报表";
        //导出顺序是反的
        OrderItem orderItem = OrderItem.asc("createTime");
        query.defaultOrder(orderItem);
        List<AdmTripTicketVO> admTripTicketVOS = queryListDynamic(query);
        if (admTripTicketVOS.size() > 50000) {
            throw TwException.error("500", "导出数据过大，请分批操作，目前导出数量：" + admTripTicketVOS.size());
        }
        List<AdmTripTicketVO> invBatchInvdtlVOS = udcUtil.translateList(admTripTicketVOS);
        for (AdmTripTicketVO invBatchInvdtlVO : invBatchInvdtlVOS) {
            invBatchInvdtlVO.setBookingDateStr(invBatchInvdtlVO.getBookingDate() == null ? "" : invBatchInvdtlVO.getBookingDate() + "");
            invBatchInvdtlVO.setTripDateStr(invBatchInvdtlVO.getTripDate() == null ? "" : invBatchInvdtlVO.getTripDate() + "");
            invBatchInvdtlVO.setReimTimeStr(invBatchInvdtlVO.getReimTime() == null ? "" : invBatchInvdtlVO.getReimTime() + "");

        }
        //对文件名进行固定格式编码
        String fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
        //设置请求响应内容类型
        //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        //设置请求响应内容编码方式
        response.setCharacterEncoding("utf-8");
        //文件下载，指定默认名
        response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

        final ExcelWriterSheetBuilder sheet = EasyExcel.write(response.getOutputStream(), AdmTripTicketVO.class)
                .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                .sheet(sheetName);
        // 列
        ExcelUtil.excelHelper(sheet, AdmTripTicketVO.class, null);
        //写入
        sheet.doWrite(invBatchInvdtlVOS);
    }

}
