package com.elitesland.tw.tw5.server.prd.borrow.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.borrow.payload.BorrowMoneyPayload;
import com.elitesland.tw.tw5.api.prd.borrow.query.BorrowMoneyQuery;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowMoneyVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.borrow.constant.BorrowStatusEnum;
import com.elitesland.tw.tw5.server.prd.borrow.entity.BorrowMoneyDO;
import com.elitesland.tw.tw5.server.prd.borrow.entity.QBorrowMoneyDO;
import com.elitesland.tw.tw5.server.prd.borrow.repo.BorrowMoneyRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * 借款管理
 *
 * @author wangly
 * @date 2024-07-15
 */
@Repository
@RequiredArgsConstructor
public class BorrowMoneyDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BorrowMoneyRepo repo;
    private final QBorrowMoneyDO qdo = QBorrowMoneyDO.borrowMoneyDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BorrowMoneyVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BorrowMoneyVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 借款编号
                qdo.borrowNo,
                // 借款名称
                qdo.borrowName,
                // 申请人
                qdo.applicantUserId,
                // 出让日期
                qdo.transferDate,
                // 受让人ID
                qdo.receiveUserId,
                // 受让人所属bu
                qdo.receiveBuId,
                // 申请日期
                qdo.applicantTime,
                // 申请人所属bu
                qdo.applicantBuId,
                // 费用BU
                qdo.costBuId,
                // 单据状态 udc[BORROW_MONEY:STATUS]
                qdo.apprStatus,
                // 业务类型 udc[BORROW_MONEY:BUSINESS_TYPE]
                qdo.businessType,
                // 是否出让
                qdo.transferFlag,
                // 还款人ID
                qdo.repaymentUserId,
                // 借款金额
                qdo.borrowAmt,
                // 已核销金额
                qdo.alreadyWriteOffAmt,
                // 未核销金额
                qdo.unwriteOffAmt,
                // 预计核销日期
                qdo.preWriteOffDate,
                // 附件
                qdo.fileCodes,
                // 收款账户
                qdo.accountNo,
                // 户名
                qdo.holderName,
                // 收款银行
                qdo.bankName,
                // 扩展字段1
                qdo.ext1,
                // 扩展字段2
                qdo.ext2,
                // 扩展字段3
                qdo.ext3,
                // 扩展字4
                qdo.ext4,
                // 扩展字5
                qdo.ext5,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 提交时间
                qdo.submitTime,
                // 审批时间
                qdo.approvedTime
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BorrowMoneyVO> getJpaQueryWhere(BorrowMoneyQuery query) {
        JPAQuery<BorrowMoneyVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BorrowMoneyQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BorrowMoneyQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 记录唯一ID 精确 */
        if (!CollectionUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 借款编号 精确 */
        if (!ObjectUtils.isEmpty(query.getBorrowNo())) {
            list.add(qdo.borrowNo.eq(query.getBorrowNo()));
        }
        /** 借款名称 精确 */
        if (!ObjectUtils.isEmpty(query.getBorrowName())) {
            list.add(qdo.borrowName.eq(query.getBorrowName()));
        }
        if (!ObjectUtils.isEmpty(query.getBorrowNoOrName())) {
            list.add((qdo.borrowNo.like(SqlUtil.toSqlLikeString(query.getBorrowNoOrName())).or(qdo.borrowName.like(SqlUtil.toSqlLikeString(query.getBorrowNoOrName())))));
        }
        /** 申请人 精确 */
        if (!ObjectUtils.isEmpty(query.getApplicantUserId())) {
            list.add(qdo.applicantUserId.eq(query.getApplicantUserId()));
        }
        /** 出让日期 精确 */
        if (!ObjectUtils.isEmpty(query.getTransferDate())) {
            list.add(qdo.transferDate.eq(query.getTransferDate()));
        }
        /** 受让人ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReceiveUserId())) {
            list.add(qdo.receiveUserId.eq(query.getReceiveUserId()));
        }
        /** 受让人所属bu 精确 */
        if (!ObjectUtils.isEmpty(query.getReceiveBuId())) {
            list.add(qdo.receiveBuId.eq(query.getReceiveBuId()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplicantTime())) {
            list.add(qdo.applicantTime.eq(query.getApplicantTime()));
        }
        /** 申请人所属bu 精确 */
        if (!ObjectUtils.isEmpty(query.getApplicantBuId())) {
            list.add(qdo.applicantBuId.eq(query.getApplicantBuId()));
        }
        /** 费用BU 精确 */
        if (!ObjectUtils.isEmpty(query.getCostBuId())) {
            list.add(qdo.costBuId.eq(query.getCostBuId()));
        }
        /** 单据状态 udc[BORROW_MONEY:STATUS] 精确 */
        if (!ObjectUtils.isEmpty(query.getApprStatus())) {
            if (BorrowStatusEnum.TRANSFER.getCode().equals(query.getApprStatus())) {
                list.add(qdo.transferFlag.eq(true));
            }else {
                list.add(qdo.apprStatus.eq(query.getApprStatus()));
            }
        }
        /** 单据状态 udc[BORROW_MONEY:STATUS] 精确 */
        if (!CollectionUtils.isEmpty(query.getApprStatusList())) {
            list.add(qdo.apprStatus.in(query.getApprStatusList()));
        }
        /** 业务类型 udc[BORROW_MONEY:BUSINESS_TYPE] 精确 */
        if (!ObjectUtils.isEmpty(query.getBusinessType())) {
            list.add(qdo.businessType.eq(query.getBusinessType()));
        }
        /** 是否出让 精确 */
        if (!ObjectUtils.isEmpty(query.getTransferFlag())) {
            list.add(qdo.transferFlag.eq(query.getTransferFlag()));
        }
        /** 还款人ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRepaymentUserId())) {
            list.add(qdo.repaymentUserId.eq(query.getRepaymentUserId()));
        }
        /** 借款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getBorrowAmt())) {
            list.add(qdo.borrowAmt.eq(query.getBorrowAmt()));
        }
        /** 已核销金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAlreadyWriteOffAmt())) {
            list.add(qdo.alreadyWriteOffAmt.eq(query.getAlreadyWriteOffAmt()));
        }
        /** 未核销金额 精确 */
        if (!ObjectUtils.isEmpty(query.getUnwriteOffAmt())) {
            list.add(qdo.unwriteOffAmt.eq(query.getUnwriteOffAmt()));
        }
        /** 预计核销日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPreWriteOffDate())) {
            list.add(qdo.preWriteOffDate.eq(query.getPreWriteOffDate()));
        }
        /** 附件 精确 */
        if (!ObjectUtils.isEmpty(query.getFileCodes())) {
            list.add(qdo.fileCodes.eq(query.getFileCodes()));
        }
        /** 收款账户 精确 */
        if (!ObjectUtils.isEmpty(query.getAccountNo())) {
            list.add(qdo.accountNo.eq(query.getAccountNo()));
        }
        /** 户名 精确 */
        if (!ObjectUtils.isEmpty(query.getHolderName())) {
            list.add(qdo.holderName.eq(query.getHolderName()));
        }
        /** 收款银行 精确 */
        if (!ObjectUtils.isEmpty(query.getBankName())) {
            list.add(qdo.bankName.eq(query.getBankName()));
        }
        /** 扩展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 扩展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 扩展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 扩展字4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 扩展字5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        /** 流程实例ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 流程审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
            list.add(qdo.procInstStatus.eq(query.getProcInstStatus()));
        }
        /** 提交时间 精确 */
        if (!ObjectUtils.isEmpty(query.getSubmitTime())) {
            list.add(qdo.submitTime.eq(query.getSubmitTime()));
        }
        /** 审批时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovedTime())) {
            list.add(qdo.approvedTime.eq(query.getApprovedTime()));
        }
        /**
         * 申请日期起止
         */
        if (!ObjectUtils.isEmpty(query.getApplicantTimeStart()) && !ObjectUtils.isEmpty(query.getApplicantTimeEnd())) {
            list.add(qdo.applicantTime.goe(query.getApplicantTimeStart()).and(qdo.applicantTime.loe(query.getApplicantTimeEnd())));
        }
        /**
         * 出让日期起止
         */
        if (!ObjectUtils.isEmpty(query.getTransferDateStart()) && !ObjectUtils.isEmpty(query.getTransferDateEnd())) {
            list.add(qdo.transferDate.goe(query.getTransferDateStart()).and(qdo.transferDate.loe(query.getTransferDateEnd())));
        }
        /** 逾期标志 预计核销日期小于当前日期 且状态不是已核销 */
        if (!ObjectUtils.isEmpty(query.getSlippageFlag())) {
            if ("YES".equals(query.getSlippageFlag())) {
                list.add(qdo.preWriteOffDate.lt(LocalDate.now()).and(qdo.apprStatus.ne(BorrowStatusEnum.WRITTEN_OFF.getCode())));
            }else if ("NO".equals(query.getSlippageFlag())) {
                list.add((qdo.preWriteOffDate.goe(LocalDate.now()).or(qdo.apprStatus.eq(BorrowStatusEnum.WRITTEN_OFF.getCode()))));
            }
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BorrowMoneyVO queryByKey(Long id) {
        JPAQuery<BorrowMoneyVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BorrowMoneyVO> queryListDynamic(BorrowMoneyQuery query) {
        JPAQuery<BorrowMoneyVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BorrowMoneyVO> queryPaging(BorrowMoneyQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<BorrowMoneyVO> jpaQuery = getJpaQueryWhere(query);
        List<BorrowMoneyVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BorrowMoneyVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BorrowMoneyDO save(BorrowMoneyDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BorrowMoneyDO> saveAll(List<BorrowMoneyDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BorrowMoneyPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 借款编号
        if (payload.getBorrowNo() != null) {
            update.set(qdo.borrowNo, payload.getBorrowNo());
        }
        // 借款名称
        if (payload.getBorrowName() != null) {
            update.set(qdo.borrowName, payload.getBorrowName());
        }
        // 申请人
        if (payload.getApplicantUserId() != null) {
            update.set(qdo.applicantUserId, payload.getApplicantUserId());
        }
        // 出让日期
        if (payload.getTransferDate() != null) {
            update.set(qdo.transferDate, payload.getTransferDate());
        }
        // 受让人ID
        if (payload.getReceiveUserId() != null) {
            update.set(qdo.receiveUserId, payload.getReceiveUserId());
        }
        // 受让人所属bu
        if (payload.getReceiveBuId() != null) {
            update.set(qdo.receiveBuId, payload.getReceiveBuId());
        }
        // 申请日期
        if (payload.getApplicantTime() != null) {
            update.set(qdo.applicantTime, payload.getApplicantTime());
        }
        // 申请人所属bu
        if (payload.getApplicantBuId() != null) {
            update.set(qdo.applicantBuId, payload.getApplicantBuId());
        }
        // 费用BU
        if (payload.getCostBuId() != null) {
            update.set(qdo.costBuId, payload.getCostBuId());
        }
        // 单据状态 udc[BORROW_MONEY:STATUS]
        if (payload.getApprStatus() != null) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        }
        // 业务类型 udc[BORROW_MONEY:BUSINESS_TYPE]
        if (payload.getBusinessType() != null) {
            update.set(qdo.businessType, payload.getBusinessType());
        }
        // 是否出让
        if (payload.getTransferFlag() != null) {
            update.set(qdo.transferFlag, payload.getTransferFlag());
        }
        // 还款人ID
        if (payload.getRepaymentUserId() != null) {
            update.set(qdo.repaymentUserId, payload.getRepaymentUserId());
        }
        // 借款金额
        if (payload.getBorrowAmt() != null) {
            update.set(qdo.borrowAmt, payload.getBorrowAmt());
        }
        // 已核销金额
        if (payload.getAlreadyWriteOffAmt() != null) {
            update.set(qdo.alreadyWriteOffAmt, payload.getAlreadyWriteOffAmt());
        }
        // 未核销金额
        if (payload.getUnwriteOffAmt() != null) {
            update.set(qdo.unwriteOffAmt, payload.getUnwriteOffAmt());
        }
        // 预计核销日期
        if (payload.getPreWriteOffDate() != null) {
            update.set(qdo.preWriteOffDate, payload.getPreWriteOffDate());
        }
        // 附件
        if (payload.getFileCodes() != null) {
            update.set(qdo.fileCodes, payload.getFileCodes());
        }
        // 收款账户
        if (payload.getAccountNo() != null) {
            update.set(qdo.accountNo, payload.getAccountNo());
        }
        // 户名
        if (payload.getHolderName() != null) {
            update.set(qdo.holderName, payload.getHolderName());
        }
        // 收款银行
        if (payload.getBankName() != null) {
            update.set(qdo.bankName, payload.getBankName());
        }
        // 扩展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 扩展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 扩展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 扩展字4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 扩展字5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 提交时间
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批时间
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 借款编号
            if (nullFields.contains("borrowNo")) {
                update.setNull(qdo.borrowNo);
            }
            // 借款名称
            if (nullFields.contains("borrowName")) {
                update.setNull(qdo.borrowName);
            }
            // 申请人
            if (nullFields.contains("applicantUserId")) {
                update.setNull(qdo.applicantUserId);
            }
            // 出让日期
            if (nullFields.contains("transferDate")) {
                update.setNull(qdo.transferDate);
            }
            // 受让人ID
            if (nullFields.contains("receiveUserId")) {
                update.setNull(qdo.receiveUserId);
            }
            // 受让人所属bu
            if (nullFields.contains("receiveBuId")) {
                update.setNull(qdo.receiveBuId);
            }
            // 申请日期
            if (nullFields.contains("applicantTime")) {
                update.setNull(qdo.applicantTime);
            }
            // 申请人所属bu
            if (nullFields.contains("applicantBuId")) {
                update.setNull(qdo.applicantBuId);
            }
            // 费用BU
            if (nullFields.contains("costBuId")) {
                update.setNull(qdo.costBuId);
            }
            // 单据状态 udc[BORROW_MONEY:STATUS]
            if (nullFields.contains("apprStatus")) {
                update.setNull(qdo.apprStatus);
            }
            // 业务类型 udc[BORROW_MONEY:BUSINESS_TYPE]
            if (nullFields.contains("businessType")) {
                update.setNull(qdo.businessType);
            }
            // 是否出让
            if (nullFields.contains("transferFlag")) {
                update.setNull(qdo.transferFlag);
            }
            // 是否出让
            if (nullFields.contains("repaymentUserId")) {
                update.setNull(qdo.repaymentUserId);
            }
            // 借款金额
            if (nullFields.contains("borrowAmt")) {
                update.setNull(qdo.borrowAmt);
            }
            // 已核销金额
            if (nullFields.contains("alreadyWriteOffAmt")) {
                update.setNull(qdo.alreadyWriteOffAmt);
            }
            // 未核销金额
            if (nullFields.contains("unwriteOffAmt")) {
                update.setNull(qdo.unwriteOffAmt);
            }
            // 预计核销日期
            if (nullFields.contains("preWriteOffDate")) {
                update.setNull(qdo.preWriteOffDate);
            }
            // 附件
            if (nullFields.contains("fileCodes")) {
                update.setNull(qdo.fileCodes);
            }
            // 收款账户
            if (nullFields.contains("accountNo")) {
                update.setNull(qdo.accountNo);
            }
            // 户名
            if (nullFields.contains("holderName")) {
                update.setNull(qdo.holderName);
            }
            // 收款银行
            if (nullFields.contains("bankName")) {
                update.setNull(qdo.bankName);
            }
            // 扩展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 扩展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 扩展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 扩展字4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 扩展字5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 提交时间
            if (nullFields.contains("submitTime")) {
                update.setNull(qdo.submitTime);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 更新单据状态
     *
     * @param keys 主集合
     * @return 更新的行数
     */
    public long updateBorrowStatus(List<Long> keys,String status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.apprStatus, status)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

