package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalAccountTurnoverPayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalAccountTurnoverQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalAccountTurnoverVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalAccountTurnoverDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalAccountTurnoverDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalAccountTurnoverRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 核算-账户流水记录管理
 *
 * @author carl
 * @date 2023-11-13
 */
@Repository
@RequiredArgsConstructor
public class CalAccountTurnoverDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalAccountTurnoverRepo repo;
    private final QCalAccountTurnoverDO qdo = QCalAccountTurnoverDO.calAccountTurnoverDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalAccountTurnoverVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalAccountTurnoverVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 账户id
                qdo.accId,
                // 账户名称（冗余）
                qdo.accName,
                // 核算年度
                qdo.finYear,
                // 核算期间
                qdo.finPeriod,
                // 交易期间
                qdo.ioDate,
                // 交易时间
                qdo.ioTime,
                // 交易单据类型
                qdo.sourceType,
                // 交易单据id
                qdo.sourceId,
                // 交易单据名称（冗余）
                qdo.sourceName,
                // 交易当量
                qdo.turnoverQty,
                // 当量价格
                qdo.eqvaPrice,
                // 交易前当量
                qdo.beforeQty,
                // 交易后当量
                qdo.afterQty,
                // 交易前金额
                qdo.beforeAmt,
                // 交易后金额
                qdo.afterAmt
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalAccountTurnoverVO> getJpaQueryWhere(CalAccountTurnoverQuery query) {
        JPAQuery<CalAccountTurnoverVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalAccountTurnoverQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalAccountTurnoverQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 账户id 精确 */
        if (!ObjectUtils.isEmpty(query.getAccId())) {
            list.add(qdo.accId.eq(query.getAccId()));
        }
        /** 账户名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getAccName())) {
            list.add(qdo.accName.like(SqlUtil.toSqlLikeString(query.getAccName())));
        }
        /** 核算年度 精确 */
        if (!ObjectUtils.isEmpty(query.getFinYear())) {
            list.add(qdo.finYear.eq(query.getFinYear()));
        }
        /** 核算期间 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
            list.add(qdo.finPeriod.eq(query.getFinPeriod()));
        }
        /** 交易期间 精确 */
        if (!ObjectUtils.isEmpty(query.getIoDate())) {
            list.add(qdo.ioDate.eq(query.getIoDate()));
        }
        /** 交易期间 精确 */
        if (!ObjectUtils.isEmpty(query.getIoStartDate())) {
            list.add(qdo.ioDate.goe(query.getIoStartDate()));
        }
        /** 交易期间 精确 */
        if (!ObjectUtils.isEmpty(query.getIoEndDate())) {
            list.add(qdo.ioDate.loe(query.getIoEndDate()));
        }
        /** 交易时间 精确 */
        if (!ObjectUtils.isEmpty(query.getIoTime())) {
            list.add(qdo.ioTime.eq(query.getIoTime()));
        }
        /** 交易单据类型 精确 */
        if (!ObjectUtils.isEmpty(query.getSourceType())) {
            list.add(qdo.sourceType.eq(query.getSourceType()));
        }
        /** 交易单据id 精确 */
        if (!ObjectUtils.isEmpty(query.getSourceId())) {
            list.add(qdo.sourceId.eq(query.getSourceId()));
        }
        /** 交易单据名称（冗余） 模糊 */
        if (!ObjectUtils.isEmpty(query.getSourceName())) {
            list.add(qdo.sourceName.like(SqlUtil.toSqlLikeString(query.getSourceName())));
        }
        /** 交易当量 精确 */
        if (!ObjectUtils.isEmpty(query.getTurnoverQty())) {
            list.add(qdo.turnoverQty.eq(query.getTurnoverQty()));
        }
        /** 当量价格 精确 */
        if (!ObjectUtils.isEmpty(query.getEqvaPrice())) {
            list.add(qdo.eqvaPrice.eq(query.getEqvaPrice()));
        }
        /** 交易前当量 精确 */
        if (!ObjectUtils.isEmpty(query.getBeforeQty())) {
            list.add(qdo.beforeQty.eq(query.getBeforeQty()));
        }
        /** 交易后当量 精确 */
        if (!ObjectUtils.isEmpty(query.getAfterQty())) {
            list.add(qdo.afterQty.eq(query.getAfterQty()));
        }
        /** 交易前金额 精确 */
        if (!ObjectUtils.isEmpty(query.getBeforeAmt())) {
            list.add(qdo.beforeAmt.eq(query.getBeforeAmt()));
        }
        /** 交易后金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAfterAmt())) {
            list.add(qdo.afterAmt.eq(query.getAfterAmt()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalAccountTurnoverVO queryByKey(Long id) {
        JPAQuery<CalAccountTurnoverVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalAccountTurnoverVO> queryListDynamic(CalAccountTurnoverQuery query) {
        JPAQuery<CalAccountTurnoverVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalAccountTurnoverVO> queryPaging(CalAccountTurnoverQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CalAccountTurnoverVO> jpaQuery = getJpaQueryWhere(query);
        List<CalAccountTurnoverVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalAccountTurnoverVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalAccountTurnoverDO save(CalAccountTurnoverDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalAccountTurnoverDO> saveAll(List<CalAccountTurnoverDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalAccountTurnoverPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 账户id
        if (payload.getAccId() != null) {
            update.set(qdo.accId, payload.getAccId());
        }
        // 账户名称（冗余）
        if (payload.getAccName() != null) {
            update.set(qdo.accName, payload.getAccName());
        }
        // 核算年度
        if (payload.getFinYear() != null) {
            update.set(qdo.finYear, payload.getFinYear());
        }
        // 核算期间
        if (payload.getFinPeriod() != null) {
            update.set(qdo.finPeriod, payload.getFinPeriod());
        }
        // 交易期间
        if (payload.getIoDate() != null) {
            update.set(qdo.ioDate, payload.getIoDate());
        }
        // 交易时间
        if (payload.getIoTime() != null) {
            update.set(qdo.ioTime, payload.getIoTime());
        }
        // 交易单据类型
        if (payload.getSourceType() != null) {
            update.set(qdo.sourceType, payload.getSourceType());
        }
        // 交易单据id
        if (payload.getSourceId() != null) {
            update.set(qdo.sourceId, payload.getSourceId());
        }
        // 交易单据名称（冗余）
        if (payload.getSourceName() != null) {
            update.set(qdo.sourceName, payload.getSourceName());
        }
        // 交易当量
        if (payload.getTurnoverQty() != null) {
            update.set(qdo.turnoverQty, payload.getTurnoverQty());
        }
        // 当量价格
        if (payload.getEqvaPrice() != null) {
            update.set(qdo.eqvaPrice, payload.getEqvaPrice());
        }
        // 交易前当量
        if (payload.getBeforeQty() != null) {
            update.set(qdo.beforeQty, payload.getBeforeQty());
        }
        // 交易后当量
        if (payload.getAfterQty() != null) {
            update.set(qdo.afterQty, payload.getAfterQty());
        }
        // 交易前金额
        if (payload.getBeforeAmt() != null) {
            update.set(qdo.beforeAmt, payload.getBeforeAmt());
        }
        // 交易后金额
        if (payload.getAfterAmt() != null) {
            update.set(qdo.afterAmt, payload.getAfterAmt());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 账户id
            if (nullFields.contains("accId")) {
                update.setNull(qdo.accId);
            }
            // 账户名称（冗余）
            if (nullFields.contains("accName")) {
                update.setNull(qdo.accName);
            }
            // 核算年度
            if (nullFields.contains("finYear")) {
                update.setNull(qdo.finYear);
            }
            // 核算期间
            if (nullFields.contains("finPeriod")) {
                update.setNull(qdo.finPeriod);
            }
            // 交易期间
            if (nullFields.contains("ioDate")) {
                update.setNull(qdo.ioDate);
            }
            // 交易时间
            if (nullFields.contains("ioTime")) {
                update.setNull(qdo.ioTime);
            }
            // 交易单据类型
            if (nullFields.contains("sourceType")) {
                update.setNull(qdo.sourceType);
            }
            // 交易单据id
            if (nullFields.contains("sourceId")) {
                update.setNull(qdo.sourceId);
            }
            // 交易单据名称（冗余）
            if (nullFields.contains("sourceName")) {
                update.setNull(qdo.sourceName);
            }
            // 交易当量
            if (nullFields.contains("turnoverQty")) {
                update.setNull(qdo.turnoverQty);
            }
            // 当量价格
            if (nullFields.contains("eqvaPrice")) {
                update.setNull(qdo.eqvaPrice);
            }
            // 交易前当量
            if (nullFields.contains("beforeQty")) {
                update.setNull(qdo.beforeQty);
            }
            // 交易后当量
            if (nullFields.contains("afterQty")) {
                update.setNull(qdo.afterQty);
            }
            // 交易前金额
            if (nullFields.contains("beforeAmt")) {
                update.setNull(qdo.beforeAmt);
            }
            // 交易后金额
            if (nullFields.contains("afterAmt")) {
                update.setNull(qdo.afterAmt);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

