package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectBriefPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectBriefQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectBriefVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectBriefDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectBriefDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectBriefRepo;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * 项目汇报管理
 *
 * @author carl
 * @date 2023-09-05
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectBriefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectBriefRepo repo;
    private final QPmsProjectBriefDO qdo = QPmsProjectBriefDO.pmsProjectBriefDO;
    private final QPmsProjectDO qdoProject = QPmsProjectDO.pmsProjectDO;
    private final QSaleConContractDO qdoSaleCon = QSaleConContractDO.saleConContractDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectBriefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectBriefVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 汇报编号
                qdo.briefNo,
                // 汇报状态
                qdo.briefStatus,
                //流程实例id
                qdo.procInstId,
                // 审批状态
                qdo.apprStatus,
                // 财务期间
                qdo.finPeriodDate,
                // 提交日期
                qdo.applyDate,
                // 提交资源userid
                qdo.applyUserId,
                // 项目id
                qdo.projId,
                // 项目名称
                qdo.projName,
                // 费用预算总金额
                qdo.feeBudgetAmt,
                // 费用预算已拨付金额
                qdo.feeReleasedAmt,
                // 费用预算已使用金额
                qdo.feeUsedAmt,
                // 当量预算总数
                qdo.eqvaBudgetCnt,
                // 当量预算总金额
                qdo.eqvaBudgetAmt,
                // 当量已拨付数量
                qdo.eqvaReleasedQty,
                // 当量已拨付金额
                qdo.eqvaReleasedAmt,
                // 当量已派发数量
                qdo.eqvaDistedQty,
                // 当量已派发金额
                qdo.eqvaDistedAmt,
                // 当量已结算数量
                qdo.eqvaSettledQty,
                // 当量已结算金额
                qdo.eqvaSettledAmt,
                // 当量派发百分比
                qdo.eqvaDistedPercnet,
                // 当量结算百分比
                qdo.eqvaSettledPercent,
                // 报告完工百分比
                qdo.reprotCompPercent,
                // 调整完工百分比
                qdo.confirmCompPercent,
                // 财务调整说明
                qdo.confirmCompDesc,
                // 项目金额
                qdo.projAmt,
                // 税率
                qdo.taxRate,
                // 已收款金额
                qdo.recvedAmt,
                // 已开票金额
                qdo.invoicedAmt,
                // 确认收入金额
                qdo.confirmedAmt,
                // 当期确认收入
                qdo.confirmAmt,
                // 实际确认收入百分比
                qdo.confirmedPercent,
                // 申请拨付费用金额
                qdo.applyFeeAmt,
                // 申请拨付当量数
                qdo.applyEqvaCnt,
                // 申请拨付当量金额
                qdo.applyEqvaAmt,
                // 实际拨付费用金额
                qdo.approveFeeAmt,
                // 实际拨付当量数
                qdo.approveEqvaCnt,
                // 实际拨付当量金额
                qdo.approveEqvaAmt,
                // 币种
                qdo.currCode,
                // 进度汇报说明
                qdo.briefDesc,
                // 费用申请说明
                qdo.feeApplyDesc,
                // 当量申请说明
                qdo.eqvaApplyDesc,
                // 项目进度状态
                qdo.projProcessStatus,
                // 风险描述
                qdo.riskDesc,
                // 预计还会发生的报销费用
                qdo.predictReimAmt,
                // 预计还会发生的劳务费用
                qdo.predictLaborAmt,
                // 已发生的劳务费用
                qdo.passedLaborAmt
        )).from(qdo);
    }

    /**
     * 根据项目主键查询上次提价数据
     *
     * @param projectId 主键
     * @return 结果
     */
    public PmsProjectBriefVO queryByProjectId(Long projectId, String status) {
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        if (StringUtils.hasText(status)) {
            jpaQuery.where((qdo.briefStatus.eq("FINISHED")));
        }
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.desc("finPeriodDate"));
        // 根据期间降序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));
        return jpaQuery.fetchFirst();
    }
    /**
     * 根据项目主键查询上次提价数据
     *
     * @param projectId 主键
     * @return 结果
     */
    public PmsProjectBriefVO queryByProjectIdAndDate(Long projectId, LocalDate finPeriodDate) {
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        if (finPeriodDate!=null) {
            jpaQuery.where((qdo.finPeriodDate.eq(finPeriodDate)));
        }
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.desc("finPeriodDate"));
        // 根据期间降序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectBriefVO queryByKey(Long id) {
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<PmsProjectBriefVO> queryByKeys(List<Long> ids) {
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }


    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectBriefVO> getJpaQuerySelect0() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectBriefVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                // 汇报编号
                qdo.briefNo,
                // 汇报状态
                qdo.briefStatus,
                // 财务期间
                qdo.finPeriodDate,
                // 项目id
                qdo.projId,
                //项目名称
                qdoProject.projName,
                qdoProject.projNo,
                //参考合同号
                qdoSaleCon.referCode,
                //合同状态
                qdoSaleCon.status.as("contractStatus"),
                //签单BU_ID
                qdoSaleCon.signBuId,
                // 工作类型
                qdoSaleCon.workType,
                //交付BU_ID
                qdoProject.deliBuId,
                // 报告完工百分比
                qdo.reprotCompPercent,
                //汇报日期
                qdo.applyDate,
                // 项目金额
                qdo.projAmt,
                // 累计确认收入金额
                qdo.confirmedAmt,
                // 当期确认收入
                qdo.confirmAmt,
                // 项目进度状态
                qdo.projProcessStatus

        )).from(qdo).leftJoin(qdoProject).on(qdo.projId.eq(qdoProject.id)).leftJoin(qdoSaleCon).on(qdoSaleCon.id.eq(qdoProject.contractId));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectBriefVO> getJpaQueryWhere(PmsProjectBriefQuery query) {
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQuerySelect0();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        //分组
        jpaQuery.groupBy(qdo.id);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectBriefQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qdoProject).on(qdo.projId.eq(qdoProject.id)).leftJoin(qdoSaleCon).on(qdoSaleCon.id.eq(qdoProject.contractId));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectBriefQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** 汇报编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getBriefNo())) {
            list.add(qdo.briefNo.like(SqlUtil.toSqlLikeString(query.getBriefNo())));
        }
        /** 汇报状态 精确 */
        if (!ObjectUtils.isEmpty(query.getBriefStatus())) {
            list.add(qdo.briefStatus.eq(query.getBriefStatus()));
        }
        /** 汇报日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyDate())) {
            list.add(qdo.applyDate.eq(query.getApplyDate()));
        }
        /** 财务期间 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriodDate())) {
            list.add(qdo.finPeriodDate.eq(query.getFinPeriodDate()));
        }
        /** 创建人userid 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }
        /** 项目 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjSearch())) {
            list.add(qdoProject.projName.like(SqlUtil.toSqlLikeString(query.getProjSearch())).or(qdoProject.productNo.like(SqlUtil.toSqlLikeString(query.getProjSearch()))));
        }
        /** 合同状态 精确 */
        if (!ObjectUtils.isEmpty(query.getContractStatus())) {
            list.add(qdoSaleCon.status.eq(query.getContractStatus()));
        }
        /** 交付BU_ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliBuId())) {
            list.add(qdoProject.deliBuId.eq(query.getDeliBuId()));
        }
        /** 签单BU_ID 精确 */
        if (!ObjectUtils.isEmpty(query.getSignBuId())) {
            list.add(qdoProject.signBuId.eq(query.getSignBuId()));
        }
        /** 项目进度状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProjProcessStatus())) {
            list.add(qdo.projProcessStatus.eq(query.getProjProcessStatus()));
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectBriefVO> queryListDynamic(PmsProjectBriefQuery query) {
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectBriefVO> queryPaging(PmsProjectBriefQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectBriefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectBriefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectBriefDO save(PmsProjectBriefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectBriefDO> saveAll(List<PmsProjectBriefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectBriefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        // 汇报编号
        if (payload.getBriefNo() != null) {
            update.set(qdo.briefNo, payload.getBriefNo());
        }
        // 汇报状态
        if (payload.getBriefStatus() != null) {
            update.set(qdo.briefStatus, payload.getBriefStatus());
        }
        // 审批状态
        if (payload.getApprStatus() != null) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        }
        // 财务期间
        if (payload.getFinPeriodDate() != null) {
            update.set(qdo.finPeriodDate, payload.getFinPeriodDate());
        }
        // 提交日期
        if (payload.getApplyDate() != null) {
            update.set(qdo.applyDate, payload.getApplyDate());
        }
        // 提交资源userid
        if (payload.getApplyUserId() != null) {
            update.set(qdo.applyUserId, payload.getApplyUserId());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 费用预算总金额
        if (payload.getFeeBudgetAmt() != null) {
            update.set(qdo.feeBudgetAmt, payload.getFeeBudgetAmt());
        }
        // 费用预算已拨付金额
        if (payload.getFeeReleasedAmt() != null) {
            update.set(qdo.feeReleasedAmt, payload.getFeeReleasedAmt());
        }
        // 费用预算已使用金额
        if (payload.getFeeUsedAmt() != null) {
            update.set(qdo.feeUsedAmt, payload.getFeeUsedAmt());
        }
        // 当量预算总数
        if (payload.getEqvaBudgetCnt() != null) {
            update.set(qdo.eqvaBudgetCnt, payload.getEqvaBudgetCnt());
        }
        // 当量预算总金额
        if (payload.getEqvaBudgetAmt() != null) {
            update.set(qdo.eqvaBudgetAmt, payload.getEqvaBudgetAmt());
        }
        // 当量已拨付数量
        if (payload.getEqvaReleasedQty() != null) {
            update.set(qdo.eqvaReleasedQty, payload.getEqvaReleasedQty());
        }
        // 当量已拨付金额
        if (payload.getEqvaReleasedAmt() != null) {
            update.set(qdo.eqvaReleasedAmt, payload.getEqvaReleasedAmt());
        }
        // 当量已派发数量
        if (payload.getEqvaDistedQty() != null) {
            update.set(qdo.eqvaDistedQty, payload.getEqvaDistedQty());
        }
        // 当量已派发金额
        if (payload.getEqvaDistedAmt() != null) {
            update.set(qdo.eqvaDistedAmt, payload.getEqvaDistedAmt());
        }
        // 当量已结算数量
        if (payload.getEqvaSettledQty() != null) {
            update.set(qdo.eqvaSettledQty, payload.getEqvaSettledQty());
        }
        // 当量已结算金额
        if (payload.getEqvaSettledAmt() != null) {
            update.set(qdo.eqvaSettledAmt, payload.getEqvaSettledAmt());
        }
        // 当量派发百分比
        if (payload.getEqvaDistedPercnet() != null) {
            update.set(qdo.eqvaDistedPercnet, payload.getEqvaDistedPercnet());
        }
        // 当量结算百分比
        if (payload.getEqvaSettledPercent() != null) {
            update.set(qdo.eqvaSettledPercent, payload.getEqvaSettledPercent());
        }
        // 报告完工百分比
        if (payload.getReprotCompPercent() != null) {
            update.set(qdo.reprotCompPercent, payload.getReprotCompPercent());
        }
        // 调整完工百分比
        if (payload.getConfirmCompPercent() != null) {
            update.set(qdo.confirmCompPercent, payload.getConfirmCompPercent());
        }
        // 财务调整说明
        if (payload.getConfirmCompDesc() != null) {
            update.set(qdo.confirmCompDesc, payload.getConfirmCompDesc());
        }
        // 项目金额
        if (payload.getProjAmt() != null) {
            update.set(qdo.projAmt, payload.getProjAmt());
        }
        // 已收款金额
        if (payload.getRecvedAmt() != null) {
            update.set(qdo.recvedAmt, payload.getRecvedAmt());
        }
        // 已开票金额
        if (payload.getInvoicedAmt() != null) {
            update.set(qdo.invoicedAmt, payload.getInvoicedAmt());
        }
        // 确认收入金额
        if (payload.getConfirmedAmt() != null) {
            update.set(qdo.confirmedAmt, payload.getConfirmedAmt());
        }
        // 当期确认收入
        if (payload.getConfirmAmt() != null) {
            update.set(qdo.confirmAmt, payload.getConfirmAmt());
        }
        // 实际确认收入百分比
        if (payload.getConfirmedPercent() != null) {
            update.set(qdo.confirmedPercent, payload.getConfirmedPercent());
        }
        // 申请拨付费用金额
        if (payload.getApplyFeeAmt() != null) {
            update.set(qdo.applyFeeAmt, payload.getApplyFeeAmt());
        }
        // 申请拨付当量数
        if (payload.getApplyEqvaCnt() != null) {
            update.set(qdo.applyEqvaCnt, payload.getApplyEqvaCnt());
        }
        // 申请拨付当量金额
        if (payload.getApplyEqvaAmt() != null) {
            update.set(qdo.applyEqvaAmt, payload.getApplyEqvaAmt());
        }
        // 实际拨付费用金额
        if (payload.getApproveFeeAmt() != null) {
            update.set(qdo.approveFeeAmt, payload.getApproveFeeAmt());
        }
        // 实际拨付当量数
        if (payload.getApproveEqvaCnt() != null) {
            update.set(qdo.approveEqvaCnt, payload.getApproveEqvaCnt());
        }
        // 实际拨付当量金额
        if (payload.getApproveEqvaAmt() != null) {
            update.set(qdo.approveEqvaAmt, payload.getApproveEqvaAmt());
        }
        // 币种
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 进度汇报说明
        if (payload.getBriefDesc() != null) {
            update.set(qdo.briefDesc, payload.getBriefDesc());
        }
        // 费用申请说明
        if (payload.getFeeApplyDesc() != null) {
            update.set(qdo.feeApplyDesc, payload.getFeeApplyDesc());
        }
        // 当量申请说明
        if (payload.getEqvaApplyDesc() != null) {
            update.set(qdo.eqvaApplyDesc, payload.getEqvaApplyDesc());
        }
        // 项目进度状态
        if (payload.getProjProcessStatus() != null) {
            update.set(qdo.projProcessStatus, payload.getProjProcessStatus());
        }
        // 风险描述
        if (payload.getRiskDesc() != null) {
            update.set(qdo.riskDesc, payload.getRiskDesc());
        }
        // 预计还会发生的报销费用
        if (payload.getPredictReimAmt() != null) {
            update.set(qdo.predictReimAmt, payload.getPredictReimAmt());
        }
        // 预计还会发生的劳务费用
        if (payload.getPredictLaborAmt() != null) {
            update.set(qdo.predictLaborAmt, payload.getPredictLaborAmt());
        }
        // 已发生的劳务费用
        if (payload.getPassedLaborAmt() != null) {
            update.set(qdo.passedLaborAmt, payload.getPassedLaborAmt());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 汇报编号
            if (nullFields.contains("briefNo")) {
                update.setNull(qdo.briefNo);
            }
            // 汇报状态
            if (nullFields.contains("briefStatus")) {
                update.setNull(qdo.briefStatus);
            }
            // 审批状态
            if (nullFields.contains("apprStatus")) {
                update.setNull(qdo.apprStatus);
            }
            // 财务期间
            if (nullFields.contains("finPeriodDate")) {
                update.setNull(qdo.finPeriodDate);
            }
            // 提交日期
            if (nullFields.contains("applyDate")) {
                update.setNull(qdo.applyDate);
            }
            // 提交资源userid
            if (nullFields.contains("applyUserId")) {
                update.setNull(qdo.applyUserId);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 费用预算总金额
            if (nullFields.contains("feeBudgetAmt")) {
                update.setNull(qdo.feeBudgetAmt);
            }
            // 费用预算已拨付金额
            if (nullFields.contains("feeReleasedAmt")) {
                update.setNull(qdo.feeReleasedAmt);
            }
            // 费用预算已使用金额
            if (nullFields.contains("feeUsedAmt")) {
                update.setNull(qdo.feeUsedAmt);
            }
            // 当量预算总数
            if (nullFields.contains("eqvaBudgetCnt")) {
                update.setNull(qdo.eqvaBudgetCnt);
            }
            // 当量预算总金额
            if (nullFields.contains("eqvaBudgetAmt")) {
                update.setNull(qdo.eqvaBudgetAmt);
            }
            // 当量已拨付数量
            if (nullFields.contains("eqvaReleasedQty")) {
                update.setNull(qdo.eqvaReleasedQty);
            }
            // 当量已拨付金额
            if (nullFields.contains("eqvaReleasedAmt")) {
                update.setNull(qdo.eqvaReleasedAmt);
            }
            // 当量已派发数量
            if (nullFields.contains("eqvaDistedQty")) {
                update.setNull(qdo.eqvaDistedQty);
            }
            // 当量已派发金额
            if (nullFields.contains("eqvaDistedAmt")) {
                update.setNull(qdo.eqvaDistedAmt);
            }
            // 当量已结算数量
            if (nullFields.contains("eqvaSettledQty")) {
                update.setNull(qdo.eqvaSettledQty);
            }
            // 当量已结算金额
            if (nullFields.contains("eqvaSettledAmt")) {
                update.setNull(qdo.eqvaSettledAmt);
            }
            // 当量派发百分比
            if (nullFields.contains("eqvaDistedPercnet")) {
                update.setNull(qdo.eqvaDistedPercnet);
            }
            // 当量结算百分比
            if (nullFields.contains("eqvaSettledPercent")) {
                update.setNull(qdo.eqvaSettledPercent);
            }
            // 报告完工百分比
            if (nullFields.contains("reprotCompPercent")) {
                update.setNull(qdo.reprotCompPercent);
            }
            // 调整完工百分比
            if (nullFields.contains("confirmCompPercent")) {
                update.setNull(qdo.confirmCompPercent);
            }
            // 财务调整说明
            if (nullFields.contains("confirmCompDesc")) {
                update.setNull(qdo.confirmCompDesc);
            }
            // 项目金额
            if (nullFields.contains("projAmt")) {
                update.setNull(qdo.projAmt);
            }
            // 已收款金额
            if (nullFields.contains("recvedAmt")) {
                update.setNull(qdo.recvedAmt);
            }
            // 已开票金额
            if (nullFields.contains("invoicedAmt")) {
                update.setNull(qdo.invoicedAmt);
            }
            // 确认收入金额
            if (nullFields.contains("confirmedAmt")) {
                update.setNull(qdo.confirmedAmt);
            }
            // 当期确认收入
            if (nullFields.contains("confirmAmt")) {
                update.setNull(qdo.confirmAmt);
            }
            // 实际确认收入百分比
            if (nullFields.contains("confirmedPercent")) {
                update.setNull(qdo.confirmedPercent);
            }
            // 申请拨付费用金额
            if (nullFields.contains("applyFeeAmt")) {
                update.setNull(qdo.applyFeeAmt);
            }
            // 申请拨付当量数
            if (nullFields.contains("applyEqvaCnt")) {
                update.setNull(qdo.applyEqvaCnt);
            }
            // 申请拨付当量金额
            if (nullFields.contains("applyEqvaAmt")) {
                update.setNull(qdo.applyEqvaAmt);
            }
            // 实际拨付费用金额
            if (nullFields.contains("approveFeeAmt")) {
                update.setNull(qdo.approveFeeAmt);
            }
            // 实际拨付当量数
            if (nullFields.contains("approveEqvaCnt")) {
                update.setNull(qdo.approveEqvaCnt);
            }
            // 实际拨付当量金额
            if (nullFields.contains("approveEqvaAmt")) {
                update.setNull(qdo.approveEqvaAmt);
            }
            // 币种
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 进度汇报说明
            if (nullFields.contains("briefDesc")) {
                update.setNull(qdo.briefDesc);
            }
            // 费用申请说明
            if (nullFields.contains("feeApplyDesc")) {
                update.setNull(qdo.feeApplyDesc);
            }
            // 当量申请说明
            if (nullFields.contains("eqvaApplyDesc")) {
                update.setNull(qdo.eqvaApplyDesc);
            }
            // 项目进度状态
            if (nullFields.contains("projProcessStatus")) {
                update.setNull(qdo.projProcessStatus);
            }
            // 风险描述
            if (nullFields.contains("riskDesc")) {
                update.setNull(qdo.riskDesc);
            }
            // 预计还会发生的报销费用
            if (nullFields.contains("predictReimAmt")) {
                update.setNull(qdo.predictReimAmt);
            }
            // 预计还会发生的劳务费用
            if (nullFields.contains("predictLaborAmt")) {
                update.setNull(qdo.predictLaborAmt);
            }
            // 已发生的劳务费用
            if (nullFields.contains("passedLaborAmt")) {
                update.setNull(qdo.passedLaborAmt);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 变更工作流相关数据
     *
     * @param payload
     * @return
     */

    public long updateWorkFlow(PmsProjectBriefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getProcInstId())) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(payload.getApprStatus())) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getBriefStatus())) {
            update.set(qdo.briefStatus, payload.getBriefStatus());
        }

        if (!ObjectUtils.isEmpty(payload.getDeleteFlag())) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null) {
            // 流程实例id
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
        }
        // 执行修改
        return update.execute();
    }


    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectBriefVO> getJpaQuerySelect1() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectBriefVO.class,
                qdo.id,
                // 财务期间
                qdo.finPeriodDate,
                //进度状态
                qdo.projProcessStatus,
                //确认收入金额
                qdo.confirmedAmt,
                // 项目id
                qdo.projId,
                // 报告完工百分比
                qdo.reprotCompPercent
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @return jpaQuery对象
     */
    public List<PmsProjectBriefVO> projectBriefWork(List<Long> projectIds) {
        JPAQuery<PmsProjectBriefVO> jpaQuery = getJpaQuerySelect1();
        jpaQuery.where(qdo.projId.in(projectIds));
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.desc("createTime"));
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));
        jpaQuery.groupBy(qdo.projId);
        return jpaQuery.fetch();
    }
}

