package com.elitesland.tw.tw5.server.prd.product.dao;

import com.elitesland.tw.tw5.api.prd.product.payload.PrdProductClassPayload;
import com.elitesland.tw.tw5.api.prd.product.query.PrdProductClassQuery;
import com.elitesland.tw.tw5.api.prd.product.vo.PrdProductClassVO;
import com.elitesland.tw.tw5.server.prd.product.entity.PrdProductClassDO;
import com.elitesland.tw.tw5.server.prd.product.entity.QPrdProductClassDO;
import com.elitesland.tw.tw5.server.prd.product.repo.PrdProductClassRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 产品分类
 *
 * @author likunpeng
 * @date 2023-05-22
 */
@Repository
@RequiredArgsConstructor
public class PrdProductClassDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PrdProductClassRepo repo;
    private final QPrdProductClassDO qdo = QPrdProductClassDO.prdProductClassDO;
    private final QPrdProductClassDO seniorQdo = new QPrdProductClassDO("seniorQdo");

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdProductClassVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdProductClassVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 分类编号
                qdo.classCode,
                // 分类名称
                qdo.className,
                // 分类类型 0小类/1大类
                qdo.classType,
                // 上级ID
                qdo.seniorId,
                // 上级分类名称
                seniorQdo.className.as("seniorName")
        )).from(qdo)
                .leftJoin(seniorQdo).on(qdo.seniorId.longValue().eq(seniorQdo.id.longValue()));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdProductClassVO> getJpaQueryWhere(PrdProductClassQuery query) {
        JPAQuery<PrdProductClassVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PrdProductClassQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PrdProductClassQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 分类编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getClassCode())) {
             list.add(qdo.classCode.like(SqlUtil.toSqlLikeString(query.getClassCode())));
        }
        /** 分类名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getClassName())) {
             list.add(qdo.className.like(SqlUtil.toSqlLikeString(query.getClassName())));
        }
        /** 分类类型 0小类/1大类 精确 */
        if (!ObjectUtils.isEmpty(query.getClassType())) {
            list.add(qdo.classType.eq(query.getClassType()));
        }
        /** 上级ID 精确 */
        if (!ObjectUtils.isEmpty(query.getSeniorId())) {
            list.add(qdo.seniorId.eq(query.getSeniorId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdProductClassVO queryByKey(Long id) {
        JPAQuery<PrdProductClassVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdProductClassVO> queryListDynamic(PrdProductClassQuery query) {
        JPAQuery<PrdProductClassVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdProductClassVO> queryPaging(PrdProductClassQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PrdProductClassVO> jpaQuery = getJpaQueryWhere(query);
        List<PrdProductClassVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PrdProductClassVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdProductClassDO save(PrdProductClassDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdProductClassDO> saveAll(List<PrdProductClassDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PrdProductClassPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 分类编号
        if (payload.getClassCode() != null) {
            update.set(qdo.classCode, payload.getClassCode());
        }
        // 分类名称
        if (payload.getClassName() != null) {
            update.set(qdo.className, payload.getClassName());
        }
        // 分类类型 0小类/1大类
        if (payload.getClassType() != null) {
            update.set(qdo.classType, payload.getClassType());
        }
        // 上级ID
        if (payload.getSeniorId() != null) {
            update.set(qdo.seniorId, payload.getSeniorId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 分类编号
            if (nullFields.contains("classCode")) {
                update.setNull(qdo.classCode);
            }
            // 分类名称
            if (nullFields.contains("className")) {
                update.setNull(qdo.className);
            }
            // 分类类型 0小类/1大类
            if (nullFields.contains("classType")) {
                update.setNull(qdo.classType);
            }
            // 上级ID
            if (nullFields.contains("seniorId")) {
                update.setNull(qdo.seniorId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

