package com.elitesland.tw.tw5.server.prd.shorturl.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.shorturl.payload.ShortUrlConfPayload;
import com.elitesland.tw.tw5.api.prd.shorturl.query.ShortUrlConfQuery;
import com.elitesland.tw.tw5.api.prd.shorturl.vo.ShortUrlConfVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.shorturl.entity.QShortUrlConfDO;
import com.elitesland.tw.tw5.server.prd.shorturl.entity.ShortUrlConfDO;
import com.elitesland.tw.tw5.server.prd.shorturl.repo.ShortUrlConfRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 短链服务配置
 *
 * @author duwh
 * @date 2023-05-19
 */
@Repository
@RequiredArgsConstructor
public class ShortUrlConfDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ShortUrlConfRepo repo;
    private final QShortUrlConfDO qdo = QShortUrlConfDO.shortUrlConfDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ShortUrlConfVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ShortUrlConfVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 短链地址前缀
                qdo.shortUrlPre,
                qdo.shortUrlRelative,
                // 长链地址后缀
                qdo.longUrlPre,
                qdo.secTenantId,
                // 拓展1
                qdo.ext1,
                // 拓展2
                qdo.ext2,
                // 拓展3
                qdo.ext3,
                // 拓展4
                qdo.ext4,
                // 拓展5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ShortUrlConfVO> getJpaQueryWhere(ShortUrlConfQuery query) {
        JPAQuery<ShortUrlConfVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ShortUrlConfQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ShortUrlConfQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 短链地址前缀 精确 */
        if (!ObjectUtils.isEmpty(query.getShortUrlPre())) {
            list.add(qdo.shortUrlPre.eq(query.getShortUrlPre()));
        }
        /** 短链地址前缀 精确 */
        if (!ObjectUtils.isEmpty(query.getShortUrlRelative())) {
            list.add(qdo.shortUrlRelative.eq(query.getShortUrlRelative()));
        }
        /** 长链地址后缀 精确 */
        if (!ObjectUtils.isEmpty(query.getLongUrlPre())) {
            list.add(qdo.longUrlPre.eq(query.getLongUrlPre()));
        }
        /** 数据归属租户id 精确 */
        if (!ObjectUtils.isEmpty(query.getSecTenantId())) {
            list.add(qdo.secTenantId.eq(query.getSecTenantId()));
        }
        /** 拓展1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ShortUrlConfVO queryByKey(Long id) {
        JPAQuery<ShortUrlConfVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ShortUrlConfVO> queryListDynamic(ShortUrlConfQuery query) {
        JPAQuery<ShortUrlConfVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ShortUrlConfVO> queryPaging(ShortUrlConfQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ShortUrlConfVO> jpaQuery = getJpaQueryWhere(query);
        List<ShortUrlConfVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ShortUrlConfVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ShortUrlConfDO save(ShortUrlConfDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ShortUrlConfDO> saveAll(List<ShortUrlConfDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ShortUrlConfPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 短链地址前缀
        if (payload.getShortUrlPre() != null) {
            update.set(qdo.shortUrlPre, payload.getShortUrlPre());
        }
        if (payload.getShortUrlRelative() != null) {
            update.set(qdo.shortUrlRelative, payload.getShortUrlRelative());
        }
        // 长链地址后缀
        if (payload.getLongUrlPre() != null) {
            update.set(qdo.longUrlPre, payload.getLongUrlPre());
        }
        // 拓展1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 短链地址前缀
            if (nullFields.contains("shortUrlPre")) {
                update.setNull(qdo.shortUrlPre);
            }
            // 长链地址后缀
            if (nullFields.contains("longUrlPre")) {
                update.setNull(qdo.longUrlPre);
            }
            // 拓展1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

