package com.elitesland.tw.tw5.server.prd.ts.dao;

import com.elitesland.tw.tw5.api.prd.ts.payload.TsApprovalResPayload;
import com.elitesland.tw.tw5.api.prd.ts.query.TsApprovalResQuery;
import com.elitesland.tw.tw5.api.prd.ts.vo.TsApprovalResVO;
import com.elitesland.tw.tw5.server.prd.ts.entity.TsApprovalResDO;
import com.elitesland.tw.tw5.server.prd.ts.entity.QTsApprovalResDO;
import com.elitesland.tw.tw5.server.prd.ts.repo.TsApprovalResRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 工时审批资源管理
 *
 * @author carl
 * @date 2024-05-04
 */
@Repository
@RequiredArgsConstructor
public class TsApprovalResDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TsApprovalResRepo repo;
    private final QTsApprovalResDO qdo = QTsApprovalResDO.tsApprovalResDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TsApprovalResVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TsApprovalResVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 工时id
                qdo.timesheetId,
                // 配置id
                qdo.configId,
                // 审批资源id
                qdo.approvalResId,
                // 审批状态
                qdo.approvalStatus,
                // 审批资源描述(项目经理，交付负责人等)
                qdo.approvalSource,
                // 审批时间
                qdo.approvalTime,
                // 首次审批时间（前端不用）
                qdo.fristApprovalTime,
                // 排序
                qdo.sortIndex,
                // 最后审批人
                qdo.lastFlag
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TsApprovalResVO> getJpaQueryWhere(TsApprovalResQuery query) {
        JPAQuery<TsApprovalResVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TsApprovalResQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TsApprovalResQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 工时id 精确 */
        if (!ObjectUtils.isEmpty(query.getTimesheetId())) {
            list.add(qdo.timesheetId.eq(query.getTimesheetId()));
        }
        /** 工时id 精确 */
        if (!ObjectUtils.isEmpty(query.getTimesheetIds())) {
            list.add(qdo.timesheetId.in(query.getTimesheetIds()));
        }
        /** 配置id 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigId())) {
            list.add(qdo.configId.eq(query.getConfigId()));
        }
        /** 审批资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovalResId())) {
            list.add(qdo.approvalResId.eq(query.getApprovalResId()));
        }
        /** 审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovalStatus())) {
            list.add(qdo.approvalStatus.eq(query.getApprovalStatus()));
        }
        /** 审批资源描述(项目经理，交付负责人等) 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovalSource())) {
            list.add(qdo.approvalSource.eq(query.getApprovalSource()));
        }
        /** 审批时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovalTime())) {
            list.add(qdo.approvalTime.eq(query.getApprovalTime()));
        }
        /** 首次审批时间（前端不用） 精确 */
        if (!ObjectUtils.isEmpty(query.getFristApprovalTime())) {
            list.add(qdo.fristApprovalTime.eq(query.getFristApprovalTime()));
        }
        /** 排序 精确 */
        if (!ObjectUtils.isEmpty(query.getSortIndex())) {
            list.add(qdo.sortIndex.eq(query.getSortIndex()));
        }
        /** 最后审批人 精确 */
        if (!ObjectUtils.isEmpty(query.getLastFlag())) {
            list.add(qdo.lastFlag.eq(query.getLastFlag()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TsApprovalResVO queryByKey(Long id) {
        JPAQuery<TsApprovalResVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TsApprovalResVO> queryListDynamic(TsApprovalResQuery query) {
        JPAQuery<TsApprovalResVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TsApprovalResVO> queryPaging(TsApprovalResQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<TsApprovalResVO> jpaQuery = getJpaQueryWhere(query);
        List<TsApprovalResVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TsApprovalResVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TsApprovalResDO save(TsApprovalResDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TsApprovalResDO> saveAll(List<TsApprovalResDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TsApprovalResPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 工时id
        if (payload.getTimesheetId() != null) {
            update.set(qdo.timesheetId, payload.getTimesheetId());
        }
        // 配置id
        if (payload.getConfigId() != null) {
            update.set(qdo.configId, payload.getConfigId());
        }
        // 审批资源id
        if (payload.getApprovalResId() != null) {
            update.set(qdo.approvalResId, payload.getApprovalResId());
        }
        // 审批状态
        if (payload.getApprovalStatus() != null) {
            update.set(qdo.approvalStatus, payload.getApprovalStatus());
        }
        // 审批资源描述(项目经理，交付负责人等)
        if (payload.getApprovalSource() != null) {
            update.set(qdo.approvalSource, payload.getApprovalSource());
        }
        // 审批时间
        if (payload.getApprovalTime() != null) {
            update.set(qdo.approvalTime, payload.getApprovalTime());
        }
        // 首次审批时间（前端不用）
        if (payload.getFristApprovalTime() != null) {
            update.set(qdo.fristApprovalTime, payload.getFristApprovalTime());
        }
        // 排序
        if (payload.getSortIndex() != null) {
            update.set(qdo.sortIndex, payload.getSortIndex());
        }
        // 最后审批人
        if (payload.getLastFlag() != null) {
            update.set(qdo.lastFlag, payload.getLastFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 工时id
            if (nullFields.contains("timesheetId")) {
                update.setNull(qdo.timesheetId);
            }
            // 配置id
            if (nullFields.contains("configId")) {
                update.setNull(qdo.configId);
            }
            // 审批资源id
            if (nullFields.contains("approvalResId")) {
                update.setNull(qdo.approvalResId);
            }
            // 审批状态
            if (nullFields.contains("approvalStatus")) {
                update.setNull(qdo.approvalStatus);
            }
            // 审批资源描述(项目经理，交付负责人等)
            if (nullFields.contains("approvalSource")) {
                update.setNull(qdo.approvalSource);
            }
            // 审批时间
            if (nullFields.contains("approvalTime")) {
                update.setNull(qdo.approvalTime);
            }
            // 首次审批时间（前端不用）
            if (nullFields.contains("fristApprovalTime")) {
                update.setNull(qdo.fristApprovalTime);
            }
            // 排序
            if (nullFields.contains("sortIndex")) {
                update.setNull(qdo.sortIndex);
            }
            // 最后审批人
            if (nullFields.contains("lastFlag")) {
                update.setNull(qdo.lastFlag);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param timesheetId 主集合
     * @return 删除的行数
     */
    public long deleteSoftByTimesheetId(Long timesheetId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.timesheetId.eq(timesheetId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


}

