package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalTaskSettleDetailPayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalTaskSettleDetailQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalTaskSettleDetailVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalTaskSettleDetailDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalTaskSettleDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalTaskSettleDetailDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalTaskSettleDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 任务包
 *
 * @author carl
 * @date 2023-11-15
 */
@Repository
@RequiredArgsConstructor
public class CalTaskSettleDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalTaskSettleDetailRepo repo;
    private final QCalTaskSettleDetailDO qdo = QCalTaskSettleDetailDO.calTaskSettleDetailDO;
    private final QCalTaskSettleDO qdoSettle = QCalTaskSettleDO.calTaskSettleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalTaskSettleDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalTaskSettleDetailVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 结算单id
                qdo.settleId,
                // 项目id
                qdo.projId,
                // 任务id
                qdo.taskId,
                qdo.taskName,
                // 派发当量
                qdo.distedEqva,
                // 已结算当量
                qdo.settledEqva,
                // 申请结算当量数
                qdo.applySettleEqva,
                // 快照完工百分比
                qdo.ssCompPercent,
                // 申请结算百分比
                qdo.applySettlePercent,
                // 申请结算总额
                qdo.applySettleAmt,
                // 实际结算当量数
                qdo.approveSettleEqva,
                // 实际结算总额
                qdo.approveSettleAmt,
                qdo.planStartDate,
                qdo.planEndDate,
                qdo.taskStatus,
                // 币种
                qdo.currCode,
                // 结算说明
                qdo.settleDesc
        )).from(qdo);
    }


    /**
     * 查询任务在指定状态下的结算明细数据
     *
     * @param taskIds
     * @param statuss
     * @return
     */
    public List<String> queryTaskSettleDetails(List<Long> taskIds, List<String> statuss) {
        JPAQuery<String> jpaQuery = jpaQueryFactory.select(qdoSettle.settleNo).from(qdo).leftJoin(qdoSettle).on(qdo.settleId.eq(qdoSettle.id));
        jpaQuery.where(qdoSettle.deleteFlag.eq(0));
        jpaQuery.where(qdoSettle.settleStatus.in(statuss));
        jpaQuery.where(qdo.taskId.in(taskIds));
        jpaQuery.groupBy(qdoSettle.settleNo);
        return jpaQuery.fetch();
    }


    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalTaskSettleDetailVO> getJpaQueryWhere(CalTaskSettleDetailQuery query) {
        JPAQuery<CalTaskSettleDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalTaskSettleDetailQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalTaskSettleDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 结算单id 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleId())) {
            list.add(qdo.settleId.eq(query.getSettleId()));
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        }
        /** 任务id 精确 */
        if (!ObjectUtils.isEmpty(query.getTaskId())) {
            list.add(qdo.taskId.eq(query.getTaskId()));
        }
        /** 派发当量 精确 */
        if (!ObjectUtils.isEmpty(query.getDistedEqva())) {
            list.add(qdo.distedEqva.eq(query.getDistedEqva()));
        }
        /** 已结算当量 精确 */
        if (!ObjectUtils.isEmpty(query.getSettledEqva())) {
            list.add(qdo.settledEqva.eq(query.getSettledEqva()));
        }
        /** 申请结算当量数 精确 */
        if (!ObjectUtils.isEmpty(query.getApplySettleEqva())) {
            list.add(qdo.applySettleEqva.eq(query.getApplySettleEqva()));
        }
        /** 快照完工百分比 精确 */
        if (!ObjectUtils.isEmpty(query.getSsCompPercent())) {
            list.add(qdo.ssCompPercent.eq(query.getSsCompPercent()));
        }
        /** 申请结算百分比 精确 */
        if (!ObjectUtils.isEmpty(query.getApplySettlePercent())) {
            list.add(qdo.applySettlePercent.eq(query.getApplySettlePercent()));
        }
        /** 申请结算总额 精确 */
        if (!ObjectUtils.isEmpty(query.getApplySettleAmt())) {
            list.add(qdo.applySettleAmt.eq(query.getApplySettleAmt()));
        }
        /** 实际结算当量数 精确 */
        if (!ObjectUtils.isEmpty(query.getApproveSettleEqva())) {
            list.add(qdo.approveSettleEqva.eq(query.getApproveSettleEqva()));
        }
        /** 实际结算总额 精确 */
        if (!ObjectUtils.isEmpty(query.getApproveSettleAmt())) {
            list.add(qdo.approveSettleAmt.eq(query.getApproveSettleAmt()));
        }
        /** 币种 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrCode())) {
            list.add(qdo.currCode.eq(query.getCurrCode()));
        }
        /** 结算说明 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleDesc())) {
            list.add(qdo.settleDesc.eq(query.getSettleDesc()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalTaskSettleDetailVO queryByKey(Long id) {
        JPAQuery<CalTaskSettleDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalTaskSettleDetailVO> queryListDynamic(CalTaskSettleDetailQuery query) {
        JPAQuery<CalTaskSettleDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalTaskSettleDetailVO> queryPaging(CalTaskSettleDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CalTaskSettleDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<CalTaskSettleDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalTaskSettleDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalTaskSettleDetailDO save(CalTaskSettleDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalTaskSettleDetailDO> saveAll(List<CalTaskSettleDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalTaskSettleDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 结算单id
        if (payload.getSettleId() != null) {
            update.set(qdo.settleId, payload.getSettleId());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 任务id
        if (payload.getTaskId() != null) {
            update.set(qdo.taskId, payload.getTaskId());
        }
        // 派发当量
        if (payload.getDistedEqva() != null) {
            update.set(qdo.distedEqva, payload.getDistedEqva());
        }
        // 已结算当量
        if (payload.getSettledEqva() != null) {
            update.set(qdo.settledEqva, payload.getSettledEqva());
        }
        // 申请结算当量数
        if (payload.getApplySettleEqva() != null) {
            update.set(qdo.applySettleEqva, payload.getApplySettleEqva());
        }
        // 快照完工百分比
        if (payload.getSsCompPercent() != null) {
            update.set(qdo.ssCompPercent, payload.getSsCompPercent());
        }
        // 申请结算百分比
        if (payload.getApplySettlePercent() != null) {
            update.set(qdo.applySettlePercent, payload.getApplySettlePercent());
        }
        // 申请结算总额
        if (payload.getApplySettleAmt() != null) {
            update.set(qdo.applySettleAmt, payload.getApplySettleAmt());
        }
        // 实际结算当量数
        if (payload.getApproveSettleEqva() != null) {
            update.set(qdo.approveSettleEqva, payload.getApproveSettleEqva());
        }
        // 实际结算总额
        if (payload.getApproveSettleAmt() != null) {
            update.set(qdo.approveSettleAmt, payload.getApproveSettleAmt());
        }
        // 币种
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 结算说明
        if (payload.getSettleDesc() != null) {
            update.set(qdo.settleDesc, payload.getSettleDesc());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 结算单id
            if (nullFields.contains("settleId")) {
                update.setNull(qdo.settleId);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 任务id
            if (nullFields.contains("taskId")) {
                update.setNull(qdo.taskId);
            }
            // 派发当量
            if (nullFields.contains("distedEqva")) {
                update.setNull(qdo.distedEqva);
            }
            // 已结算当量
            if (nullFields.contains("settledEqva")) {
                update.setNull(qdo.settledEqva);
            }
            // 申请结算当量数
            if (nullFields.contains("applySettleEqva")) {
                update.setNull(qdo.applySettleEqva);
            }
            // 快照完工百分比
            if (nullFields.contains("ssCompPercent")) {
                update.setNull(qdo.ssCompPercent);
            }
            // 申请结算百分比
            if (nullFields.contains("applySettlePercent")) {
                update.setNull(qdo.applySettlePercent);
            }
            // 申请结算总额
            if (nullFields.contains("applySettleAmt")) {
                update.setNull(qdo.applySettleAmt);
            }
            // 实际结算当量数
            if (nullFields.contains("approveSettleEqva")) {
                update.setNull(qdo.approveSettleEqva);
            }
            // 实际结算总额
            if (nullFields.contains("approveSettleAmt")) {
                update.setNull(qdo.approveSettleAmt);
            }
            // 币种
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 结算说明
            if (nullFields.contains("settleDesc")) {
                update.setNull(qdo.settleDesc);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据结算单id逻辑删除
     *
     * @param settleIds 结算单id
     * @return 删除的行数
     */
    public long deleteSoftBySettleId(List<Long> settleIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.settleId.in(settleIds));
        return update.execute();
    }
}

