package com.elitesland.tw.tw5.server.prd.cal.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalNormSettlePayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalAccountQuery;
import com.elitesland.tw.tw5.api.prd.cal.query.CalNormSettleQuery;
import com.elitesland.tw.tw5.api.prd.cal.service.CalAccountService;
import com.elitesland.tw.tw5.api.prd.cal.service.CalNormSettleService;
import com.elitesland.tw.tw5.api.prd.cal.service.CalResourceService;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalAccountVO;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalNormSettleVO;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConReceivablePlanQuery;
import com.elitesland.tw.tw5.api.prd.salecon.service.ConReceivablePlanService;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConReceivablePlanVO;
import com.elitesland.tw.tw5.server.common.ExcelUtil;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.DateUtil;
import com.elitesland.tw.tw5.server.prd.cal.convert.CalNormSettleConvert;
import com.elitesland.tw.tw5.server.prd.cal.dao.CalNormSettleDAO;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalNormSettleDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalNormSettleRepo;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CalNormCreateTypeEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CalSettleStatusEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 泛用金额结算管理
 *
 * @author carl
 * @date 2023-11-20
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CalNormSettleServiceImpl extends BaseServiceImpl implements CalNormSettleService {

    private final CalNormSettleRepo calNormSettleRepo;
    private final CalNormSettleDAO calNormSettleDAO;
    private final CalResourceService calResourceService;
    //    private final TransactionUtilService transactionUtilService;
    private final ExcelUtil excelUtil;
    private final ConReceivablePlanService conReceivablePlanService;
    private final CalAccountService calAccountService;

    @Override
    public PagingVO<CalNormSettleVO> queryPaging(CalNormSettleQuery query) {
        return calNormSettleDAO.queryPaging(query);
    }

    @Override
    public List<CalNormSettleVO> queryListDynamic(CalNormSettleQuery query) {
        return calNormSettleDAO.queryListDynamic(query);
    }

    @Override
    public CalNormSettleVO queryByKey(Long key) {
        return calNormSettleDAO.queryByKey(key);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public CalNormSettleVO insertOrUpdate(CalNormSettlePayload payload) {
        //数据校验
        checkData(payload);
        Long loginUserId = GlobalUtil.getLoginUserId();
        LocalDateTime applyDate = LocalDateTime.now();
        if (payload.getId() == null) {
            String code = generateSeqNum("CAL_NORM_SETTLE");
            payload.setSettleNo(code);
        } else {
            CalNormSettleVO calNormSettleVO = calNormSettleDAO.queryByKey(payload.getId());
            if (calNormSettleVO == null || !calNormSettleVO.getSettleStatus().equals(CalSettleStatusEnum.CREATE.getCode())) {
                throw TwException.error("", "不支持改状态数据修改");
            }
            applyDate = calNormSettleVO.getApplyDate();
            loginUserId = calNormSettleVO.getApplyResId();
            payload.setCreateTime(calNormSettleVO.getCreateTime());
            payload.setCreateUserId(calNormSettleVO.getCreateUserId());
        }
        String status = CalSettleStatusEnum.CREATE.getCode();
        if (payload.getIsSubmit() != null && payload.getIsSubmit() == 1) {
            status = CalSettleStatusEnum.FINISH.getCode();
        }
        payload.setSettleStatus(status);
        payload.setApplyResId(loginUserId);
        payload.setApplyDate(applyDate);

        CalNormSettleDO entityDo = CalNormSettleConvert.INSTANCE.toDo(payload);
        CalNormSettleVO calNormSettleVO = CalNormSettleConvert.INSTANCE.toVo(calNormSettleRepo.save(entityDo));
        //审批通过产生流水记录
        if (status.equals(CalSettleStatusEnum.FINISH.getCode())) {
            calResourceService.amountSettleTurnover(calNormSettleVO, 1);
//            //开启事务执行
//            transactionUtilService.executeWithRunnable(() -> {
//                calResourceService.amountSettleTurnover(calNormSettleVO, 1);
//            });
        }
        return calNormSettleVO;
    }

    /**
     * 数据校验
     *
     * @param payload
     */
    void checkData(CalNormSettlePayload payload) {
        if (payload.getSettleDate() == null) {
            throw TwException.error("", "结算期间不可为空");
        }
        if (ObjectUtils.isEmpty(payload.getCreateType())) {
            throw TwException.error("", "单据创建类型不可为空");
        }
        if (ObjectUtils.isEmpty(payload.getBusiType())) {
            throw TwException.error("", "业务类型不可为空");
        }
        if (payload.getApproveSettleAmt() == null || payload.getApproveSettleAmt().compareTo(BigDecimal.ZERO) <= 0) {
            throw TwException.error("", "交易额不可为空且大于0");
        }
        if (payload.getInAccountId() == null) {
            throw TwException.error("", "支出方账户id不可为空");
        }
        if (payload.getInAccount() == null) {
            throw TwException.error("", "支出方账户不可为空");
        }
        if (payload.getOutAccountId() == null) {
            throw TwException.error("", "收入方账户id不可为空");
        }
        if (payload.getOutAccount() == null) {
            throw TwException.error("", "收入方账户不可为空");
        }
        if (payload.getContractId() != null) {
            if (!StringUtils.hasText(payload.getContractName())) {
                throw TwException.error("", "子合同名称不可为空");
            }
        }
        if (payload.getProjId() != null) {
            if (!StringUtils.hasText(payload.getProjName())) {
                throw TwException.error("", "项目名称不可为空");
            }
        }
        if (payload.getInAccountId().equals(payload.getOutAccountId())) {
            throw TwException.error("", "支出、收入核算主体不可相同");
        }
        /**
         * 1、业务类型!=“合同收益分配”时，可选所有账户类型=“BU”和“项目”的账户，组件内进行符合数据过滤条件的“账户编号”和“账户名称”合并展示
         *
         * 2、业务类型=“合同收益分配”时，根据“相关项目”中的项目ID到账户表中取核算主体ID项目ID的账户数据，组件内对符合数据过滤条件的“账户编号”和“账户名称”合并展示
         */
        if (payload.getBusiType().equals("CONTRACT")) {
            if (payload.getProjId() == null) {
                throw TwException.error("", "业务类型为【合同收益分配】，项目不可为空");
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(CalNormSettlePayload payload) {
        long result = calNormSettleDAO.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            List<CalNormSettleVO> calNormSettleVOS = calNormSettleDAO.queryByKeys(keys);
            if (calNormSettleVOS.size() > 0) {
                List<CalNormSettleVO> collect = calNormSettleVOS.stream().filter(vo -> !vo.getSettleStatus().equals(CalSettleStatusEnum.CREATE.getCode())).collect(Collectors.toList());
                if (collect.size() > 0) {
                    throw TwException.error("", "仅支持新建状态删除");
                }
            }
            calNormSettleDAO.deleteSoft(keys);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancelNorm(Long key) {
        CalNormSettleVO calNormSettleVO = calNormSettleDAO.queryByKey(key);
        if (calNormSettleVO == null || !calNormSettleVO.getSettleStatus().equals(CalSettleStatusEnum.FINISH.getCode())) {
            throw TwException.error("", "不支持改状态数据修改");
        }
        calNormSettleVO.setRemark("泛用金额结算取消过账 单号：" + calNormSettleVO.getSettleNo());
        calResourceService.amountSettleTurnover(calNormSettleVO, 0);

        CalNormSettlePayload payload = new CalNormSettlePayload();
        payload.setId(key);
        payload.setSettleStatus(CalSettleStatusEnum.CREATE.getCode());
        calNormSettleDAO.updateByKeyDynamic(payload);
    }

    @Override
    public void downloadPlus(HttpServletResponse response) {
        ClassPathResource classPathResource = new ClassPathResource("template/calNormSettle.xlsx");
        try {
            InputStream inputStream = classPathResource.getInputStream();
            Workbook workbook = WorkbookFactory.create(inputStream);
            XSSFSheet batchProjectSheet = (XSSFSheet) workbook.getSheet("泛用金额数据");
            excelUtil.generateRangeList(batchProjectSheet, 1, 2, "LOV", 2, "A"); // 业务类型 数据验证
            String fileName = "泛用金额导入模板-" + LocalDate.now();
            excelUtil.setColumnFormulas(batchProjectSheet, 8, "VLOOKUP(B" + ExcelUtil.formulasReplace + ",LOV!A:B,2,FALSE)");//市场活动
            batchProjectSheet.setColumnHidden(8, true);
            ExcelUtil.writeResponse(response, fileName, workbook);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchImport(MultipartFile file) {
        if (file == null) {
            throw TwException.error("", "上传文件异常");
        }
        Workbook workbook = null;
        try {
            workbook = WorkbookFactory.create(file.getInputStream());
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
            throw TwException.error("", "文件解析异常");
        }
        Sheet sheet = workbook.getSheet("泛用金额数据");
        if (sheet == null) {
            throw TwException.error("", "表结构错误");
        }
        //需处理单据
        List<CalNormSettlePayload> payloads = new ArrayList<>();

        //收款计划编号
        Set<String> recvplanNos = new HashSet<>();
        //账户编号
        Set<String> accountNos = new HashSet<>();
        Long loginUserId = GlobalUtil.getLoginUserId();
        LocalDateTime applyDate = LocalDateTime.now();
        String status = CalSettleStatusEnum.FINISH.getCode();
        String createType = CalNormCreateTypeEnum.HANDWORK.getCode();
        int dataStartRow = 2;
        for (int i = dataStartRow; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            String busiTypeName = ExcelUtil.getCellFormatValue(row.getCell(1));
            if (!StringUtils.hasText(busiTypeName)) {
                break;
            }


            CalNormSettlePayload payload = new CalNormSettlePayload();

            //合同编号
            String contractNo = ExcelUtil.getCellFormatValue(row.getCell(2));
            //收款号
            String recvplanNo = ExcelUtil.getCellFormatValue(row.getCell(3));
            if (StringUtils.hasText(recvplanNo)) {
                if (!StringUtils.hasText(contractNo)) {
                    throw TwException.error("", "相关收款号存在，子合同编号不可为空");
                }
                recvplanNos.add(recvplanNo.trim());
                payload.setRecvplanNo(recvplanNo.trim());
            }
            //结算金额
            String approveSettleAmt = ExcelUtil.getCellFormatValue(row.getCell(4));
            if (!StringUtils.hasText(approveSettleAmt)) {
                throw TwException.error("", "结算金额不可为空");
            }
            //结算期间
            String settleDate = ExcelUtil.getCellFormatValue(row.getCell(5));
            if (!StringUtils.hasText(settleDate)) {
                throw TwException.error("", "结算期间不可为空");
            }
            //支出账户
            String outAccountNo = ExcelUtil.getCellFormatValue(row.getCell(6));
            if (!StringUtils.hasText(outAccountNo)) {
                throw TwException.error("", "支出账户不可为空");
            }
            //收入账户
            String inAccountNo = ExcelUtil.getCellFormatValue(row.getCell(7));
            if (!StringUtils.hasText(inAccountNo)) {
                throw TwException.error("", "收入账户不可为空");
            }
            if (outAccountNo.equals(inAccountNo)) {
                throw TwException.error("", "支出、收入核算主体不可相同");
            }
            accountNos.add(outAccountNo.trim());
            accountNos.add(inAccountNo.trim());
            //合作方式
            String busiType = ExcelUtil.getCellFormatValue(row.getCell(8));

            String code = generateSeqNum("CAL_NORM_SETTLE");
            payload.setSettleNo(code);
            payload.setSettleStatus(status);
            payload.setApplyDate(applyDate);
            payload.setApplyResId(loginUserId);
            payload.setCreateType(createType);
            payload.setApproveSettleAmt(new BigDecimal(approveSettleAmt));
            payload.setBusiType(busiType);
            payload.setContractNo(contractNo);

            payload.setSettleDate(DateUtil.strToLocalDate(settleDate));
            payload.setInAccount(inAccountNo.trim());
            payload.setOutAccount(outAccountNo.trim());

            payloads.add(payload);

        }
        log.info("初始payloads：{}" + payloads.toString());
        if (recvplanNos.size() > 0) {
            //核验收款计划及合同
            checkContractAndRecePlan(payloads, new ArrayList<>(recvplanNos));
        }
        //账户校验
        List<CalAccountVO> calAccountVOS = checkAccount(payloads, new ArrayList<>(accountNos));
        log.info("赋值后payloads：{}" + payloads.toString());

        List<CalNormSettleDO> calNormSettleDOS = CalNormSettleConvert.INSTANCE.toDos(payloads);
        List<CalNormSettleDO> calNormSettleDOS1 = calNormSettleDAO.saveAll(calNormSettleDOS);
        List<CalNormSettleVO> calNormSettleVOS = CalNormSettleConvert.INSTANCE.toVos(calNormSettleDOS1);
        //执行流水增加和账户金额修改
        calResourceService.amountBatchSettleTurnover(calNormSettleVOS, calAccountVOS);
        return true;
    }

    /**
     * 校验账户
     *
     * @param payloads
     * @param accountNos
     */
    List<CalAccountVO> checkAccount(List<CalNormSettlePayload> payloads, List<String> accountNos) {
        CalAccountQuery query = new CalAccountQuery();
        query.setLedgerNos(accountNos);
        List<CalAccountVO> calAccountVOS = calAccountService.queryListDynamic(query);

        if (ObjectUtils.isEmpty(calAccountVOS)) {
            throw TwException.error("", "账户数据不存在");
        }
        payloads.forEach(payload -> {
            Optional<CalAccountVO> first = calAccountVOS.stream().filter(vo -> vo.getLedgerNo().equals(payload.getOutAccount())).findFirst();
            if (!first.isPresent()) {
                throw TwException.error("", "编号：" + payload.getOutAccount() + "账户数据不存在");
            }
            CalAccountVO outAccountVO = first.get();
            if (outAccountVO.getLedgerStatus().equals("0")) {
                throw TwException.error("", outAccountVO.getLedgerNo() + "账户已禁用");
            }

            payload.setOutAccountId(outAccountVO.getId());
            payload.setOutAccount(outAccountVO.getLedgerName());
            Optional<CalAccountVO> first0 = calAccountVOS.stream().filter(vo -> vo.getLedgerNo().equals(payload.getInAccount())).findFirst();
            if (!first0.isPresent()) {
                throw TwException.error("", "编号：" + payload.getInAccount() + "账户数据不存在");
            }
            CalAccountVO inAccountVO = first0.get();
            if (inAccountVO.getLedgerStatus().equals("0")) {
                throw TwException.error("", payload.getInAccount() + "账户已禁用");
            }
            payload.setInAccountId(inAccountVO.getId());
            payload.setInAccount(inAccountVO.getLedgerName());

        });
        return calAccountVOS;
    }

    /**
     * 核验收款计划及合同
     *
     * @param payloads
     * @param receNos
     */
    void checkContractAndRecePlan(List<CalNormSettlePayload> payloads, List<String> receNos) {
        ConReceivablePlanQuery query = new ConReceivablePlanQuery();
        query.setReceNos(receNos);
        List<ConReceivablePlanVO> conReceivablePlanVOS = conReceivablePlanService.queryListDynamic(query);
        if (ObjectUtils.isEmpty(conReceivablePlanVOS)) {
            throw TwException.error("", "收款计划数据不存在");
        }
        payloads.forEach(payload -> {
            if (StringUtils.hasText(payload.getRecvplanNo())) {
                Optional<ConReceivablePlanVO> first = conReceivablePlanVOS.stream().filter(vo -> vo.getReceNo().equals(payload.getRecvplanNo().trim())).findFirst();
                if (!first.isPresent()) {
                    throw TwException.error("", "编号：" + payload.getRecvplanNo() + "收款计划数据不存在");
                }
                ConReceivablePlanVO planVO = first.get();
                if (!StringUtils.hasText(planVO.getSaleConCode()) || !planVO.getSaleConCode().equals(payload.getContractNo().trim())) {
                    throw TwException.error("", "编号：" + payload.getRecvplanNo() + "收款计划归属合同编号为：" + planVO.getSaleConCode());
                }
                payload.setContractId(planVO.getSaleConId());
                payload.setContractName(planVO.getSaleConName());
                payload.setRecvplanId(planVO.getId());
                payload.setReceStage(planVO.getReceStage());
            }

        });
    }
}
