package com.elitesland.tw.tw5.server.prd.my.controller;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.UserVacationPayload;
import com.elitesland.tw.tw5.api.prd.my.query.UserVacationQuery;
import com.elitesland.tw.tw5.api.prd.my.service.UserVacationService;
import com.elitesland.tw.tw5.api.prd.my.vo.UserVacationVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.excel.ExcelUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.List;

@Api(tags = "假期管理")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/my/userVacation")
@Slf4j
public class UserVacationController {

    private final UserVacationService userVacationService;

    private final UdcUtil udcUtil;

    /**
     * 分页
     *
     * @param query
     * @return
     */
    @ApiModelProperty("假期管理分页查询")
    @GetMapping("/paging")
    @UdcNameClass
    public TwOutputUtil<PagingVO<UserVacationVO>> paging(UserVacationQuery query) {
        PagingVO<UserVacationVO> paging = userVacationService.page(query);
        return TwOutputUtil.ok(paging);
    }

    /**
     * 保存
     *
     * @param payload
     * @return
     */
    @ApiModelProperty("假期管理保存")
    @PostMapping("/save")
    public TwOutputUtil save(UserVacationPayload payload) {
        userVacationService.save(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 查询详情
     *
     * @param id
     * @return
     */
    @ApiModelProperty("假期管理查询")
    @GetMapping("/query/{id}")
    public TwOutputUtil query(@PathVariable Long id) {
        return TwOutputUtil.ok(userVacationService.queryById(id));
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @ApiModelProperty("假期管理删除")
    @DeleteMapping("/delete")
    public TwOutputUtil delete(Long[] ids) {
        userVacationService.delete(ids);
        return TwOutputUtil.ok();
    }

    /**
     * 批量修改有效期
     *
     * @param payloadList
     * @return
     */
    @ApiModelProperty("假期管理-批量修改有效期")
    @PostMapping("/update/expirationDate/batch")
    public TwOutputUtil updateExpirationBatch(@RequestBody List<UserVacationPayload> payloadList) {
        userVacationService.updateExpirationBatch(payloadList);
        return TwOutputUtil.ok();
    }

    /**
     * 查询本人剩余假期
     *
     * @param
     * @return
     */
    @ApiModelProperty("假期管理-本人可用假期")
    @GetMapping("/available")
    @UdcNameClass
    public TwOutputUtil available() {
        final Long loginUserId = GlobalUtil.getLoginUserId();
        return TwOutputUtil.ok(userVacationService.getAvailableVacation(loginUserId, null));
    }

    /**
     * 查询员工可用剩余假期
     *
     * @param
     * @return
     */
    @ApiModelProperty("假期管理-查询员工可用剩余假期")
    @GetMapping("/available/{userId}")
    @UdcNameClass
    public TwOutputUtil available(@PathVariable Long userId) {
        return TwOutputUtil.ok(userVacationService.getAvailableVacation(userId, null));
    }

    /**
     * 假期管理导出
     *
     * @param
     * @return
     */
    @ApiModelProperty("假期管理导出")
    @GetMapping("/exportExcel")
    public void exportExcel(UserVacationQuery query, HttpServletResponse response) throws IOException {
        List<UserVacationVO> list = userVacationService.list(query);
        if (!CollectionUtils.isEmpty(list)) {
            list = udcUtil.translateList(list);
            //定义文件名称
            String sheetName = "假期管理导出";
            //对文件名进行固定格式编码
            String fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
            //设置请求响应内容类型
            //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            //设置请求响应内容编码方式
            response.setCharacterEncoding("utf-8");
            //文件下载，指定默认名
            response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

            final ExcelWriterSheetBuilder sheet = EasyExcel.write(response.getOutputStream(), UserVacationVO.class)
                    .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                    .sheet(sheetName);
            // 列
            ExcelUtil.excelHelper(sheet, UserVacationVO.class, null);
            //写入
            sheet.doWrite(list);
        }
    }

    /**
     * 下载导入模板
     *
     * @return result
     */
    @ApiOperation("下载导入模板")
    @GetMapping(value = "/tmpDownload")
    public void tmpDownload(HttpServletResponse response) {
        userVacationService.tmpDownload(response);
    }

    /**
     * 批量导入excel
     *
     * @return result
     */
    @ApiOperation("批量导入excel")
    @PostMapping(value = "/batchImport", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public TwOutputUtil<String> batchImport(MultipartFile file, Boolean force) {
        return TwOutputUtil.ok(userVacationService.importBatch(file, force));
    }

}
