package com.elitesland.tw.tw5.server.prd.partner.business.dao;

import com.elitescloud.cloudt.common.base.PagingVO;

import com.elitesland.tw.tw5.api.prd.partner.business.payload.BusinessOperationEmployeesPayload;
import com.elitesland.tw.tw5.api.prd.partner.business.query.BusinessOperationEmployeesQuery;
import com.elitesland.tw.tw5.api.prd.partner.business.vo.BusinessOperationEmployeesVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;

import com.elitesland.tw.tw5.server.prd.partner.business.entity.BusinessOperationEmployeesDO;
import com.elitesland.tw.tw5.server.prd.partner.business.entity.QBusinessOperationEmployeesDO;
import com.elitesland.tw.tw5.server.prd.partner.business.repo.BusinessOperationEmployeesRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 业务合作伙伴-企业主要人员
 *
 * @author wangly
 * @date 2023-05-18
 */
@Repository
@RequiredArgsConstructor
public class BusinessOperationEmployeesDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BusinessOperationEmployeesRepo repo;
    private final QBusinessOperationEmployeesDO qdo = QBusinessOperationEmployeesDO.businessOperationEmployeesDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessOperationEmployeesVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BusinessOperationEmployeesVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 地址簿id
                qdo.bookId,
                // 主要人员姓名
                qdo.name,
                // 主要人员职位
                qdo.title,
                // 是否历史，是否历史；0：否1：是
                qdo.isHistory,
                // 业务伙伴主键 business_partner.id
                qdo.partnerId,
                // 业务伙伴名称
                qdo.partnerName
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessOperationEmployeesVO> getJpaQueryWhere(BusinessOperationEmployeesQuery query) {
        JPAQuery<BusinessOperationEmployeesVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BusinessOperationEmployeesQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BusinessOperationEmployeesQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 地址簿id 精确 */
        if (!ObjectUtils.isEmpty(query.getBookId())) {
            list.add(qdo.bookId.eq(query.getBookId()));
        }
        /** 主要人员姓名 精确 */
        if (!ObjectUtils.isEmpty(query.getName())) {
            list.add(qdo.name.eq(query.getName()));
        }
        /** 主要人员职位 精确 */
        if (!ObjectUtils.isEmpty(query.getTitle())) {
            list.add(qdo.title.eq(query.getTitle()));
        }
        /** 是否历史，是否历史；0：否1：是 精确 */
        if (!ObjectUtils.isEmpty(query.getIsHistory())) {
            list.add(qdo.isHistory.eq(query.getIsHistory()));
        }
        /** 业务伙伴主键 business_partner.id 精确 */
        if (!ObjectUtils.isEmpty(query.getPartnerId())) {
            list.add(qdo.partnerId.eq(query.getPartnerId()));
        }
        /** 业务伙伴名称 精确 */
        if (!ObjectUtils.isEmpty(query.getPartnerName())) {
            list.add(qdo.partnerName.eq(query.getPartnerName()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BusinessOperationEmployeesVO queryByKey(Long id) {
        JPAQuery<BusinessOperationEmployeesVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BusinessOperationEmployeesVO> queryListDynamic(BusinessOperationEmployeesQuery query) {
        JPAQuery<BusinessOperationEmployeesVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BusinessOperationEmployeesVO> queryPaging(BusinessOperationEmployeesQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<BusinessOperationEmployeesVO> jpaQuery = getJpaQueryWhere(query);
        List<BusinessOperationEmployeesVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BusinessOperationEmployeesVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BusinessOperationEmployeesDO save(BusinessOperationEmployeesDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BusinessOperationEmployeesDO> saveAll(List<BusinessOperationEmployeesDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BusinessOperationEmployeesPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 地址簿id
        if (payload.getBookId() != null) {
            update.set(qdo.bookId, payload.getBookId());
        }
        // 主要人员姓名
        if (payload.getName() != null) {
            update.set(qdo.name, payload.getName());
        }
        // 主要人员职位
        if (payload.getTitle() != null) {
            update.set(qdo.title, payload.getTitle());
        }
        // 是否历史，是否历史；0：否1：是
        if (payload.getIsHistory() != null) {
            update.set(qdo.isHistory, payload.getIsHistory());
        }
        // 业务伙伴主键 business_partner.id
        if (payload.getPartnerId() != null) {
            update.set(qdo.partnerId, payload.getPartnerId());
        }
        // 业务伙伴名称
        if (payload.getPartnerName() != null) {
            update.set(qdo.partnerName, payload.getPartnerName());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 地址簿id
            if (nullFields.contains("bookId")) {
                update.setNull(qdo.bookId);
            }
            // 主要人员姓名
            if (nullFields.contains("name")) {
                update.setNull(qdo.name);
            }
            // 主要人员职位
            if (nullFields.contains("title")) {
                update.setNull(qdo.title);
            }
            // 是否历史，是否历史；0：否1：是
            if (nullFields.contains("isHistory")) {
                update.setNull(qdo.isHistory);
            }
            // 业务伙伴主键 business_partner.id
            if (nullFields.contains("partnerId")) {
                update.setNull(qdo.partnerId);
            }
            // 业务伙伴名称
            if (nullFields.contains("partnerName")) {
                update.setNull(qdo.partnerName);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 根据partnerId删除
     *
     * @param partnerId 业务伙伴id
     * @return 删除的行数
     */
    public void deleteByPartnerId(Long partnerId){
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.partnerId.eq(partnerId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }
}

