package com.elitesland.tw.tw5.server.prd.acc.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimExportVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccPayBatchTypeEnum;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimDocStatusEnum;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimPayModeEnum;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccFinancialSubjectDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimDetailDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * 费用报销管理
 *
 * @author sunxw
 * @date 2023-11-22
 */
@Repository
@RequiredArgsConstructor
public class AccReimDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimRepo repo;
    private final QAccReimDO qdo = QAccReimDO.accReimDO;
    private final QAccReimDetailDO qDtldo = QAccReimDetailDO.accReimDetailDO;
    private final QAccFinancialSubjectDO qFinancialSubjectDO = QAccFinancialSubjectDO.accFinancialSubjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 报销单号
                qdo.reimNo,
                // 报销人RES_ID
                qdo.reimUserId,
                // 报销人BU_ID
                qdo.reimOrgId,
                // 报销人职级
                qdo.reimResGrade,
                qdo.reimType,
                qdo.reasonType,
                // 单据类型
                qdo.reimDocType,
                // 相关申请单
                qdo.relatedDocId,
                qdo.relatedDocName,
                // 相关预算
                qdo.relatedBudgetId,
                // 费用归属
                qdo.expenseClassification,
                // 费用承担项目
                qdo.expenseProjectId,
                // 费用承担部门
                qdo.expenseOrgId,
                // 费用承担公司
                qdo.expenseCompany,
                qdo.expenseByType,
                // 记账日期
                qdo.accountingDate,
                // 报销单状态
                qdo.reimStatus,
                qdo.reimTotalAmt,
                qdo.adjustAmt,
                // 报销说明
                qdo.reimRemark,
                // 支付金额
                qdo.payAmt,
                qdo.borrowWriteOffAmt,
                // 付款日期
                qdo.payDate,
                qdo.payStatus,
                // 事由号
                qdo.reasonId,
                // 事由名称
                qdo.reasonName,
                // 付款方式
                qdo.payMethod,
                // 收款账户
                qdo.accountNo,
                // 户名
                qdo.holderName,
                // 收款银行
                qdo.bankName,
                // 收款银行网点
                qdo.bankBranch,
                // 付款批次号
                qdo.batchNo,
                // 付款银行账号
                qdo.payAccountNo,
                // 付款方式
                qdo.payMode,
                qdo.finChargeUpTime,
                qdo.fileCode,
                qdo.applyDate,
                qdo.apprProcName,
                qdo.finPicApprTime,
                qdo.bankTime,
                qdo.apprStatus,
                qdo.contractId,
                qdo.procInstId,
                qdo.procInstName,
                qdo.finPeriodId,
                qdo.finRejectFlag,
                //财务代收单的流程节点key
                qdo.procTaskKey,
                qdo.abnormalFlag,
                qdo.reimDetailFlag,
                qdo.ext1,
                qdo.ext2,
                qdo.ext3,
                qdo.ext4,
                qdo.ext5,
                qdo.ext6
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimVO> getJpaQueryWhere(AccReimQuery query) {
        JPAQuery<AccReimVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        //ID集合
        if (!ObjectUtils.isEmpty(query.getIdList())) {
            list.add(qdo.id.in(query.getIdList()));
        }
        /** 报销单号 精确 */
        if (!ObjectUtils.isEmpty(query.getReimNo())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getReimNo());
            list.add(qdo.reimNo.like(likeStr));
        }
        /** 报销人BU_ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimOrgId())) {
            list.add(qdo.reimOrgId.eq(query.getReimOrgId()));
        }
        /** 报销人职级 精确 */
        if (!ObjectUtils.isEmpty(query.getReimResGrade())) {
            list.add(qdo.reimResGrade.eq(query.getReimResGrade()));
        }
        /** 单据类型 精确 */
        if (!ObjectUtils.isEmpty(query.getReimDocType())) {
            list.add(qdo.reimDocType.eq(query.getReimDocType()));
        }
        /** 相关申请单 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedDocId())) {
            list.add(qdo.relatedDocId.eq(query.getRelatedDocId()));
        }
        /** 相关预算 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedBudgetId())) {
            list.add(qdo.relatedBudgetId.eq(query.getRelatedBudgetId()));
        }
        /** 费用归属 精确 */
        if (!ObjectUtils.isEmpty(query.getExpenseClassification())) {
            list.add(qdo.expenseClassification.eq(query.getExpenseClassification()));
        }
        /** 费用承担项目 精确 */
        if (!ObjectUtils.isEmpty(query.getExpenseProjectId())) {
            list.add(qdo.expenseProjectId.eq(query.getExpenseProjectId()));
        }
        /** 费用承担部门 精确 */
        if (!ObjectUtils.isEmpty(query.getExpenseOrgId())) {
            list.add(qdo.expenseOrgId.eq(query.getExpenseOrgId()));
        }
        /** 费用承担公司 精确 */
        if (!ObjectUtils.isEmpty(query.getExpenseCompany())) {
            list.add(qdo.expenseCompany.eq(query.getExpenseCompany()));
        }

        /** 记账日期 精确 */
        if (!ObjectUtils.isEmpty(query.getAccountingDate())) {
            list.add(qdo.accountingDate.eq(query.getAccountingDate()));
        }
        /** 报销单状态 精确 */
        if (!ObjectUtils.isEmpty(query.getReimStatus())) {
            list.add(qdo.reimStatus.eq(query.getReimStatus()));
        }
        /** 报销说明 精确 */
        if (!ObjectUtils.isEmpty(query.getReimRemark())) {
            list.add(qdo.reimRemark.eq(query.getReimRemark()));
        }
        /** 支付金额 精确 */
        if (!ObjectUtils.isEmpty(query.getPayAmt())) {
            list.add(qdo.payAmt.eq(query.getPayAmt()));
        }
        /** 付款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPayDate())) {
            list.add(qdo.payDate.eq(query.getPayDate()));
        }
        /** 事由号 精确 */
        if (!ObjectUtils.isEmpty(query.getReasonId())) {
            list.add(qdo.reasonId.eq(query.getReasonId()));
        }
        //事由类型 精确
        if (!ObjectUtils.isEmpty(query.getReasonType())) {
            list.add(qdo.reasonType.eq(query.getReasonType()));
        }
        /** 事由名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getReasonName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getReasonName());
            list.add(qdo.reasonName.like(likeStr));
        }
        /** 支付方式 精确 */
        if (!ObjectUtils.isEmpty(query.getPayMethod())) {
            list.add(qdo.payMethod.eq(query.getPayMethod()));
        }
        /** 收款账户 精确 */
        if (!ObjectUtils.isEmpty(query.getAccountNo())) {
            list.add(qdo.accountNo.eq(query.getAccountNo()));
        }
        /** 户名 精确 */
        if (!ObjectUtils.isEmpty(query.getHolderName())) {
            list.add(qdo.holderName.eq(query.getHolderName()));
        }
        /** 收款银行 精确 */
        if (!ObjectUtils.isEmpty(query.getBankName())) {
            list.add(qdo.bankName.eq(query.getBankName()));
        }
        /** 收款银行网点 精确 */
        if (!ObjectUtils.isEmpty(query.getBankBranch())) {
            list.add(qdo.bankBranch.eq(query.getBankBranch()));
        }
        /** 费用批次号 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchNo())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getBatchNo());
            list.add(qdo.batchNo.like(likeStr));
        }
        /** 费用批次号集合 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchNoList())) {
            list.add(qdo.batchNo.in(query.getBatchNoList()));
        }
        /** 付款批次号 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchNoLast())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getBatchNoLast());
            list.add(qdo.batchNoLast.like(likeStr));
        }
        /** 付款银行账号 精确 */
        if (!ObjectUtils.isEmpty(query.getPayAccountNo())) {
            list.add(qdo.payAccountNo.eq(query.getPayAccountNo()));
        }
        if (!ObjectUtils.isEmpty(query.getApprProcName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getApprProcName());
            list.add(qdo.apprProcName.like(likeStr));
        }
        if (!ObjectUtils.isEmpty(query.getNotId())) {
            list.add(qdo.id.ne(query.getNotId()));
        }
        if (!ObjectUtils.isEmpty(query.getApprStatus())) {
            list.add(qdo.apprStatus.eq(query.getApprStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getReimUserId())) {
            if (!ObjectUtils.isEmpty(query.getMyReimUserId())) {
                list.add(qdo.reimUserId.eq(query.getReimUserId()).or(qdo.createUserId.eq(query.getMyReimUserId())));
            } else {
                list.add(qdo.reimUserId.eq(query.getReimUserId()));
            }
        }
        if (!ObjectUtils.isEmpty(query.getReasonTypeList())) {
            list.add(qdo.reasonType.in(query.getReasonTypeList()));
        }
        if (!ObjectUtils.isEmpty(query.getReasonIdList())) {
            list.add(qdo.reasonId.in(query.getReasonIdList()));
        }
        if (!ObjectUtils.isEmpty(query.getReimStatusList())) {
            list.add(qdo.reimStatus.in(query.getReimStatusList()));
        }
        if (!ObjectUtils.isEmpty(query.getNotReimStatusList())) {
            list.add(qdo.reimStatus.notIn(query.getNotReimStatusList()));
        }
        //根据预算科目id获取数据
        if (!ObjectUtils.isEmpty(query.getBudgetItemId())) {
            list.add(qDtldo.budgetItemId.eq(query.getBudgetItemId()));
        }
        //根据预算科目id集合获取数据
        if (!ObjectUtils.isEmpty(query.getBudgetItemIdList())) {
            list.add(qDtldo.budgetItemId.in(query.getBudgetItemIdList()));
        }
        //申请日期
        if (!ObjectUtils.isEmpty(query.getApplyDate())) {
            list.add(qdo.applyDate.eq(query.getApplyDate()));
        }
        //申请日期起止
        if (!ObjectUtils.isEmpty(query.getApplyDateStart()) && !ObjectUtils.isEmpty(query.getApplyDateEnd())) {
            list.add(qdo.applyDate.goe(query.getApplyDateStart()).and(qdo.applyDate.loe(query.getApplyDateEnd())));
        }
        //支付日期起止
        if (!ObjectUtils.isEmpty(query.getPayDateStart()) && !ObjectUtils.isEmpty(query.getPayDateEnd())) {
            list.add(qdo.payDate.goe(query.getPayDateStart()).and(qdo.payDate.loe(query.getPayDateEnd())));
        }
        //记账日期起止
        if (!ObjectUtils.isEmpty(query.getAccountingDateStart()) && !ObjectUtils.isEmpty(query.getAccountingDateEnd())) {
            list.add(qdo.accountingDate.goe(query.getAccountingDateStart()).and(qdo.accountingDate.loe(query.getAccountingDateEnd())));
        }
        //财务收单时间起止
        if (!ObjectUtils.isEmpty(query.getFinChargeUpTimeStart()) && !ObjectUtils.isEmpty(query.getFinChargeUpTimeEnd())) {
            list.add(qdo.finChargeUpTime.goe(query.getFinChargeUpTimeStart()).and(qdo.finChargeUpTime.loe(query.getFinChargeUpTimeEnd())));
        }
        //财务负责人审批时间起止
        if (!ObjectUtils.isEmpty(query.getFinPicApprTimeStart()) && !ObjectUtils.isEmpty(query.getFinPicApprTimeEnd())) {
            list.add(qdo.finPicApprTime.goe(query.getFinPicApprTimeStart()).and(qdo.finPicApprTime.loe(query.getFinPicApprTimeEnd())));
        }
        //支付状态
        if (!ObjectUtils.isEmpty(query.getPayStatus())) {
            list.add(qdo.payStatus.eq(query.getPayStatus()));
        }
        //付款方式
        if (!ObjectUtils.isEmpty(query.getPayMode())) {
            list.add(qdo.payMode.eq(query.getPayMode()));
        }
        //费用承担日期，起始
        if (!ObjectUtils.isEmpty(query.getExpenseDateStart()) && !ObjectUtils.isEmpty(query.getExpenseDateEnd())) {
            list.add(qDtldo.expenseDate.goe(query.getExpenseDateStart()).and(qDtldo.expenseDate.lt(query.getExpenseDateEnd())));
        }
        // 是否异常
        if (!ObjectUtils.isEmpty(query.getAbnormalFlag())) {
            list.add(qdo.abnormalFlag.eq(query.getAbnormalFlag()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimVO queryByKey(Long id) {
        JPAQuery<AccReimVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param reimNo 报销单号
     * @return 结果
     */
    public AccReimVO queryOneByReimNo(String reimNo) {
        JPAQuery<AccReimVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.reimNo.eq(reimNo));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimVO> queryListDynamic(AccReimQuery query) {
        JPAQuery<AccReimVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimVO> queryPaging(AccReimQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimDO save(AccReimDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimDO> saveAll(List<AccReimDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 报销单号
        if (payload.getReimNo() != null) {
            update.set(qdo.reimNo, payload.getReimNo());
        }
        // 报销人RES_ID
        if (payload.getReimUserId() != null) {
            update.set(qdo.reimUserId, payload.getReimUserId());
        }
        // 报销人BU_ID
        if (payload.getReimOrgId() != null) {
            update.set(qdo.reimOrgId, payload.getReimOrgId());
        }
        //更新报销总额
        if (payload.getReimTotalAmt() != null) {
            update.set(qdo.reimTotalAmt, payload.getReimTotalAmt());
        }
        // 报销人职级
        if (payload.getReimResGrade() != null) {
            update.set(qdo.reimResGrade, payload.getReimResGrade());
        }
        // 单据类型
        if (payload.getReimDocType() != null) {
            update.set(qdo.reimDocType, payload.getReimDocType());
        }
        // 相关申请单
        if (payload.getRelatedDocId() != null) {
            update.set(qdo.relatedDocId, payload.getRelatedDocId());
        }
        if (payload.getRelatedDocName() != null) {
            update.set(qdo.relatedDocName, payload.getRelatedDocName());
        }
        // 相关预算
        if (payload.getRelatedBudgetId() != null) {
            update.set(qdo.relatedBudgetId, payload.getRelatedBudgetId());
        }
        // 费用归属
        if (payload.getExpenseClassification() != null) {
            update.set(qdo.expenseClassification, payload.getExpenseClassification());
        }
        // 费用承担项目
        if (payload.getExpenseProjectId() != null) {
            update.set(qdo.expenseProjectId, payload.getExpenseProjectId());
        }
        // 费用承担部门
        if (payload.getExpenseOrgId() != null) {
            update.set(qdo.expenseOrgId, payload.getExpenseOrgId());
        }
        // 费用承担公司
        if (payload.getExpenseCompany() != null) {
            update.set(qdo.expenseCompany, payload.getExpenseCompany());
        }
        // 记账日期
        if (payload.getAccountingDate() != null) {
            update.set(qdo.accountingDate, payload.getAccountingDate());
        }
        // 报销单状态
        if (payload.getReimStatus() != null) {
            update.set(qdo.reimStatus, payload.getReimStatus());
        }
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        if (payload.getApplyDate() != null) {
            update.set(qdo.applyDate, payload.getApplyDate());
        }
        // 报销说明
        if (payload.getReimRemark() != null) {
            update.set(qdo.reimRemark, payload.getReimRemark());
        }
        // 支付金额
        if (payload.getPayAmt() != null) {
            update.set(qdo.payAmt, payload.getPayAmt());
        }
        // 付款日期
        if (payload.getPayDate() != null) {
            update.set(qdo.payDate, payload.getPayDate());
        }
        // 支付方式
        if (payload.getPayMethod() != null) {
            update.set(qdo.payMethod, payload.getPayMethod());
        }
        // 收款账户
        if (payload.getAccountNo() != null) {
            update.set(qdo.accountNo, payload.getAccountNo());
        }
        // 户名
        if (payload.getHolderName() != null) {
            update.set(qdo.holderName, payload.getHolderName());
        }
        // 收款银行
        if (payload.getBankName() != null) {
            update.set(qdo.bankName, payload.getBankName());
        }
        // 收款银行网点
        if (payload.getBankBranch() != null) {
            update.set(qdo.bankBranch, payload.getBankBranch());
        }
        // 记账批次号
        if (payload.getBatchNo() != null) {
            update.set(qdo.batchNo, payload.getBatchNo());
        }
        if (payload.getBankFlag() != null) {
            update.set(qdo.bankFlag, payload.getBankFlag());
        }
        if (payload.getBankTime() != null) {
            update.set(qdo.bankTime, payload.getBankTime());
        }
        if (payload.getPayStatus() != null) {
            update.set(qdo.payStatus, payload.getPayStatus());
        }
        // 付款银行账号
        if (payload.getPayAccountNo() != null) {
            update.set(qdo.payAccountNo, payload.getPayAccountNo());
        }
        if (payload.getApprStatus() != null) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        }
        if (payload.getFinChargeUpTime() != null) {
            update.set(qdo.finChargeUpTime, payload.getFinChargeUpTime());
        }
        if (payload.getFinPicApprTime() != null) {
            update.set(qdo.finPicApprTime, payload.getFinPicApprTime());
        }
        if (payload.getFileCode() != null) {
            update.set(qdo.fileCode, payload.getFileCode());
        }
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        //流程实例名称
        if (payload.getProcInstName() != null) {
            update.set(qdo.procInstName, payload.getProcInstName());
        }
        //财务稽核退回标记
        if (payload.getFinRejectFlag() != null) {
            update.set(qdo.finRejectFlag, payload.getFinRejectFlag());
        }
        //删除标记
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        // 财务代收单的流程节点key
        if (StringUtils.hasText(payload.getProcTaskKey())) {
            update.set(qdo.procTaskKey, payload.getProcTaskKey());
        }
        // 是否异常
        if (!ObjectUtils.isEmpty(payload.getAbnormalFlag())) {
            update.set(qdo.abnormalFlag, payload.getAbnormalFlag());
        }
        // 是否异常
        if (!ObjectUtils.isEmpty(payload.getReimDetailFlag())) {
            update.set(qdo.reimDetailFlag, payload.getReimDetailFlag());
        }
        // 扩展字段1
        if (!ObjectUtils.isEmpty(payload.getExt1())) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 扩展字段2
        if (!ObjectUtils.isEmpty(payload.getExt2())) {
            update.set(qdo.ext2, payload.getExt2());
        } // 扩展字段3
        if (!ObjectUtils.isEmpty(payload.getExt3())) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 扩展字段4
        if (!ObjectUtils.isEmpty(payload.getExt4())) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 扩展字段5
        if (!ObjectUtils.isEmpty(payload.getExt5())) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 扩展字段6
        if (!ObjectUtils.isEmpty(payload.getExt6())) {
            update.set(qdo.ext6, payload.getExt6());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 报销单号
            if (nullFields.contains("reimNo")) {
                update.setNull(qdo.reimNo);
            }
            // 报销人RES_ID
            if (nullFields.contains("reimUserId")) {
                update.setNull(qdo.reimUserId);
            }
            // 报销人BU_ID
            if (nullFields.contains("reimOrgId")) {
                update.setNull(qdo.reimOrgId);
            }
            // 报销人职级
            if (nullFields.contains("reimResGrade")) {
                update.setNull(qdo.reimResGrade);
            }
            // 单据类型
            if (nullFields.contains("reimDocType")) {
                update.setNull(qdo.reimDocType);
            }
            // 相关申请单
            if (nullFields.contains("relatedDocId")) {
                update.setNull(qdo.relatedDocId);
            }
            // 相关预算
            if (nullFields.contains("relatedBudgetId")) {
                update.setNull(qdo.relatedBudgetId);
            }
            // 费用归属
            if (nullFields.contains("expenseClassification")) {
                update.setNull(qdo.expenseClassification);
            }
            // 费用承担项目
            if (nullFields.contains("expenseProjectId")) {
                update.setNull(qdo.expenseProjectId);
            }
            // 费用承担部门
            if (nullFields.contains("expenseOrgId")) {
                update.setNull(qdo.expenseOrgId);
            }
            // 费用承担公司
            if (nullFields.contains("expenseCompany")) {
                update.setNull(qdo.expenseCompany);
            }
            // 记账日期
            if (nullFields.contains("accountingDate")) {
                update.setNull(qdo.accountingDate);
            }
            // 报销单状态
            if (nullFields.contains("reimStatus")) {
                update.setNull(qdo.reimStatus);
            }
            // 报销说明
            if (nullFields.contains("reimRemark")) {
                update.setNull(qdo.reimRemark);
            }
            // 支付金额
            if (nullFields.contains("payAmt")) {
                update.setNull(qdo.payAmt);
            }
            // 付款日期
            if (nullFields.contains("payDate")) {
                update.setNull(qdo.payDate);
            }
            // 事由号
            if (nullFields.contains("reasonId")) {
                update.setNull(qdo.reasonId);
            }
            // 事由名称
            if (nullFields.contains("reasonName")) {
                update.setNull(qdo.reasonName);
            }
            // 支付方式
            if (nullFields.contains("payMethod")) {
                update.setNull(qdo.payMethod);
            }
            // 收款账户
            if (nullFields.contains("accountNo")) {
                update.setNull(qdo.accountNo);
            }
            // 户名
            if (nullFields.contains("holderName")) {
                update.setNull(qdo.holderName);
            }
            // 收款银行
            if (nullFields.contains("bankName")) {
                update.setNull(qdo.bankName);
            }
            // 收款银行网点
            if (nullFields.contains("bankBranch")) {
                update.setNull(qdo.bankBranch);
            }
            // 付款批次号
            if (nullFields.contains("batchNo")) {
                update.setNull(qdo.batchNo);
            }
            // 付款银行账号
            if (nullFields.contains("payAccountNo")) {
                update.setNull(qdo.payAccountNo);
            }
            // 财务付款记账审批时间
            if (nullFields.contains("finChargeUpTime")) {
                update.setNull(qdo.finChargeUpTime);
            }
            // 是否异常
            if (nullFields.contains("abnormalFlag")) {
                update.setNull(qdo.abnormalFlag);
            }
            // 报销明细是否需要事由
            if (nullFields.contains("reimDetailFlag")) {
                update.setNull(qdo.reimDetailFlag);
            }
            // 扩展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 扩展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 扩展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 扩展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 扩展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
            // 扩展字段6
            if (nullFields.contains("ext6")) {
                update.setNull(qdo.ext6);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新付款方式
     *
     * @param keys
     * @param payMode
     */
    public long updatePayMode(Long[] keys, String payMode) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.payMode, payMode)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 财务收单 1.记录首收单时间 2.修改单据状态为财务审核中
     *
     * @param key
     */
    public long updateFinChargeUp(Long key) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.finChargeUpTime, LocalDateTime.now())
                .set(qdo.reimStatus, AccReimDocStatusEnum.FINANCIAL_AUDIT.getCode())
                .where(qdo.id.eq(key));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新批次号
     * 做区分，若为记账处理，对batchNo赋值；若为付款除了，对batchNoLast赋值
     * 注：在batchNoLast赋值前，batchNo必定有值的
     * 2024-04-24 报销单对接jde 将报销单状态 费用记账审批时间修改为jde对接成功后更新
     *
     * @param keys
     */
    public long updateBatchNo(List<Long> keys, String batchNo, String type, Long finPeriodId, LocalDate expenseAccountDate, String payMode) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        if (AccPayBatchTypeEnum.ACCOUNT.getCode().equals(type)) {
            update.set(qdo.batchNo, batchNo);
        } else {
            // 线下支付直接更新状态 否则银企回调成功后再更新单据状态
            if (AccReimPayModeEnum.PAY_MODE_2.getCode().equals(payMode)) {
                update.set(qdo.reimStatus, AccReimDocStatusEnum.APPROVED.getCode());
                update.set(qdo.payDate,LocalDateTime.now());
            }
            update.set(qdo.batchNoLast, batchNo);
        }
        update.set(qdo.finPeriodId, finPeriodId);
        update.where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新批次号
     * 做区分，若为记账处理，对batchNo赋值；若为付款除了，对batchNoLast赋值
     * 注：在batchNoLast赋值前，batchNo必定有值的
     * 2024-04-24 报销单对接jde 将报销单状态 费用记账审批时间修改为jde对接成功后更新
     *
     * @param keys
     */
    public long updateBatchNo(List<Long> keys, String batchNo) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        update.set(qdo.batchNo, batchNo);
        update.where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据记账批次号获取导出的数据
     *
     * @param batchNo
     */
    public List<AccReimExportVO> getExportChargeData(String batchNo) {
        JPAQuery<AccReimExportVO> query = jpaQueryFactory.selectDistinct(Projections.bean(AccReimExportVO.class,
                        qdo.id,
                        qdo.expenseCompany,
                        qdo.reasonType,
                        qdo.reasonId,
                        qdo.reasonName,
                        qdo.reimNo,
                        qdo.reimUserId,
                        qdo.reimDocType,
                        qdo.reimOrgId,
                        qdo.expenseOrgId,
                        qdo.expenseProjectId,
                        qDtldo.id.as("detailId"),
                        qDtldo.reimAmt,
                        qDtldo.adjustAmt,
                        qDtldo.taxAmt,
                        qDtldo.expenseDate,
                        qDtldo.finAccSubjId,
                        qFinancialSubjectDO.accName,
                        qFinancialSubjectDO.accCode.as("finAccSubjCode"),
                        qDtldo.reimRemark.as("dtlReimRemark"),
                        qDtldo.invoiceNum
                )).from(qdo)
                .leftJoin(qDtldo).on(qDtldo.masId.eq(qdo.id).and(qDtldo.deleteFlag.eq(0)))
                .leftJoin(qFinancialSubjectDO).on(qDtldo.finAccSubjId.eq(qFinancialSubjectDO.id).and(qFinancialSubjectDO.deleteFlag.eq(0)))
                .where(qdo.batchNo.eq(batchNo));
        return query.fetch();
    }


    public List<AccReimExportVO> getExportChargeDataByKeys(List<Long> keys) {
        JPAQuery<AccReimExportVO> query = jpaQueryFactory.selectDistinct(Projections.bean(AccReimExportVO.class,
                        qdo.id,
                        qdo.expenseCompany,
                        qdo.reasonType,
                        qdo.reasonId,
                        qdo.reasonName,
                        qdo.reimNo,
                        qdo.reimUserId,
                        qdo.reimDocType,
                        qdo.reimOrgId,
                        qdo.expenseOrgId,
                        qdo.expenseProjectId,
                        qdo.batchNo,
                        qDtldo.id.as("detailId"),
                        qDtldo.reimAmt,
                        qDtldo.taxAmt,
                        qDtldo.expenseDate,
                        qDtldo.finAccSubjId,
                        qDtldo.adjustAmt,
                        qFinancialSubjectDO.accName,
                        qFinancialSubjectDO.accCode.as("finAccSubjCode"),
                        qDtldo.reimRemark.as("dtlReimRemark"),
                        qDtldo.invoiceNum
                )).from(qdo)
                .leftJoin(qDtldo).on(qDtldo.masId.eq(qdo.id).and(qDtldo.deleteFlag.eq(0)))
                .leftJoin(qFinancialSubjectDO).on(qDtldo.finAccSubjId.eq(qFinancialSubjectDO.id).and(qFinancialSubjectDO.deleteFlag.eq(0)))
                .where(qdo.id.in(keys));
        return query.fetch();
    }

    /**
     * 更新收款状态，由于回调时获取不到登录人信息，所以不拼装基础更新
     *
     * @param payload
     * @return
     */
    public long updatePayStatus(AccReimPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        update.set(qdo.payDate, payload.getPayDate());
        update.set(qdo.payStatus, payload.getPayStatus());
        if (!ObjectUtils.isEmpty(payload.getReimStatus())) {
            update.set(qdo.reimStatus, payload.getReimStatus());
        }
        update.set(qdo.modifyTime, payload.getPayDate());
        update.where(qdo.id.eq(payload.getId()));
        return update.execute();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimVO> paging2Budget(AccReimQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qDtldo.count())
                .from(qDtldo)
                .leftJoin(qdo).on(qdo.id.eq(qDtldo.masId).and(qdo.deleteFlag.eq(0)));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimVO> jpaDataQuery = getJpaQueryWhere4Budget(query);
        List<AccReimVO> result = jpaDataQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimVO>builder().records(result).total(total).build();
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimVO> getJpaQueryWhere4Budget(AccReimQuery query) {
        JPAQuery<AccReimVO> jpaQuery = getJpaQuerySelect4Budget();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimVO> getJpaQuerySelect4Budget() {
        return jpaQueryFactory.select(Projections.bean(AccReimVO.class,
                        qdo.id,
                        // 报销单号
                        qdo.reimNo,
                        // 报销人RES_ID
                        qdo.reimUserId,
                        // 报销人BU_ID
                        qdo.reimOrgId,
                        // 报销人职级
                        qdo.reimResGrade,
                        qdo.reimType,
                        qdo.reasonType,
                        // 单据类型
                        qdo.reimDocType,
                        // 费用承担部门
                        qdo.expenseOrgId,
                        // 费用承担公司
                        qdo.expenseCompany,
                        qdo.expenseByType,
                        // 记账日期
                        qdo.accountingDate,
                        // 报销单状态
                        qdo.reimStatus,
                        qDtldo.reimAmt.as("reimTotalAmt"),
                        qdo.adjustAmt,
                        // 报销说明
                        qdo.reimRemark,
                        // 支付金额
                        qdo.payAmt,
                        // 付款日期
                        qdo.payDate,
                        qdo.payStatus,
                        // 事由号
                        qdo.reasonId,
                        // 事由名称
                        qdo.reasonName,
                        qdo.applyDate
                )).from(qDtldo)
                .leftJoin(qdo).on(qDtldo.masId.eq(qdo.id).and(qdo.deleteFlag.eq(0)));
    }

    /**
     * 报销规则专属查询
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimVO> queryList4Rule(AccReimQuery query) {
        JPAQuery<AccReimVO> jpaQuery = getJpaQuerySelect4Rule(query);
        return jpaQuery.fetch();
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimVO> getJpaQuerySelect4Rule(AccReimQuery query) {
        JPAQuery<AccReimVO> jpaQuery = getJpaQuerySelect4Rule();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        //因为关联了子表，所以对主表数据进行去重
        return jpaQuery;
    }

    private JPAQuery<AccReimVO> getJpaQuerySelect4Rule() {
        return jpaQueryFactory.select(Projections.bean(AccReimVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.modifyUserId,
                        qdo.updater,
                        qdo.modifyTime,
                        // 报销单号
                        qdo.reimNo,
                        // 报销人RES_ID
                        qdo.reimUserId,
                        // 报销人BU_ID
                        qdo.reimOrgId,
                        // 报销人职级
                        qdo.reimResGrade,
                        qdo.reimType,
                        qdo.reasonType,
                        // 单据类型
                        qdo.reimDocType,
                        // 相关申请单
                        qdo.relatedDocId,
                        qdo.relatedDocName,
                        // 相关预算
                        qdo.relatedBudgetId,
                        // 费用归属
                        qdo.expenseClassification,
                        // 费用承担项目
                        qdo.expenseProjectId,
                        // 费用承担部门
                        qdo.expenseOrgId,
                        // 费用承担公司
                        qdo.expenseCompany,
                        qdo.expenseByType,
                        // 记账日期
                        qdo.accountingDate,
                        // 报销单状态
                        qdo.reimStatus,
                        qdo.reimTotalAmt,
                        qdo.adjustAmt,
                        // 报销说明
                        qdo.reimRemark,
                        // 支付金额
                        qdo.payAmt,
                        // 付款日期
                        qdo.payDate,
                        qdo.payStatus,
                        // 事由号
                        qdo.reasonId,
                        // 事由名称
                        qdo.reasonName,
                        // 付款方式
                        qdo.payMethod,
                        // 收款账户
                        qdo.accountNo,
                        // 户名
                        qdo.holderName,
                        // 收款银行
                        qdo.bankName,
                        // 收款银行网点
                        qdo.bankBranch,
                        // 付款批次号
                        qdo.batchNo,
                        // 付款银行账号
                        qdo.payAccountNo,
                        // 付款方式
                        qdo.payMode,
                        qdo.finChargeUpTime,
                        qdo.fileCode,
                        qdo.applyDate,
                        qdo.apprProcName,
                        qdo.finPicApprTime,
                        qdo.bankTime,
                        qdo.apprStatus,
                        qdo.contractId,
                        qdo.procInstId,
                        qdo.procInstName,
                        qdo.finPeriodId
                )).from(qdo)
                .leftJoin(qDtldo).on(qDtldo.masId.eq(qdo.id).and(qDtldo.deleteFlag.eq(0)));
    }

    public void updateAccountDigestAndStatusByKeys(List<Long> keys, String expenseProofDigest, String expenseProofNo, String expenseProofStatus, String expenseProofFailReason) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        update.set(qdo.reimStatus, AccReimDocStatusEnum.PENDING_PAYMENT.getCode());
        update.where(qdo.id.in(keys));
        update.execute();
    }

    public void updatePayDigestAndStatusByKeys(List<Long> keys, String payProofDigest, String payProofNo, String payProofStatus, String payProofFailReason) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        update.set(qdo.reimStatus, AccReimDocStatusEnum.APPROVED.getCode());
        update.where(qdo.id.in(keys));
        update.execute();
    }

}

