package com.elitesland.tw.tw5.server.prd.acc.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccBusinessItemPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccBusinessItemQuery;
import com.elitesland.tw.tw5.api.prd.acc.service.AccBusinessItemService;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccBusinessItemVO;
import com.elitesland.tw.tw5.server.common.ExcelUtil;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.prd.acc.convert.AccBusinessItemConvert;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccBusinessItemDAO;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccBusinessItemDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccBusinessItemRepo;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.time.LocalDate;
import java.util.*;

/**
 * 核算项目管理
 *
 * @author carl
 * @date 2023-09-25
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AccBusinessItemServiceImpl extends BaseServiceImpl implements AccBusinessItemService {

    private final AccBusinessItemRepo accBusinessItemRepo;
    private final AccBusinessItemDAO accBusinessItemDAO;
    private final ExcelUtil excelUtil;
    private final UdcUtil udcUtil;
    private final TransactionUtilService transactionUtilService;

    @Override
    public PagingVO<AccBusinessItemVO> queryPaging(AccBusinessItemQuery query) {
        return accBusinessItemDAO.queryPaging(query);
    }

    @Override
    public List<AccBusinessItemVO> queryListDynamic(AccBusinessItemQuery query) {
        return accBusinessItemDAO.queryListDynamic(query);
    }

    @Override
    public AccBusinessItemVO queryByKey(Long key) {
        return accBusinessItemDAO.queryByKey(key);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AccBusinessItemVO insert(AccBusinessItemPayload payload) {
        checkData(payload);
        List<Long> ids = accBusinessItemDAO.queryByCodeAndName(payload.getBusinessCode(), payload.getBusinessName());
        if (ObjectUtils.isEmpty(ids)) {
            if (!StringUtils.hasText(payload.getBusinessStatus())) {
                payload.setBusinessStatus("ACTIVE");
            }

            payload.setBusinessLevel(1);
            //如果是,要判断上级
            if (!ObjectUtils.isEmpty(payload.getParentId())) {
                AccBusinessItemVO accBusinessItemVO = accBusinessItemDAO.queryByKey(payload.getParentId());
                payload.setBusinessLevel(accBusinessItemVO.getBusinessLevel() + 1);
            }
            AccBusinessItemDO entityDo = AccBusinessItemConvert.INSTANCE.toDo(payload);
            return AccBusinessItemConvert.INSTANCE.toVo(accBusinessItemRepo.save(entityDo));
        } else {
            throw TwException.error("", "科目编号和名称不可重复");
        }
    }


    /**
     * 数据校验
     *
     * @param payload
     */
    void checkData(AccBusinessItemPayload payload) {
        if (!StringUtils.hasText(payload.getBusinessCode()) || !StringUtils.hasText(payload.getBusinessName())) {
            throw TwException.error("", "编号和名称不可为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(AccBusinessItemPayload payload) {
        List<Long> ids = accBusinessItemDAO.queryByCodeAndName(payload.getBusinessCode(), payload.getBusinessName());
        if (ObjectUtils.isEmpty(ids) || (ids.size() == 1 && ids.get(0).equals(payload.getId()))) {
            payload.setBusinessLevel(1);
            //如果是,要判断上级
            if (!ObjectUtils.isEmpty(payload.getParentId())) {
                AccBusinessItemVO accBusinessItemVO = accBusinessItemDAO.queryByKey(payload.getParentId());
                payload.setBusinessLevel(accBusinessItemVO.getBusinessLevel() + 1);
            }
            long result = accBusinessItemDAO.updateByKeyDynamic(payload);
            return result;
        } else {
            throw TwException.error("", "科目编号和名称不可重复");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            accBusinessItemDAO.deleteSoft(keys);
        }
    }

    @Override
    public void downloadPlus(HttpServletResponse response, AccBusinessItemQuery query) {
        ClassPathResource classPathResource = new ClassPathResource("template/accBusinessItem.xlsx");
        try {
            InputStream inputStream = classPathResource.getInputStream();
            Workbook workbook = WorkbookFactory.create(inputStream);
            XSSFSheet batchProjectSheet = (XSSFSheet) workbook.getSheet("科目数据");
            excelUtil.generateRangeList(batchProjectSheet, 4, 1, "LOV", 2, "C"); // 科目状态 数据验证
            excelUtil.generateRangeList(batchProjectSheet, 5, 1, "LOV", 2, "E"); // 类别
            excelUtil.generateRangeList(batchProjectSheet, 9, 1, "LOV", 2, "A"); // 汇总 数据验证
            excelUtil.generateRangeList(batchProjectSheet, 3, 1, "科目数据", 2, "C"); // 上级会计科目 数据验证
            String fileName = "核算项目数据-" + LocalDate.now();
            Boolean isHave = false;
            if (!StringUtils.hasText(query.getDownloadType()) || query.getDownloadType().equals("data")) {
                //数据导出
                List<AccBusinessItemVO> accBusinessItemVOS = accBusinessItemDAO.queryListDynamic(query);
                if (!ObjectUtils.isEmpty(accBusinessItemVOS)) {
                    accBusinessItemVOS = udcUtil.translateList(accBusinessItemVOS);
                    isHave = true;
                    int nextRow = 1;
                    for (AccBusinessItemVO accBusinessItemVO : accBusinessItemVOS) {
                        Row row = batchProjectSheet.createRow(nextRow);

                        Optional<AccBusinessItemVO> first = accBusinessItemVOS.stream().filter(vo -> vo.getId().equals(accBusinessItemVO.getParentId())).findFirst();
                        if (first.isPresent()) {
                            excelUtil.setCellValue(row, 3, first.get().getBusinessName());// 上级名称
                        }
                        excelUtil.setCellValue(row, 0, nextRow); // 序号
                        excelUtil.setCellValue(row, 1, accBusinessItemVO.getBusinessCode());// 编号
                        excelUtil.setCellValue(row, 2, accBusinessItemVO.getBusinessName());// 名称
                        excelUtil.setCellValue(row, 4, accBusinessItemVO.getBusinessStatusDesc());// 科目状态
                        excelUtil.setCellValue(row, 5, accBusinessItemVO.getBusinessTypeDesc());// 类别
                        excelUtil.setCellValue(row, 6, accBusinessItemVO.getBusinessType2());// 明细类1
                        excelUtil.setCellValue(row, 7, accBusinessItemVO.getBusinessType3());// 明细类2
                        excelUtil.setCellValue(row, 8, accBusinessItemVO.getDtlAcc());// 明细账

                        String sumFlag = "是";
                        if (accBusinessItemVO.getSumFlag() == null || accBusinessItemVO.getSumFlag() == 0) {
                            sumFlag = "否";
                        }
                        excelUtil.setCellValue(row, 9, sumFlag);// 汇总
                        excelUtil.setCellValue(row, 10, accBusinessItemVO.getLedgerType());// 子账类型
                        excelUtil.setCellValue(row, 11, accBusinessItemVO.getRemark());// 备注
                        nextRow++;
                    }
                }
            } else {
                //下载模板
                fileName = "核算项目导入模板-" + LocalDate.now();
            }

            if (!isHave) {
                excelUtil.setColumnFormulas(batchProjectSheet, 12, "VLOOKUP(D" + ExcelUtil.formulasReplace + ",B:C,1,FALSE)");//上级
                excelUtil.setColumnFormulas(batchProjectSheet, 13, "VLOOKUP(E" + ExcelUtil.formulasReplace + ",LOV!C:D,2,FALSE)");//市场活动
                excelUtil.setColumnFormulas(batchProjectSheet, 14, "VLOOKUP(F" + ExcelUtil.formulasReplace + ",LOV!E:F,2,FALSE)");
                excelUtil.setColumnFormulas(batchProjectSheet, 15, "VLOOKUP(K" + ExcelUtil.formulasReplace + ",LOV!A:B,2,FALSE)");
            }
            batchProjectSheet.setColumnHidden(12, true);
            batchProjectSheet.setColumnHidden(13, true);
            batchProjectSheet.setColumnHidden(14, true);
            batchProjectSheet.setColumnHidden(15, true);
            ExcelUtil.writeResponse(response, fileName, workbook);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchImport(MultipartFile file) {

        if (file == null) {
            throw TwException.error("", "上传文件异常");
        }
        Workbook workbook = null;
        try {
            workbook = WorkbookFactory.create(file.getInputStream());
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
            throw TwException.error("", "文件解析异常");
        }
        Sheet sheet = workbook.getSheet("科目数据");
        if (sheet == null) {
            throw TwException.error("", "表结构错误");
        }
        int dataStartRow = 1;

        List<String> codeNames = new ArrayList<>();
        Map<String, String> map = new HashMap<>();
        List<AccBusinessItemPayload> accBusinessItemPayloads = new ArrayList<>();
        for (int i = dataStartRow; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            String accCode = ExcelUtil.getCellFormatValue(row.getCell(1));
            String accName = ExcelUtil.getCellFormatValue(row.getCell(2));
            if (!StringUtils.hasText(accCode) || !StringUtils.hasText(accName)) {
                break;
            }
            map.put(accName, accCode);
            String codeStr = "," + accCode + ",";
            String nameStr = "," + accName + ",";
            if (codeNames.contains(codeStr) || codeNames.contains(nameStr)) {
                throw TwException.error("", "科目编号或名称不可重复:【" + accCode + "-" + accName + "】");
            } else {
                codeNames.add(codeStr);
                codeNames.add(nameStr);
            }
            String upperName = ExcelUtil.getCellFormatValue(row.getCell(3));//上级名称
            String accType2 = ExcelUtil.getCellFormatValue(row.getCell(6));//明细类1
            String accType3 = ExcelUtil.getCellFormatValue(row.getCell(7));//明细类2
            String dtlAcc = ExcelUtil.getCellFormatValue(row.getCell(8));//明细账
            String ledgerType = ExcelUtil.getCellFormatValue(row.getCell(10));//子账类型
            String remark = ExcelUtil.getCellFormatValue(row.getCell(11));//备注
            String upperCode = ExcelUtil.getCellFormatValue(row.getCell(12));//上级科目编号
            String accStatus = ExcelUtil.getCellFormatValue(row.getCell(13));//状态
            String businessType = ExcelUtil.getCellFormatValue(row.getCell(14));//类别
            if (!StringUtils.hasText(accStatus)) {
                accStatus = "ACTIVE";
            }
            String sumFlag = ExcelUtil.getCellFormatValue(row.getCell(15));
            if (!StringUtils.hasText(sumFlag)) {
                sumFlag = "0";
            }
            AccBusinessItemPayload accBusinessItemPayload = new AccBusinessItemPayload();
            accBusinessItemPayload.setBusinessName(accName);
            accBusinessItemPayload.setBusinessCode(accCode);
            accBusinessItemPayload.setBusinessStatus(accStatus);
            accBusinessItemPayload.setParentCode(upperCode);
            accBusinessItemPayload.setParentName(upperName);
            accBusinessItemPayload.setBusinessType(businessType);
            accBusinessItemPayload.setBusinessType2(accType2);
            accBusinessItemPayload.setBusinessType3(accType3);
            accBusinessItemPayload.setLedgerType(ledgerType);
            accBusinessItemPayload.setSumFlag(Integer.valueOf(sumFlag));
            accBusinessItemPayload.setRemark(remark);
            accBusinessItemPayload.setDtlAcc(dtlAcc);
            if (!StringUtils.hasText(upperName) && !StringUtils.hasText(upperCode)) {
                accBusinessItemPayload.setBusinessLevel(1);
            }

            accBusinessItemPayloads.add(accBusinessItemPayload);
        }
        List<AccBusinessItemDO> accBusinessItemDOs = new ArrayList<>();
        accBusinessItemPayloads.forEach(accBusinessItemPayload -> {
            String upperName = accBusinessItemPayload.getParentName();
            String upperCode = accBusinessItemPayload.getParentCode();
            if (StringUtils.hasText(upperName)) {
                if (!StringUtils.hasText(upperCode)) {
                    accBusinessItemPayload.setParentCode(map.get(upperName));
                }
                //设置科目级别
                setLevel(accBusinessItemPayloads, accBusinessItemPayload);
            }

            accBusinessItemDOs.add(AccBusinessItemConvert.INSTANCE.toDo(accBusinessItemPayload));
        });

        List<AccBusinessItemDO> accBusinessItemDOS = accBusinessItemDAO.saveAll(accBusinessItemDOs);
        List<AccBusinessItemDO> updataDOs = new ArrayList<>();
        accBusinessItemDOS.forEach(accBusinessItemDO -> {
            if (StringUtils.hasText(accBusinessItemDO.getParentCode())) {
                AccBusinessItemDO updataDO = accBusinessItemDOS.stream().filter(masPayload -> masPayload.getBusinessCode().equals(accBusinessItemDO.getParentCode())).findFirst().get();
                accBusinessItemDO.setParentId(updataDO.getId());
                updataDOs.add(accBusinessItemDO);
            }
        });
        //开启事务执行修改
        transactionUtilService.executeWithRunnable(() -> {
            accBusinessItemDAO.saveAll(updataDOs);
        });
        return true;
    }

    /**
     * 为科目设计级别
     *
     * @param accBusinessItemPayloads
     * @param accBusinessItemPayload
     */
    void setLevel(List<AccBusinessItemPayload> accBusinessItemPayloads, AccBusinessItemPayload accBusinessItemPayload) {
        Optional<AccBusinessItemPayload> optional = accBusinessItemPayloads.stream().filter(masPayload -> masPayload.getBusinessCode().equals(accBusinessItemPayload.getParentCode())).findFirst();
        if (optional.isPresent()) {
            AccBusinessItemPayload payload = optional.get();
            if (payload.getBusinessLevel() == null) {
                setLevel(accBusinessItemPayloads, payload);
            }
            accBusinessItemPayload.setBusinessLevel(payload.getBusinessLevel() + 1);
        } else {
            accBusinessItemPayload.setBusinessLevel(1);
        }
    }

}
