package com.elitesland.tw.tw5.server.prd.partner.team.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitesland.tw.tw5.api.prd.partner.identity.service.BusinessCustomerInfoService;
import com.elitesland.tw.tw5.api.prd.partner.team.payload.BusinessTeamMemberPayload;
import com.elitesland.tw.tw5.api.prd.partner.team.query.BusinessTeamMemberQuery;
import com.elitesland.tw.tw5.api.prd.partner.team.service.BusinessTeamMemberService;
import com.elitesland.tw.tw5.api.prd.partner.team.vo.BusinessTeamMemberVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemLogService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.server.common.TwException;

import com.elitesland.tw.tw5.server.prd.partner.constants.BusinessSelectionEnum;
import com.elitesland.tw.tw5.server.prd.partner.constants.BusinessTeamRoleEnum;
import com.elitesland.tw.tw5.server.prd.partner.constants.BusinessTeamTypeEnum;
import com.elitesland.tw.tw5.server.prd.partner.team.convert.BusinessTeamMemberConvert;
import com.elitesland.tw.tw5.server.prd.partner.team.dao.BusinessTeamMemberDAO;
import com.elitesland.tw.tw5.server.prd.partner.team.entity.BusinessTeamMemberDO;
import com.elitesland.tw.tw5.server.prd.partner.team.repo.BusinessTeamMemberRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemObjectEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * 业务伙伴-团队成员
 *
 * @author wangly
 * @date 2023-05-16
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class BusinessTeamMemberServiceImpl extends BaseServiceImpl implements BusinessTeamMemberService {

    private final BusinessTeamMemberRepo businessTeamMemberRepo;
    private final BusinessTeamMemberDAO businessTeamMemberDAO;
    private final PrdSystemLogService logService;
    private final CacheUtil cacheUtil;
    private final BusinessCustomerInfoService businessCustomerInfoService;

    @Override
    public PagingVO<BusinessTeamMemberVO> queryPaging(BusinessTeamMemberQuery query) {
        PagingVO<BusinessTeamMemberVO> businessTeamMemberVOPagingVO = businessTeamMemberDAO.queryPaging(query);
        businessTeamMemberVOPagingVO.stream().forEach(vo -> vo.setMemberRoleName(traslate(vo.getMemberRole(), vo.getType())));
        return businessTeamMemberVOPagingVO;
    }

    @Override
    public List<BusinessTeamMemberVO> queryListDynamic(BusinessTeamMemberQuery query) {
        List<BusinessTeamMemberVO> businessTeamMemberVOS = businessTeamMemberDAO.queryListDynamic(query);
        businessTeamMemberVOS.stream().forEach(vo -> vo.setMemberRoleName(traslate(vo.getMemberRole(), vo.getType())));
        return businessTeamMemberVOS;
    }

    @Override
    public BusinessTeamMemberVO queryByKey(Long key) {
        BusinessTeamMemberDO entity = businessTeamMemberRepo.findById(key).orElseGet(BusinessTeamMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");
        entity.setMemberRoleName(traslate(entity.getMemberRole(), entity.getType()));
        BusinessTeamMemberVO vo = BusinessTeamMemberConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<BusinessTeamMemberVO> insert(List<BusinessTeamMemberPayload> payload) {
        Long objectId = null;
        if (payload != null && payload.size() > 0) {
            objectId = payload.get(0).getObjectId();
            businessTeamMemberDAO.deleteByObjectIdAndNoCreator(objectId);
        }
        //因为是全删全插 所以插入的时候需要重新设置创建者的序号
        Set<Long> userIdset = new HashSet<>();
        Set<String> memberRoleSet = new HashSet();
        int memberRoleCount = 0;
        StringBuilder createUserId = new StringBuilder();
        for (BusinessTeamMemberPayload businessTeamMemberPayload : payload) {
            if (businessTeamMemberPayload.getIsCreator() != null && businessTeamMemberPayload.getIsCreator()) {
                businessTeamMemberPayload.setSortNo(1);
                createUserId.append(businessTeamMemberPayload.getUserId().toString());
            } else {
                userIdset.add(businessTeamMemberPayload.getUserId());
            }
            if (!BusinessTeamRoleEnum.CREATOR.getCode().equals(businessTeamMemberPayload.getMemberRole())) {
                memberRoleSet.add(businessTeamMemberPayload.getMemberRole());
            }
        }
        //人只能选择一次 如果说人数比角色数量多 说明角色重复了
        if (!CollectionUtils.isEmpty(userIdset) && userIdset.size() >memberRoleSet.size() ) {
            throw TwException.error("", "请勿添加相同的角色");
        }
        //如果说创建人重复添加了 提示他重新添加
        if (!StringUtils.isEmpty(createUserId) && userIdset.contains(Long.valueOf(createUserId.toString()))) {
            throw TwException.error("", "创建人员添加重复，请重新添加");
        }
        List<BusinessTeamMemberDO> entityDo = BusinessTeamMemberConvert.INSTANCE.toDoList(payload);
        if (objectId != null) {
            if (BusinessTeamTypeEnum.BID.getCode().equals(payload.get(0).getType())) {
                logService.saveNewLog(objectId, PrdSystemObjectEnum.BID_PROJECT.getCode(), "修改了团队成员");
            }
            if (BusinessTeamTypeEnum.PARTNER.getCode().equals(payload.get(0).getType())) {
                logService.saveNewLog(objectId, PrdSystemObjectEnum.BUSINESS_PARTNER.getCode(), "修改了团队成员");
            }
        }
        if (BusinessTeamTypeEnum.PARTNER.getCode().equals(payload.get(0).getType())) {
            //更新业务伙伴客户的更新时间 方便数据同步到4.0
            businessCustomerInfoService.updateBusinessParnerCustomerModifyTime(objectId);
        }
        return BusinessTeamMemberConvert.INSTANCE.toVoList(businessTeamMemberRepo.saveAll(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BusinessTeamMemberVO update(BusinessTeamMemberPayload payload) {
        BusinessTeamMemberDO entity = businessTeamMemberRepo.findById(payload.getId()).orElseGet(BusinessTeamMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");
        BusinessTeamMemberDO entityDo = BusinessTeamMemberConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        if (BusinessTeamTypeEnum.PARTNER.getCode().equals(entity.getType())) {
            //更新业务伙伴客户的更新时间 方便数据同步到4.0
            businessCustomerInfoService.updateBusinessParnerCustomerModifyTime(entity.getObjectId());
        }
        return BusinessTeamMemberConvert.INSTANCE.toVo(businessTeamMemberRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys, Long objectId) {
        //todo 删除关联的一系列表
        if (!keys.isEmpty()) {
            businessTeamMemberDAO.deleteSoft(keys);
            if (objectId != null) {
                // 目前是单个删 先这样写
                BusinessTeamMemberDO businessTeamMemberDO = businessTeamMemberRepo.findById(keys.get(0)).orElseGet(BusinessTeamMemberDO::new);
                if (BusinessTeamTypeEnum.BID.getCode().equals(businessTeamMemberDO.getType())) {
                    logService.saveNewLog(objectId, PrdSystemObjectEnum.BID_PROJECT.getCode(), "修改了团队成员");
                }
                if (BusinessTeamTypeEnum.PARTNER.getCode().equals(businessTeamMemberDO.getType())) {
                    logService.saveNewLog(objectId, PrdSystemObjectEnum.BUSINESS_PARTNER.getCode(), "修改了团队成员");
                    //更新业务伙伴客户的更新时间 方便数据同步到4.0
                    businessCustomerInfoService.updateBusinessParnerCustomerModifyTime(objectId);


                }

            }
        }
    }

    @Override
    public BusinessTeamMemberVO save(BusinessTeamMemberPayload payload) {
        BusinessTeamMemberDO entityDo = BusinessTeamMemberConvert.INSTANCE.toDo(payload);
        if (BusinessTeamTypeEnum.PARTNER.getCode().equals(entityDo.getType())) {
            //更新业务伙伴客户的更新时间 方便数据同步到4.0
            businessCustomerInfoService.updateBusinessParnerCustomerModifyTime(entityDo.getObjectId());
        }
        return BusinessTeamMemberConvert.INSTANCE.toVo(businessTeamMemberRepo.save(entityDo));
    }

    //调用的地方会自动更新业务伙伴/客户的更新时间 因此这里不用更新更新时间
    @Override
    public List<BusinessTeamMemberVO> saveAll(List<BusinessTeamMemberPayload> payloadList) {
        List<BusinessTeamMemberDO> entityDoList = BusinessTeamMemberConvert.INSTANCE.toDoList(payloadList);
        return BusinessTeamMemberConvert.INSTANCE.toVoList(businessTeamMemberRepo.saveAll(entityDoList));
    }

    @Override
    public List<BusinessTeamMemberVO> queryByObjectIdAndRole(Long objectId, String role) {
        return businessTeamMemberDAO.queryByObjectIdAndRole(objectId, role);
    }

    @Override
    public List<BusinessTeamMemberVO> queryByObjectIdsAndRole(List<Long> objectIds,String role){
        return businessTeamMemberDAO.queryByObjectIdsAndRole(objectIds,role);
    }

    @Override
    public List<BusinessTeamMemberVO> queryByObjectId(Long objectId) {
        return businessTeamMemberDAO.queryByObjectId(objectId);
    }

    @Override
    public Long count(BusinessTeamMemberQuery query) {
        return businessTeamMemberDAO.count(query);
    }

    //目前身份不能改变 因此这里不用更新更新时间
    @Override
    public void updateTeamByIdentityChange(Long objectId, String identity) {
        String selectKey = BusinessSelectionEnum.valueOf(identity).getCode();
        PrdSystemSelectionVO systemSelection = cacheUtil.getSystemSelection(selectKey);
        List<String> roleList = Arrays.asList(systemSelection.getExtString1().split(","));
        businessTeamMemberDAO.updateTeamByIdentityChange(objectId, roleList);
    }

    //根据角色的code和团队成员的type翻译为角色描述
    public String traslate(String roleCode, String type) {
        if (BusinessTeamTypeEnum.PARTNER.getCode().equals(type)) {
            return BusinessTeamRoleEnum.valueOf(roleCode).getDesc();
        }
        if (BusinessTeamTypeEnum.BID.getCode().equals(type)) {
            return BusinessTeamRoleEnum.valueOf(roleCode).getDesc();
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByObjectId(Long objectId) {
        businessTeamMemberRepo.deleteByObjectId(objectId);
        //更新业务伙伴客户的更新时间 方便数据同步到4.0
        businessCustomerInfoService.updateBusinessParnerCustomerModifyTime(objectId);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteTeamMember(Long objectId, String type, List<String> memberRoles) {
        businessTeamMemberDAO.deleteTeamMember(objectId, type, memberRoles);
    }
}
