package com.elitesland.tw.tw5.server.prd.pms.controller;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmCustomerOperationVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.permission.annotation.FunctionDetail;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionDomain;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionFunction;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionCodeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionTypeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionDomainEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionTypeEnum;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;

/**
 * 项目
 *
 * @author carl
 * @date 2023-08-07
 */
@Api(tags = "项目")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/pms/pmsProject")
@Slf4j
@PermissionDomain(domain = PermissionDomainEnum.PMS_PROJECT)
public class PmsProjectController {

    private final PmsProjectService pmsProjectService;

    /**
     * 项目-新增或全局修改
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping("insertOrUpdate")
    @ApiOperation("项目-新增或全局修改")
    public TwOutputUtil<PmsProjectVO> insertOrUpdate(@RequestBody PmsProjectPayload payload) {
        return TwOutputUtil.ok(pmsProjectService.insertOrUpdate(payload));
    }


    /**
     * 项目-更新-单字段,支持置空,置空的字段需需要借助 protected List<String> nullFields;
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("updateByKeyDynamic")
    @ApiOperation("项目-更新-单字段,支持置空 置空的字段需需要借助 protected List<String> nullFields;")
    public TwOutputUtil<Long> updateByKeyDynamic(@RequestBody PmsProjectPayload payload) {
        return TwOutputUtil.ok(pmsProjectService.updateByKeyDynamic(payload));
    }


    /**
     * 项目-变更
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("/updateChange")
    @ApiOperation("项目-变更")
    public TwOutputUtil<PmsProjectVO> updateChange(@RequestBody PmsProjectPayload payload) {
        return TwOutputUtil.ok(pmsProjectService.updateChange(payload));
    }

    /**
     * 项目-主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/byKey/{key}")
    @UdcNameClass
    @ApiOperation("项目-主键查询")
    public TwOutputUtil<PmsProjectVO> queryOneByKey(@PathVariable Long key) {
        return TwOutputUtil.ok(pmsProjectService.queryByKey(key));
    }

    /**
     * 项目-根据合同主键查询
     *
     * @param contractId 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryByContractId")
    @UdcNameClass
    @ApiOperation("项目-根据合同主键查询")
    public TwOutputUtil<PmsProjectVO> queryByContractId(Long contractId) {
        return TwOutputUtil.ok(pmsProjectService.queryByContractId(contractId));
    }

    /**
     * 项目-通过主键查询客户经营信息
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryProjectCustomerOperation/{key}")
    @UdcNameClass
    @ApiOperation("项目-通过主键查询客户经营信息")
    public TwOutputUtil<CrmCustomerOperationVO> queryProjectCustomerOperation(@PathVariable Long key) {
        return TwOutputUtil.ok(pmsProjectService.queryProjectCustomerOperation(key));
    }

    /**
     * 项目-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/paging")
    @UdcNameClass
    @ApiOperation("项目-分页")
    @PermissionFunction(functions = {
            @FunctionDetail(functionCode = FunctionCodeEnum.PROJECT_PAGE, type = FunctionTypeEnum.TAB)
    }, fieldPermission = true, permissionType = PermissionTypeEnum.DATA_PERMISSION)
    public TwOutputUtil<PagingVO<PmsProjectVO>> paging(@RequestBody PmsProjectQuery query) {
        return TwOutputUtil.ok(pmsProjectService.queryPaging(query));
    }


    /**
     * 项目-查询列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/list")
    @UdcNameClass
    @ApiOperation("项目-查询列表")
    public TwOutputUtil<List<PmsProjectVO>> queryList(PmsProjectQuery query) {
        return TwOutputUtil.ok(pmsProjectService.queryListDynamic(query));
    }

    /**
     * 项目-删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping("/deleteSoft")
    @ApiOperation("项目-删除")
    public TwOutputUtil deleteSoft(Long[] keys) {
        pmsProjectService.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 项目-查询汇报代办
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryProjectBriefWork")
    // @UdcNameClass
    @ApiOperation("项目-查询汇报代办")
    public TwOutputUtil<List<PmsProjectVO>> queryProjectBriefWork() {
        return TwOutputUtil.ok(pmsProjectService.queryProjectBriefWork(null));
    }

    /**
     * 项目-查询项目预计当量&费用待更新代办
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryProjectUpdateWork")
    @UdcNameClass
    @ApiOperation("项目-项目预计当量&费用待更新代办")
    public TwOutputUtil<List<PmsProjectVO>> queryProjectUpdateWork() {
        return TwOutputUtil.ok(pmsProjectService.queryProjectUpdateWork());
    }

    /**
     * 项目预算需变更代办
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/queryProjectUpdateBudgetWork")
    // @UdcNameClass
    @ApiOperation("项目预算需变更代办")
    public TwOutputUtil<List<PmsProjectVO>> queryProjectUpdateBudgetWork() {
        return TwOutputUtil.ok(pmsProjectService.queryProjectUpdateBudgetWork());
    }

    /**
     * 项目-无合同入场虚拟合同激活后发起对项目的数据变更
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping("updateNoContractData")
    @ApiOperation("项目-无合同入场虚拟合同激活后发起对项目的数据变更")
    public TwOutputUtil updateNoContractData(@RequestBody PmsProjectPayload payload) {
        pmsProjectService.updateNoContractData(payload);
        return TwOutputUtil.ok();
    }


    @GetMapping("/queryListByProjectMember")
    @UdcNameClass
    @ApiOperation("根据项目成员查询所有项目")
    public TwOutputUtil<List<PmsProjectVO>> queryListByProjectMember(Long userId, Boolean selectAll) {
        return TwOutputUtil.ok(pmsProjectService.queryListByProjectMember(userId, selectAll));
    }

    /**
     * 初始化历史激活项目结算单
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/initWideSettle")
    @ApiOperation("初始化历史激活项目结算单")
    public TwOutputUtil initWideSettle() {
        pmsProjectService.initWideSettle();
        return TwOutputUtil.ok();
    }

    /**
     * 修改项目状态
     *
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/updateProjectStatus/{key}")
    @ApiOperation("修改项目状态")
    @PermissionFunction(functions = {
            @FunctionDetail(functionCode = FunctionCodeEnum.PROJECT_CHANGE_STATUS, type = FunctionTypeEnum.BUTTON)
    })
    public TwOutputUtil<List<PmsProjectVO>> updateProjectStatus(@PathVariable Long key) {
        pmsProjectService.updateProjectStatus(key);
        return TwOutputUtil.ok();
    }

    /**
     * 批量导出excel
     *
     * @return result
     */
    @ApiOperation("批量导出excel")
    @PostMapping("/batchExport")
    @PermissionFunction(functions = {
            @FunctionDetail(functionCode = FunctionCodeEnum.PROJECT_PAGE, type = FunctionTypeEnum.BUTTON)
    }, fieldPermission = true, permissionType = PermissionTypeEnum.DATA_PERMISSION)
    public void batchExport(HttpServletResponse response, @RequestBody PmsProjectQuery query) {
        pmsProjectService.downloadBatch(response, query);
    }

    /**
     * 预实对照表
     *
     * @return result
     */
    @GetMapping("/queryPlanAndActual/{key}")
    @ApiOperation("查询项目预实对照表")
    public TwOutputUtil queryPlanAndActual(@PathVariable Long key) {
        return TwOutputUtil.ok(pmsProjectService.queryPlanAndActual(key));
    }
}
