package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchaseAgreementPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseAgreementQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseAgreementVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PurchaseAgreementDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseAgreementDO;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchaseAgreementEnum;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PurchaseAgreementRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * 采购协议
 *
 * @author Echo
 * @date 2023-06-08
 */
@Repository
@RequiredArgsConstructor
public class PurchaseAgreementDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PurchaseAgreementRepo repo;
    private final QPurchaseAgreementDO qdo = QPurchaseAgreementDO.purchaseAgreementDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PurchaseAgreementVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 采购协议编号
                qdo.purchaseAgreementNo,
                // 采购协议名称
                qdo.purchaseAgreementName,
                // 协议类型 udc[purchase:agreement_type]
                qdo.agreementType,
                // 验收方式 udc[purchase:check]
                qdo.acceptanceType,
                // 签约日期
                qdo.signDate,
                // 申请日期
                qdo.applicationDate,
                // 签约公司
                qdo.signCompanyBookId,
                // 签约BU
                qdo.signBuId,
                // 签约负责人
                qdo.signInchargeResId,
                // 供应商地址簿ID
                qdo.supplierBookId,
                // 币种 udc[SYSTEM_BASIC:CURRENCY]
                qdo.currCode,
                // 金额
                qdo.amt,
                // 税率
                qdo.taxRate,
                // 税额
                qdo.taxAmt,
                // 协议内容说明
                qdo.agreementContentDescription,
                // 比价资料
                qdo.pricecompereFileCodes,
                // 协议附件
                qdo.agreementFileCodes,
                // 盖章附件
                qdo.stampFileCodes,
                // 开票方
                qdo.invoice,
                // 付款方式
                qdo.payMethod,
                // 协议状态
                qdo.agreementStatus,
                // 协议归属编号
                qdo.agreementOwnershipNo,
                // 激活时间
                qdo.activateDate,
                // 前置单据负责人
                qdo.preReceiptResId,
                // 终止原因
                qdo.overReason,
                // 终止日期
                qdo.overDate,
                // 有效开始日期
                qdo.effectiveStartDate,
                // 有效结束日期
                qdo.effectiveEndDate,
                qdo.procInstId,
                qdo.procInstStatus,
                qdo.submitTime,
                qdo.approvedTime,

                // 收款人/单位BookId
                qdo.supplierBookId.as("receivingUnitBookId")
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementVO> getJpaQueryWhere(PurchaseAgreementQuery query) {
        JPAQuery<PurchaseAgreementVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 根据创建时间倒序排序，最新创建的显示在最上面
        jpaQuery.orderBy(qdo.createTime.desc());
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchaseAgreementQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchaseAgreementQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        if (!CollectionUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 采购协议编号 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseAgreementNo())) {
            list.add(qdo.purchaseAgreementNo.eq(query.getPurchaseAgreementNo()));
        }
        if (!ObjectUtils.isEmpty(query.getPurchaseAgreementNmNo())) {
            list.add(qdo.purchaseAgreementNo.like(SqlUtil.toSqlLikeString(query.getPurchaseAgreementNmNo())).or(qdo.purchaseAgreementName.like(SqlUtil.toSqlLikeString(query.getPurchaseAgreementNmNo()))));
        }
        /** 采购协议名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getPurchaseAgreementName())) {
            list.add(qdo.purchaseAgreementName.like(SqlUtil.toSqlLikeString(query.getPurchaseAgreementName())));
        }
        /** 采购协议名称 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseAgreementNameEq())) {
            list.add(qdo.purchaseAgreementName.eq(query.getPurchaseAgreementNameEq()));
        }
        /** 协议类型 udc[purchase:agreement_type] 精确 */
        if (!ObjectUtils.isEmpty(query.getAgreementType())) {
            list.add(qdo.agreementType.eq(query.getAgreementType()));
        }
        /** 验收方式 udc[purchase:check] 精确 */
        if (!ObjectUtils.isEmpty(query.getAcceptanceType())) {
            list.add(qdo.acceptanceType.eq(query.getAcceptanceType()));
        }
        /** 签约日期 精确 */
        if (!ObjectUtils.isEmpty(query.getSignDate())) {
            list.add(qdo.signDate.eq(query.getSignDate()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplicationDate())) {
            list.add(qdo.applicationDate.eq(query.getApplicationDate()));
        }
        /** 签约公司地址簿Id 精确 */
        if (!ObjectUtils.isEmpty(query.getSignCompanyBookId())) {
            list.add(qdo.signCompanyBookId.eq(query.getSignCompanyBookId()));
        }
        /** 签约BU 精确 */
        if (!ObjectUtils.isEmpty(query.getSignBuId())) {
            list.add(qdo.signBuId.eq(query.getSignBuId()));
        }
        /** 签约负责人 精确 */
        if (!ObjectUtils.isEmpty(query.getSignInchargeResId())) {
            list.add(qdo.signInchargeResId.eq(query.getSignInchargeResId()));
        }
        /** 供应商地址簿Id 精确 */
        if (!ObjectUtils.isEmpty(query.getSupplierBookId())) {
            list.add(qdo.supplierBookId.eq(query.getSupplierBookId()));
        }
        /** 币种 udc[SYSTEM_BASIC:CURRENCY] 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrCode())) {
            list.add(qdo.currCode.eq(query.getCurrCode()));
        }
        /** 金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAmt())) {
            list.add(qdo.amt.eq(query.getAmt()));
        }
        /** 税率 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxRate())) {
            list.add(qdo.taxRate.eq(query.getTaxRate()));
        }
        /** 税额 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxAmt())) {
            list.add(qdo.taxAmt.eq(query.getTaxAmt()));
        }
        /** 协议内容说明 精确 */
        if (!ObjectUtils.isEmpty(query.getAgreementContentDescription())) {
            list.add(qdo.agreementContentDescription.eq(query.getAgreementContentDescription()));
        }
        /** 比价资料 精确 */
        if (!ObjectUtils.isEmpty(query.getPricecompereFileCodes())) {
            list.add(qdo.pricecompereFileCodes.eq(query.getPricecompereFileCodes()));
        }
        /** 协议附件 精确 */
        if (!ObjectUtils.isEmpty(query.getAgreementFileCodes())) {
            list.add(qdo.agreementFileCodes.eq(query.getAgreementFileCodes()));
        }
        /** 盖章附件 精确 */
        if (!ObjectUtils.isEmpty(query.getStampFileCodes())) {
            list.add(qdo.stampFileCodes.eq(query.getStampFileCodes()));
        }
        /** 开票方 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoice())) {
            list.add(qdo.invoice.eq(query.getInvoice()));
        }
        /** 付款方式 精确 */
        if (!ObjectUtils.isEmpty(query.getPayMethod())) {
            list.add(qdo.payMethod.eq(query.getPayMethod()));
        }
        /** 协议状态 精确 */
        if (!ObjectUtils.isEmpty(query.getAgreementStatus())) {
            list.add(qdo.agreementStatus.eq(query.getAgreementStatus()));
        }
        /** 协议归属编号 精确 */
        if (!ObjectUtils.isEmpty(query.getAgreementOwnershipNo())) {
            list.add(qdo.agreementOwnershipNo.eq(query.getAgreementOwnershipNo()));
        }
        /** 激活时间 精确 */
        if (!ObjectUtils.isEmpty(query.getActivateDate())) {
            list.add(qdo.activateDate.eq(query.getActivateDate()));
        }
        /** 前置单据负责人 精确 */
        if (!ObjectUtils.isEmpty(query.getPreReceiptResId())) {
            list.add(qdo.preReceiptResId.eq(query.getPreReceiptResId()));
        }
        /** 终止原因 精确 */
        if (!ObjectUtils.isEmpty(query.getOverReason())) {
            list.add(qdo.overReason.eq(query.getOverReason()));
        }
        /** 终止日期 精确 */
        if (!ObjectUtils.isEmpty(query.getOverDate())) {
            list.add(qdo.overDate.eq(query.getOverDate()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchaseAgreementVO queryByKey(Long id) {
        JPAQuery<PurchaseAgreementVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchaseAgreementVO> queryListDynamic(PurchaseAgreementQuery query) {
        JPAQuery<PurchaseAgreementVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchaseAgreementVO> queryPaging(PurchaseAgreementQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PurchaseAgreementVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchaseAgreementVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchaseAgreementVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PurchaseAgreementDO save(PurchaseAgreementDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PurchaseAgreementDO> saveAll(List<PurchaseAgreementDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PurchaseAgreementPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 采购协议编号
        if (payload.getPurchaseAgreementNo() != null) {
            update.set(qdo.purchaseAgreementNo, payload.getPurchaseAgreementNo());
        }
        // 采购协议名称
        if (payload.getPurchaseAgreementName() != null) {
            update.set(qdo.purchaseAgreementName, payload.getPurchaseAgreementName());
        }
        // 协议类型 udc[purchase:agreement_type]
        if (payload.getAgreementType() != null) {
            update.set(qdo.agreementType, payload.getAgreementType());
        }
        // 验收方式 udc[purchase:check]
        if (payload.getAcceptanceType() != null) {
            update.set(qdo.acceptanceType, payload.getAcceptanceType());
        }
        // 签约日期
        if (payload.getSignDate() != null) {
            update.set(qdo.signDate, payload.getSignDate());
        }
        // 申请日期
        if (payload.getApplicationDate() != null) {
            update.set(qdo.applicationDate, payload.getApplicationDate());
        }
        // 签约公司地址簿Id
        if (payload.getSignCompanyBookId() != null) {
            update.set(qdo.signCompanyBookId, payload.getSignCompanyBookId());
        }
        // 签约BU
        if (payload.getSignBuId() != null) {
            update.set(qdo.signBuId, payload.getSignBuId());
        }
        // 签约负责人
        if (payload.getSignInchargeResId() != null) {
            update.set(qdo.signInchargeResId, payload.getSignInchargeResId());
        }
        // 供应商地址簿Id
        if (payload.getSupplierBookId() != null) {
            update.set(qdo.supplierBookId, payload.getSupplierBookId());
        }
        // 币种 udc[SYSTEM_BASIC:CURRENCY]
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 金额
        if (payload.getAmt() != null) {
            update.set(qdo.amt, payload.getAmt());
        }
        // 税率
        if (payload.getTaxRate() != null) {
            update.set(qdo.taxRate, payload.getTaxRate());
        }
        // 税额
        if (payload.getTaxAmt() != null) {
            update.set(qdo.taxAmt, payload.getTaxAmt());
        }
        // 协议内容说明
        if (payload.getAgreementContentDescription() != null) {
            update.set(qdo.agreementContentDescription, payload.getAgreementContentDescription());
        }
        // 比价资料
        if (payload.getPricecompereFileCodes() != null) {
            update.set(qdo.pricecompereFileCodes, payload.getPricecompereFileCodes());
        }
        // 协议附件
        if (payload.getAgreementFileCodes() != null) {
            update.set(qdo.agreementFileCodes, payload.getAgreementFileCodes());
        }
        // 盖章附件
        if (payload.getStampFileCodes() != null) {
            update.set(qdo.stampFileCodes, payload.getStampFileCodes());
        }
        // 开票方
        if (payload.getInvoice() != null) {
            update.set(qdo.invoice, payload.getInvoice());
        }
        // 付款方式
        if (payload.getPayMethod() != null) {
            update.set(qdo.payMethod, payload.getPayMethod());
        }
        // 协议状态
        if (payload.getAgreementStatus() != null) {
            update.set(qdo.agreementStatus, payload.getAgreementStatus());
        }
        // 协议归属编号
        if (payload.getAgreementOwnershipNo() != null) {
            update.set(qdo.agreementOwnershipNo, payload.getAgreementOwnershipNo());
        }
        // 激活时间
        if (payload.getActivateDate() != null) {
            update.set(qdo.activateDate, payload.getActivateDate());
        }
        // 前置单据负责人
        if (payload.getPreReceiptResId() != null) {
            update.set(qdo.preReceiptResId, payload.getPreReceiptResId());
        }
        // 终止原因
        if (payload.getOverReason() != null) {
            update.set(qdo.overReason, payload.getOverReason());
        }
        // 终止日期
        if (payload.getOverDate() != null) {
            update.set(qdo.overDate, payload.getOverDate());
        }
        // 有效开始日期
        if (payload.getEffectiveStartDate() != null) {
            update.set(qdo.effectiveStartDate, payload.getEffectiveStartDate());
        }
        // 有效开始日期
        if (payload.getEffectiveEndDate() != null) {
            update.set(qdo.effectiveEndDate, payload.getEffectiveEndDate());
        }
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 采购协议编号
            if (nullFields.contains("purchaseAgreementNo")) {
                update.setNull(qdo.purchaseAgreementNo);
            }
            // 采购协议名称
            if (nullFields.contains("purchaseAgreementName")) {
                update.setNull(qdo.purchaseAgreementName);
            }
            // 协议类型 udc[purchase:agreement_type]
            if (nullFields.contains("agreementType")) {
                update.setNull(qdo.agreementType);
            }
            // 验收方式 udc[purchase:check]
            if (nullFields.contains("acceptanceType")) {
                update.setNull(qdo.acceptanceType);
            }
            // 签约日期
            if (nullFields.contains("signDate")) {
                update.setNull(qdo.signDate);
            }
            // 申请日期
            if (nullFields.contains("applicationDate")) {
                update.setNull(qdo.applicationDate);
            }
            // 签约公司地址簿Id
            if (nullFields.contains("signCompanyBookId")) {
                update.setNull(qdo.signCompanyBookId);
            }
            // 签约BU
            if (nullFields.contains("signBuId")) {
                update.setNull(qdo.signBuId);
            }
            // 签约负责人
            if (nullFields.contains("signInchargeResId")) {
                update.setNull(qdo.signInchargeResId);
            }
            // 供应商地址簿Id
            if (nullFields.contains("supplierBookId")) {
                update.setNull(qdo.supplierBookId);
            }
            // 币种 udc[SYSTEM_BASIC:CURRENCY]
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 金额
            if (nullFields.contains("amt")) {
                update.setNull(qdo.amt);
            }
            // 税率
            if (nullFields.contains("taxRate")) {
                update.setNull(qdo.taxRate);
            }
            // 税额
            if (nullFields.contains("taxAmt")) {
                update.setNull(qdo.taxAmt);
            }
            // 协议内容说明
            if (nullFields.contains("agreementContentDescription")) {
                update.setNull(qdo.agreementContentDescription);
            }
            // 比价资料
            if (nullFields.contains("pricecompereFileCodes")) {
                update.setNull(qdo.pricecompereFileCodes);
            }
            // 协议附件
            if (nullFields.contains("agreementFileCodes")) {
                update.setNull(qdo.agreementFileCodes);
            }
            // 盖章附件
            if (nullFields.contains("stampFileCodes")) {
                update.setNull(qdo.stampFileCodes);
            }
            // 开票方
            if (nullFields.contains("invoice")) {
                update.setNull(qdo.invoice);
            }
            // 付款方式
            if (nullFields.contains("payMethod")) {
                update.setNull(qdo.payMethod);
            }
            // 协议状态
            if (nullFields.contains("agreementStatus")) {
                update.setNull(qdo.agreementStatus);
            }
            // 协议归属编号
            if (nullFields.contains("agreementOwnershipNo")) {
                update.setNull(qdo.agreementOwnershipNo);
            }
            // 激活时间
            if (nullFields.contains("activateDate")) {
                update.setNull(qdo.activateDate);
            }
            // 前置单据负责人
            if (nullFields.contains("preReceiptResId")) {
                update.setNull(qdo.preReceiptResId);
            }
            // 终止原因
            if (nullFields.contains("overReason")) {
                update.setNull(qdo.overReason);
            }
            // 终止日期
            if (nullFields.contains("overDate")) {
                update.setNull(qdo.overDate);
            }
            if (nullFields.contains("effectiveStartDate")) {
                update.setNull(qdo.effectiveStartDate);
            }
            if (nullFields.contains("effectiveEndDate")) {
                update.setNull(qdo.effectiveEndDate);
            }
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            if (nullFields.contains("submitTime")) {
                update.setNull(qdo.submitTime);
            }
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
            if (nullFields.contains("deleteFlag")) {
                update.setNull(qdo.deleteFlag);
            }
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long updateStatusByIds(List<Long> keys, String status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.agreementStatus, status)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long purAgreementBreakByIds(List<Long> keys, String overReason) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.overReason, overReason)
                .set(qdo.agreementStatus, PurchaseAgreementEnum.AgreementStatus.BREAK.getCode())
                .set(qdo.overDate, LocalDateTime.now())
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 查询协议名称（是否重复）
     *
     * @param payload
     * @return
     */
    public boolean queryByName(PurchaseAgreementPayload payload) {
        JPAQuery<PurchaseAgreementVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.purchaseAgreementName.eq(payload.getPurchaseAgreementName()));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        PurchaseAgreementVO purchaseAgreementVO = jpaQuery.fetchOne();
        return purchaseAgreementVO != null;
    }

    /**
     * 根据编号查询采购协议
     * @param docNo 采购协议编号
     * @return 采购协议
     */
    public PurchaseAgreementVO findByPurchaseAgreementNo(String docNo) {
        JPAQuery<PurchaseAgreementVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.deleteFlag.eq(0)
                .and(qdo.purchaseAgreementNo.eq(docNo)));
        return jpaQuery.fetchFirst();
    }

    /**
     * 通过协议编号查询协议Id
     * @param docNo 协议编号
     * @return 协议Id
     */
    public Long findIdByNo(String docNo) {
        return jpaQueryFactory.select(qdo.id).from(qdo).where(qdo.deleteFlag.eq(0)
                .and(qdo.purchaseAgreementNo.eq(docNo))).fetchFirst();
    }

    /**
     * 通过协议编号查询协议详情
     * @param docNo 协议编号
     * @return 协议详情
     */
    public PurchaseAgreementVO queryByNo(String docNo) {
        return jpaQueryFactory.select(Projections.bean(PurchaseAgreementVO.class,
                        qdo.id,
                        qdo.amt,
                        qdo.signBuId,
                        // 采购协议编号
                        qdo.purchaseAgreementNo,
                        // 采购协议名称
                        qdo.purchaseAgreementName,
                        // 协议类型 udc[purchase:agreement_type]
                        qdo.agreementType,
                        // 验收方式 udc[purchase:check]
                        qdo.acceptanceType)).
                from(qdo).where(qdo.deleteFlag.eq(0)
                .and(qdo.purchaseAgreementNo.eq(docNo))).fetchFirst();
    }
}

