package com.elitesland.tw.tw5.server.prd.purchase.service;

import cn.hutool.extra.spring.SpringUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.bank.feign.payload.PayListPayload;
import com.elitesland.tw.tw5.api.bank.feign.payload.PayPayload;
import com.elitesland.tw.tw5.api.bank.feign.service.TwBankPayService;
import com.elitesland.tw.tw5.api.bank.vo.BkPayDetailVO;
import com.elitesland.tw.tw5.api.prd.humanresources.service.ResWithdrawApplyService;
import com.elitesland.tw.tw5.api.prd.partner.common.service.BookAccountService;
import com.elitesland.tw.tw5.api.prd.partner.common.service.BusinessPartnerService;
import com.elitesland.tw.tw5.api.prd.partner.common.vo.BookAccountVO;
import com.elitesland.tw.tw5.api.prd.pay.payload.BkPayDetailRecordPayload;
import com.elitesland.tw.tw5.api.prd.pay.payload.TDocHistPayload;
import com.elitesland.tw.tw5.api.prd.pay.service.BkPayDetailRecordService;
import com.elitesland.tw.tw5.api.prd.pay.service.TDocHistService;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PaymentSlipPayload;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchaseContractManagerPayload;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchasePaymentPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PaymentSlipQuery;
import com.elitesland.tw.tw5.api.prd.purchase.service.PaymentSlipService;
import com.elitesland.tw.tw5.api.prd.purchase.service.PurchaseAgreementService;
import com.elitesland.tw.tw5.api.prd.purchase.service.PurchaseContractManagerService;
import com.elitesland.tw.tw5.api.prd.purchase.service.PurchasePaymentService;
import com.elitesland.tw.tw5.api.prd.purchase.vo.DibPaymentSlipEmailVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PaymentSlipVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseContractManagerVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchasePaymentVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdMessageConfigService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdMessageConfigVO;
import com.elitesland.tw.tw5.server.common.GenerateSeqNumConstants;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.purchase.convert.PaymentSlipConvert;
import com.elitesland.tw.tw5.server.prd.purchase.dao.PaymentSlipDAO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PaymentSlipDO;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchaseContractStatusEnum;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchasePaymentEnum;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PaymentSlipRepo;
import com.querydsl.core.Tuple;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 付款单记录
 *
 * @author likunpeng
 * @date 2023-11-23
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PaymentSlipServiceImpl extends BaseServiceImpl implements PaymentSlipService {

    private final PaymentSlipRepo paymentSlipRepo;
    private final PaymentSlipDAO paymentSlipDAO;
    private final BusinessPartnerService businessPartnerService;
    @Autowired
    @Lazy
    private  PurchaseContractManagerService purchaseContractManagerService;
    private final PurchaseAgreementService purchaseAgreementService;
    @Resource
    private TwBankPayService payService;
    private final BookAccountService bookAccountService;
    private final BkPayDetailRecordService bkPayDetailRecordService;
    private final TDocHistService tDocHistService;
    private final CacheUtil cacheUtil;
    @Autowired
    @Lazy
    private PurchasePaymentService purchasePaymentService;

    @Value("${tw5.payment.platform:TW5}")
    private String platform;

    @Value("${tw5.payment.bank:CMB}")
    private String bank;

    @Value("${tw5.payment.secretKey:123456}")
    private String secretKey;

    private final ResWithdrawApplyService resWithdrawApplyService;
    private final PrdMessageConfigService messageConfigService;

    @Override
    public PagingVO<PaymentSlipVO> queryPaging(PaymentSlipQuery query) {
        PagingVO<PaymentSlipVO> paymentSlipVOPagingVO = paymentSlipDAO.queryPaging(query);
        if (!CollectionUtils.isEmpty(paymentSlipVOPagingVO.getRecords())) {
            List<Long> bookIdList = new ArrayList();
            List<Long> accountIdList = new ArrayList<>();
            Map<Long, String> resultMap = new HashMap<>();
            Map<Long, List<BookAccountVO>> accountMap = new HashMap<>();
            paymentSlipVOPagingVO.stream().forEach(p -> {
                if (p.getPayCompanyBookId() != null) {
                    bookIdList.add(p.getPayCompanyBookId());
                }
                if (p.getReceivingCompanyBookId() != null) {
                    bookIdList.add(p.getReceivingCompanyBookId());
                }
                if (StringUtils.hasText(p.getReceivingAccount())) {
                    accountIdList.add(Long.valueOf(p.getReceivingAccount()));
                }
                if (StringUtils.hasText(p.getPaymentAccount())) {
                    accountIdList.add(Long.valueOf(p.getPaymentAccount()));
                }
            });
            if (!CollectionUtils.isEmpty(bookIdList)) {
                resultMap = businessPartnerService.findNameByBookIds(bookIdList);
            }
            if (!CollectionUtils.isEmpty(accountIdList)) {
                accountMap = bookAccountService.queryByIdList(accountIdList);
            }

            for (PaymentSlipVO paymentSlipVO : paymentSlipVOPagingVO.getRecords()) {
                translate(paymentSlipVO, resultMap, accountMap);
            }
        }
        return paymentSlipVOPagingVO;
    }

    /**
     * 付款代记账记录表
     *
     * @param query 条件
     * @return
     */
    @Override
    public PagingVO<PaymentSlipVO> queryAccountPaging(PaymentSlipQuery query) {
        PagingVO<PaymentSlipVO> paymentSlipVOPagingVO = paymentSlipDAO.queryAccountPaging(query);
        if (!CollectionUtils.isEmpty(paymentSlipVOPagingVO.getRecords())) {
            List<Long> bookIdList = new ArrayList();
            List<Long> accountIdList = new ArrayList<>();
            Map<Long, String> resultMap = new HashMap<>();
            Map<Long, List<BookAccountVO>> accountMap = new HashMap<>();
            paymentSlipVOPagingVO.stream().forEach(p -> {
                if (p.getPayCompanyBookId() != null) {
                    bookIdList.add(p.getPayCompanyBookId());
                }
                if (p.getReceivingCompanyBookId() != null) {
                    bookIdList.add(p.getReceivingCompanyBookId());
                }
                if (StringUtils.hasText(p.getReceivingAccount())) {
                    accountIdList.add(Long.valueOf(p.getReceivingAccount()));
                }
                if (StringUtils.hasText(p.getPaymentAccount())) {
                    accountIdList.add(Long.valueOf(p.getPaymentAccount()));
                }
            });
            if (!CollectionUtils.isEmpty(bookIdList)) {
                resultMap = businessPartnerService.findNameByBookIds(bookIdList);
            }
            if (!CollectionUtils.isEmpty(accountIdList)) {
                accountMap = bookAccountService.queryByIdList(accountIdList);
            }

            for (PaymentSlipVO paymentSlipVO : paymentSlipVOPagingVO.getRecords()) {
                translate(paymentSlipVO, resultMap, accountMap);
            }
        }
        return paymentSlipVOPagingVO;
    }

    @Override
    public PagingVO<PaymentSlipVO> paymentApproveGroupPaging(PaymentSlipQuery query) {
        PagingVO<PaymentSlipVO> paymentSlipVOPagingVO = paymentSlipDAO.paymentApproveGroupPaging(query);
        if (!CollectionUtils.isEmpty(paymentSlipVOPagingVO.getRecords())) {
            List<Long> bookIdList = new ArrayList();
            List<Long> accountIdList = new ArrayList<>();
            Map<Long, String> resultMap = new HashMap<>();
            Map<Long, List<BookAccountVO>> accountMap = new HashMap<>();
            paymentSlipVOPagingVO.stream().forEach(p -> {
                if (p.getPayCompanyBookId() != null) {
                    bookIdList.add(p.getPayCompanyBookId());
                }
                if (p.getReceivingCompanyBookId() != null) {
                    bookIdList.add(p.getReceivingCompanyBookId());
                }
                if (StringUtils.hasText(p.getReceivingAccount())) {
                    accountIdList.add(Long.valueOf(p.getReceivingAccount()));
                }
                if (StringUtils.hasText(p.getPaymentAccount())) {
                    accountIdList.add(Long.valueOf(p.getPaymentAccount()));
                }

            });
            if (!CollectionUtils.isEmpty(bookIdList)) {
                resultMap = businessPartnerService.findNameByBookIds(bookIdList);
            }
            if (!CollectionUtils.isEmpty(accountIdList)) {
                accountMap = bookAccountService.queryByIdList(accountIdList);
            }
            for (PaymentSlipVO paymentSlipVO : paymentSlipVOPagingVO.getRecords()) {
                translate(paymentSlipVO, resultMap, accountMap);
            }
        }
        return paymentSlipVOPagingVO;
    }

    @Override
    public List<PaymentSlipVO> queryListDynamic(PaymentSlipQuery query) {
        List<PaymentSlipVO> paymentSlipVOS = paymentSlipDAO.queryListDynamic(query);
        if (!CollectionUtils.isEmpty(paymentSlipVOS)) {
            List<Long> bookIdList = new ArrayList();
            List<Long> accountIdList = new ArrayList<>();
            Map<Long, String> resultMap = new HashMap<>();
            Map<Long, List<BookAccountVO>> accountMap = new HashMap<>();
            paymentSlipVOS.forEach(p -> {
                if (p.getPayCompanyBookId() != null) {
                    bookIdList.add(p.getPayCompanyBookId());
                }
                if (p.getReceivingCompanyBookId() != null) {
                    bookIdList.add(p.getReceivingCompanyBookId());
                }
                if (StringUtils.hasText(p.getReceivingAccount())) {
                    accountIdList.add(Long.valueOf(p.getReceivingAccount()));
                }
                if (StringUtils.hasText(p.getPaymentAccount())) {
                    accountIdList.add(Long.valueOf(p.getPaymentAccount()));
                }
            });
            if (!CollectionUtils.isEmpty(bookIdList)) {
                resultMap = businessPartnerService.findNameByBookIds(bookIdList);
            }
            if (!CollectionUtils.isEmpty(accountIdList)) {
                accountMap = bookAccountService.queryByIdList(accountIdList);
            }
            for (PaymentSlipVO paymentSlipVO : paymentSlipVOS) {
                translate(paymentSlipVO, resultMap, accountMap);
            }
        }
        return paymentSlipVOS;
    }

    @Override
    public List<PaymentSlipVO> queryListByIds(List<Long> idList) {
        List<PaymentSlipVO> paymentSlipVOS = paymentSlipDAO.queryListByIds(idList);
        if (!CollectionUtils.isEmpty(paymentSlipVOS)) {
            List<Long> bookIdList = new ArrayList();
            List<Long> accountIdList = new ArrayList<>();
            Map<Long, String> resultMap = new HashMap<>();
            Map<Long, List<BookAccountVO>> accountMap = new HashMap<>();
            paymentSlipVOS.forEach(p -> {
                if (p.getPayCompanyBookId() != null) {
                    bookIdList.add(p.getPayCompanyBookId());
                }
                if (p.getReceivingCompanyBookId() != null) {
                    bookIdList.add(p.getReceivingCompanyBookId());
                }
                if (StringUtils.hasText(p.getReceivingAccount())) {
                    accountIdList.add(Long.valueOf(p.getReceivingAccount()));
                }
                if (StringUtils.hasText(p.getPaymentAccount())) {
                    accountIdList.add(Long.valueOf(p.getPaymentAccount()));
                }
            });
            if (!CollectionUtils.isEmpty(bookIdList)) {
                resultMap = businessPartnerService.findNameByBookIds(bookIdList);
            }
            if (!CollectionUtils.isEmpty(accountIdList)) {
                accountMap = bookAccountService.queryByIdList(accountIdList);
            }
            for (PaymentSlipVO paymentSlipVO : paymentSlipVOS) {
                translate(paymentSlipVO, resultMap, accountMap);
            }
        }
        return paymentSlipVOS;
    }

    @Override
    public PaymentSlipVO queryByKey(Long key) {
        PaymentSlipVO paymentSlipVO = paymentSlipDAO.queryByKey(key);
        if (paymentSlipVO == null) {
            throw TwException.error("", "付款记录对象不存在");
        }
        return paymentSlipVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PaymentSlipVO insert(PaymentSlipPayload payload) {
        PaymentSlipDO entityDo = PaymentSlipConvert.INSTANCE.toDo(payload);
        return PaymentSlipConvert.INSTANCE.toVo(paymentSlipRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PaymentSlipVO update(PaymentSlipPayload payload) {
        PaymentSlipDO entity = paymentSlipRepo.findById(payload.getId()).orElseGet(PaymentSlipDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PaymentSlipDO entityDo = PaymentSlipConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return PaymentSlipConvert.INSTANCE.toVo(paymentSlipRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(PaymentSlipPayload payload) {
        PaymentSlipDO entity = paymentSlipRepo.findById(payload.getId()).orElseGet(PaymentSlipDO::new);
        Assert.notNull(entity.getId(), "不存在");
        long result = paymentSlipDAO.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateBankCallBack(PaymentSlipPayload payload) {
        long result = paymentSlipDAO.updateBankCallBack(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            paymentSlipDAO.deleteSoft(keys);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void insertAll(List<PaymentSlipPayload> paymentSlipPayloads) {
        paymentSlipRepo.saveAll(PaymentSlipConvert.INSTANCE.toDoList(paymentSlipPayloads));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByPaymentApplyId(Long paymentApplyId) {
        paymentSlipDAO.deleteByPaymentApplyId(paymentApplyId);
    }

    @Override
    public List<PaymentSlipVO> queryListByPaymentApplyId(Long paymentApplyId) {
        List<PaymentSlipVO> paymentSlipVOS = paymentSlipDAO.queryListByPaymentApplyId(paymentApplyId);
        if (!CollectionUtils.isEmpty(paymentSlipVOS)) {
            List<Long> bookIdList = new ArrayList();
            List<Long> accountIdList = new ArrayList<>();
            Map<Long, String> resultMap = new HashMap<>();
            Map<Long, List<BookAccountVO>> accountMap = new HashMap<>();
            paymentSlipVOS.forEach(p -> {
                if (p.getPayCompanyBookId() != null) {
                    bookIdList.add(p.getPayCompanyBookId());
                }
                if (p.getReceivingCompanyBookId() != null) {
                    bookIdList.add(p.getReceivingCompanyBookId());
                }
                if (StringUtils.hasText(p.getReceivingAccount())) {
                    accountIdList.add(Long.valueOf(p.getReceivingAccount()));
                }
                if (StringUtils.hasText(p.getPaymentAccount())) {
                    accountIdList.add(Long.valueOf(p.getPaymentAccount()));
                }
            });
            if (!CollectionUtils.isEmpty(bookIdList)) {
                resultMap = businessPartnerService.findNameByBookIds(bookIdList);
            }
            if (!CollectionUtils.isEmpty(accountIdList)) {
                accountMap = bookAccountService.queryByIdList(accountIdList);
            }
            for (PaymentSlipVO paymentSlipVO : paymentSlipVOS) {
                translate(paymentSlipVO, resultMap, accountMap);
            }
        }
        return paymentSlipVOS;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateStatusByPaymentApplyId(Long paymentApplyId, String state) {
        paymentSlipDAO.updateStatusByPaymentApplyId(paymentApplyId, state);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void confirmOfflinePayment(List<PaymentSlipPayload> paymentSlipPayloadList) {
        Long userId = GlobalUtil.getLoginUserId();
        List<Long> idList = paymentSlipPayloadList.stream().map(p -> p.getId()).collect(Collectors.toList());
        //处理关闭合同的数据
        dealPaymentData(idList);
        //更新线下支付的数据为已付款
        if(!CollectionUtils.isEmpty(paymentSlipPayloadList)){
            paymentSlipPayloadList.stream().forEach(p->{
                p.setState(PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode());
                p.setPayMode(PurchasePaymentEnum.PayModeEnum.PAYMANUAL.getCode());
                p.setPayStatus(PurchasePaymentEnum.PayStatusEnum.SUCCESS.getCode());
                p.setPayTime(LocalDateTime.now());
                update(p);
                //支付履历插入数据
                TDocHistPayload tDocHistPayload = new TDocHistPayload();
                tDocHistPayload.setDocId(p.getId());
                tDocHistPayload.setDocNo(p.getPaymentApplyNo());
                tDocHistPayload.setHistTime(LocalDateTime.now());
                tDocHistPayload.setOwnerId(userId);
                tDocHistPayload.setCreateUserId(userId);
                tDocHistPayload.setDocType(PurchasePaymentEnum.PayDocType.CONTANT.getCode());
                tDocHistPayload.setHistType(PurchasePaymentEnum.PayHistType.HISTTYPE1.getDesc());
                tDocHistPayload.setHistDtl(PurchasePaymentEnum.PayHistDtl.DTL1.getDesc());
                tDocHistService.insert(tDocHistPayload);
            });
            sendEmail(idList);
        }
    }
    // 迪博：付款、预付款，财务线下付款成功后，需要邮件通知单据创建人（付款申请人）
    private void sendEmail(List<Long> idList) {
        List<DibPaymentSlipEmailVO> paymentSlipEmailByIds = paymentSlipDAO.findPaymentSlipEmailByIds(idList);
        PrdMessageConfigVO configVO = messageConfigService.queryByMessageCode("MC20240719000001");
        paymentSlipEmailByIds.forEach(dibPaymentSlipEmailVO -> {
            Map<String, Object> result = new HashMap<>();
            result.put("supplierLegalBookName", dibPaymentSlipEmailVO.getSupplierLegalBookName());
            result.put("purchaseName", dibPaymentSlipEmailVO.getPurchaseName());
            result.put("paymentNo", dibPaymentSlipEmailVO.getPaymentNo());
            result.put("paymentAmt", dibPaymentSlipEmailVO.getPaymentAmt());
            if (dibPaymentSlipEmailVO.getPayTime() != null) {
                result.put("payTime", dibPaymentSlipEmailVO.getPayTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
            }
            if (dibPaymentSlipEmailVO.getPurchaseInchargeResId() != null) {
                String noticeSource = dibPaymentSlipEmailVO.getPurchaseInchargeResId().toString();
                log.debug(dibPaymentSlipEmailVO.getPurchaseName() + "付款支付成功发送邮件：：：：：" + noticeSource);
                messageConfigService.sendMessageConfig(configVO, result, "appoint_people", noticeSource);
                log.debug(dibPaymentSlipEmailVO.getPurchaseName() + "付款支付成功发送邮件完成：：：：：" + noticeSource);
            }else {
                log.error(dibPaymentSlipEmailVO.getPurchaseName() + "缺少申请人，无法发送邮件");
            }
        });
    }

    //处理关闭合同的数据
    @Override
    @Transactional
    public void dealPaymentData(List<Long> idList) {
        //查询所有的付款记录单
        List<PaymentSlipVO> paymentSlipVOS = paymentSlipDAO.queryPaymentSlipVOListByIdlist(idList);
        //通过付款申请单id进行分组
        Map<Long, List<PaymentSlipVO>> paymentApplyIdMap = paymentSlipVOS.stream().collect(Collectors.groupingBy(PaymentSlipVO::getPaymentApplyId));
        //如果全部付款就关闭合同
        paymentApplyIdMap.forEach((k, v) -> {
            // 本次提交的付款申请单的总金额
            BigDecimal paymentAmt = v.stream().map(p -> p.getPaymentAmt()).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 关闭合同并且更新付款申请单的状态
            closePaymentContractAndUpdatePaymentApply(k, paymentAmt);
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void commitOnlinePayment(List<PaymentSlipPayload> paymentSlipPayloadList) {

        Long userId = GlobalUtil.getLoginUserId();
        if (CollectionUtils.isEmpty(paymentSlipPayloadList)) {
            throw TwException.error("", "勾选的数据不能为空");
        }
        // todo wly 校验不全
        List<Long> idList = paymentSlipPayloadList.stream().map(p -> p.getId()).collect(Collectors.toList());
        List<PaymentSlipVO> paymentSlipVOS = queryListByIds(idList);
        if (CollectionUtils.isEmpty(paymentSlipVOS)) {
            throw TwException.error("", "勾选的数据不存在,请联系管理员");
        }
        //查询每个付款记录的收款信息
        List<Long> accountList = new ArrayList();
        List<Long> receiveAccList = paymentSlipVOS.stream().filter(p -> StringUtils.hasText(p.getReceivingAccount())).map(s -> Long.valueOf(s.getReceivingAccount())).collect(Collectors.toList());
        List<Long> paymentAccList = paymentSlipVOS.stream().filter(p -> StringUtils.hasText(p.getPaymentAccount())).map(s -> Long.valueOf(s.getPaymentAccount())).collect(Collectors.toList());
        accountList.addAll(paymentAccList);
        accountList.addAll(receiveAccList);
        Map<Long, List<BookAccountVO>> bookAccountVOMap = bookAccountService.queryByIdList(accountList);
        if (CollectionUtils.isEmpty(bookAccountVOMap)) {
            throw TwException.error("", "收款账号不能为空");
        }
        //网银支付 单条河批量是不一样的
        String bthnbr = generateSeqNum(GenerateSeqNumConstants.BTHNBR);
        if (paymentSlipVOS.size() == 1) {
            PaymentSlipVO paymentSlipVO = paymentSlipVOS.get(0);
            if (!StringUtils.hasText(paymentSlipVO.getPayPurpose())) {
                throw TwException.error("", "付款申请单:" + paymentSlipVO.getPaymentApplyNo() + "的网银用途未填写,请退回至应付会计环节");
            }
            if (!StringUtils.hasText(paymentSlipVO.getCurrCode()) || !PurchasePaymentEnum.CurrCodeEnum.CNY.getCode().equals(paymentSlipVO.getCurrCode())) {
                throw TwException.error("", "付款申请单:" + paymentSlipVO.getPaymentApplyNo() + "暂不支持外币支付场景");
            }
            PayPayload payload = new PayPayload();
            payload.setBank(bank);
            payload.setPlatform(platform);
            payload.setSecretKey(secretKey);
            payload.setBatchFlag(1);
            payload.setTrxseq("1");
            payload.setBthnbr(bthnbr);//批次编号
            payload.setTtlamt(paymentSlipVO.getPaymentAmt());//总金额
            payload.setTtlcnt(paymentSlipVOS.size());//总笔数
            payload.setTtlnum(1);//总次数
            payload.setCuramt(paymentSlipVO.getPaymentAmt());//本次金额
            payload.setCurcnt(1);//本次次数


            savePayMentDetail(paymentSlipVO, bookAccountVOMap, payload);
            //先更新付款记录表
            PaymentSlipPayload paymentSlipPayload = paymentSlipPayloadList.get(0);
            paymentSlipPayload.setBankFlag(PurchasePaymentEnum.PayBankFlag.BANKFALG1.getCode());
            paymentSlipPayload.setBankTime(LocalDateTime.now());
            paymentSlipPayload.setPayMode(PurchasePaymentEnum.PayModeEnum.PAYCURRENCY.getCode());
            paymentSlipPayload.setPayStatus(PurchasePaymentEnum.PayStatusEnum.PAYIN.getCode());
            paymentSlipPayload.setPayTime(LocalDateTime.now());
            update(paymentSlipPayload);

            //支付履历插入数据
            TDocHistPayload tDocHistPayload = new TDocHistPayload();
            tDocHistPayload.setDocId(paymentSlipVO.getId());
            tDocHistPayload.setDocNo(paymentSlipVO.getPaymentApplyNo());
            tDocHistPayload.setHistTime(LocalDateTime.now());
            tDocHistPayload.setOwnerId(userId);
            tDocHistPayload.setCreateUserId(userId);
            tDocHistPayload.setDocType(PurchasePaymentEnum.PayDocType.CONTANT.getCode());
            tDocHistPayload.setHistType(PurchasePaymentEnum.PayHistType.HISTTYPE4.getDesc());
            tDocHistPayload.setHistDtl(PurchasePaymentEnum.PayHistDtl.DTL4.getDesc());
            if (PurchasePaymentEnum.PayStatusEnum.FAIL.getCode().equals(paymentSlipVO.getPayStatus())) {
                tDocHistPayload.setHistType(PurchasePaymentEnum.PayHistType.HISTTYPE3.getDesc());
                tDocHistPayload.setHistDtl(PurchasePaymentEnum.PayHistDtl.DTL3.getDesc());
            }
            tDocHistService.insert(tDocHistPayload);

            log.info("付款申请单批次号为:" + bthnbr + "调用银企直联开始");
            log.info("付款申请单批次号为:" + bthnbr + "请求数据为:" + JSONUtil.toJsonStr(payload));
            //调用银企直联
            ApiResult<BkPayDetailVO> pay = payService.pay(payload);
            log.info("付款申请单批次号为:" + bthnbr + "返回数据为:" + JSONUtil.toJsonStr(pay));
            log.info("付款申请单批次号为:" + bthnbr + "调用银企直联结束");
            if (pay.getCode() != 200) {
                log.error(pay.getErrorMsg());
                log.error(pay.getMsg());
                throw TwException.error("", "付款申请单批次号为:" + bthnbr + "调用银企直联失败,异常为:" + pay.getMsg());
            }
            // 插入付款发送记录表 因为调用银企直联已经成功了 为了防止下面失败导致事务回滚 加个try catch 写个异常日志
            try {
                BkPayDetailRecordPayload bkPayDetailRecordPayload = new BkPayDetailRecordPayload();
                bkPayDetailRecordPayload.setBatchFlag(payload.getBatchFlag());//批量标识
                bkPayDetailRecordPayload.setBkPayDetailId(pay.getData().getId());
                bkPayDetailRecordPayload.setBthnbr(payload.getBthnbr());
                bkPayDetailRecordPayload.setYurref(payload.getYurref());
                bkPayDetailRecordPayload.setTrxseq(payload.getTrxseq());
                bkPayDetailRecordPayload.setPayStatus(PurchasePaymentEnum.PayStatusDetailEnum.SEND.getCode());
                bkPayDetailRecordPayload.setOrderType(payload.getOrderType());
                bkPayDetailRecordPayload.setTrsamt(payload.getTrsamt());
                bkPayDetailRecordService.insert(bkPayDetailRecordPayload);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                log.error("付款申请单批次号为:" + bthnbr + "的付款记录单生成付款发送记录失败");
            }
        } else {
            PayListPayload payListPayload = new PayListPayload();
            List<PayPayload> payloadList = new ArrayList<>();
            payListPayload.setBank(bank);
            payListPayload.setPlatform(platform);
            payListPayload.setSecretKey(secretKey);
            BigDecimal totalPayMentAmt = paymentSlipVOS.stream().map(p -> p.getPaymentAmt()).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (int i = 0; i < paymentSlipVOS.size(); i++) {
                PaymentSlipVO paymentSlipVO = paymentSlipVOS.get(i);
                if (!StringUtils.hasText(paymentSlipVO.getPayPurpose())) {
                    throw TwException.error("", "付款申请单:" + paymentSlipVO.getPaymentApplyNo() + "的网银用途未填写,请退回至应付会计环节");
                }
                if (!StringUtils.hasText(paymentSlipVO.getCurrCode()) || !PurchasePaymentEnum.CurrCodeEnum.CNY.getCode().equals(paymentSlipVO.getCurrCode())) {
                    throw TwException.error("", "付款申请单:" + paymentSlipVO.getPaymentApplyNo() + "暂不支持外币支付场景");
                }
                PayPayload payload = new PayPayload();
                payload.setBatchFlag(2);
                payload.setBthnbr(bthnbr);//批次编号
                payload.setTrxseq(String.valueOf(i + 1));
                payload.setCuramt(paymentSlipVO.getPaymentAmt());//本次金额
                payload.setCurcnt(i + 1);//本次次数
                //总金额
                payload.setTtlamt(totalPayMentAmt);
                //总笔数
                payload.setTtlcnt(paymentSlipVOS.size() + 1);
                //总次数
                payload.setTtlnum(paymentSlipVOS.size() + 1);

                savePayMentDetail(paymentSlipVO, bookAccountVOMap, payload);
                payloadList.add(payload);

                //支付履历插入数据
                TDocHistPayload tDocHistPayload = new TDocHistPayload();
                tDocHistPayload.setDocId(paymentSlipVO.getId());
                tDocHistPayload.setDocNo(paymentSlipVO.getPaymentApplyNo());
                tDocHistPayload.setHistTime(LocalDateTime.now());
                tDocHistPayload.setOwnerId(userId);
                tDocHistPayload.setCreateUserId(userId);
                tDocHistPayload.setDocType(PurchasePaymentEnum.PayDocType.CONTANT.getCode());
                tDocHistPayload.setHistType(PurchasePaymentEnum.PayHistType.HISTTYPE4.getDesc());
                tDocHistPayload.setHistDtl(PurchasePaymentEnum.PayHistDtl.DTL4.getDesc());
                if (PurchasePaymentEnum.PayStatusEnum.FAIL.getCode().equals(paymentSlipVO.getPayStatus())) {
                    tDocHistPayload.setHistType(PurchasePaymentEnum.PayHistType.HISTTYPE3.getDesc());
                    tDocHistPayload.setHistDtl(PurchasePaymentEnum.PayHistDtl.DTL3.getDesc());
                }
                tDocHistService.insert(tDocHistPayload);
            }
            //更新付款记录
            paymentSlipPayloadList.stream().forEach(p -> {
                p.setBankFlag(PurchasePaymentEnum.PayBankFlag.BANKFALG1.getCode());
                p.setBankTime(LocalDateTime.now());
                p.setPayMode(PurchasePaymentEnum.PayModeEnum.PAYCURRENCY.getCode());
                p.setPayStatus(PurchasePaymentEnum.PayStatusEnum.PAYIN.getCode());
                p.setPayTime(LocalDateTime.now());
                update(p);
            });
            payListPayload.setPayPayloadList(payloadList);
            log.info("付款申请单批次号为:" + bthnbr + "调用银企直联开始");
            log.info("付款申请单批次号为:" + bthnbr + "请求数据为:" + JSONUtil.toJsonStr(payListPayload));
            // 调用银企直连
            ApiResult<List<BkPayDetailVO>> listApiResult = payService.payList(payListPayload);
            log.info("付款申请单批次号为:" + bthnbr + "返回数据为:" + JSONUtil.toJsonStr(listApiResult));
            log.info("付款申请单批次号为:" + bthnbr + "调用银企直联结束");
            if (listApiResult.getCode() != 200) {
                log.error(listApiResult.getErrorMsg());
                log.error(listApiResult.getMsg());
                throw TwException.error("", "付款申请单批次号为:" + bthnbr + "调用银企直联失败,异常为:" + listApiResult.getMsg());
            }
            //保存数据到付款记录 因为调用银企直联已经成功了 为了防止下面失败导致事务回滚 加个try catch 写个日志
            List<BkPayDetailVO> listApiResultData = listApiResult.getData();
            listApiResultData.stream().forEach(p -> {
                try {
                    //付款记录表插入数据
                    BkPayDetailRecordPayload bkPayDetailRecordPayload = new BkPayDetailRecordPayload();
                    bkPayDetailRecordPayload.setBatchFlag(p.getBatchFlag());//批量标识
                    // todo wly
                    bkPayDetailRecordPayload.setBkPayDetailId(p.getId());
                    bkPayDetailRecordPayload.setBthnbr(p.getBthnbr());
                    bkPayDetailRecordPayload.setYurref(p.getYurref());
                    bkPayDetailRecordPayload.setTrxseq(p.getTrxseq());
                    bkPayDetailRecordPayload.setPayStatus(PurchasePaymentEnum.PayStatusDetailEnum.SEND.getCode());
                    bkPayDetailRecordPayload.setOrderType(p.getOrderType());
                    bkPayDetailRecordPayload.setTrsamt(p.getTrsamt());
                    bkPayDetailRecordService.insert(bkPayDetailRecordPayload);
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                    log.error("付款记录单:" + p.getDocId() + "插入付款发送记录失败");
                    return;
                }
            });
        }
    }

    //生成银企直连的数据
    public void savePayMentDetail(PaymentSlipVO paymentSlipVO, Map<Long, List<BookAccountVO>> bookAccountVOMap, PayPayload payload) {
        Long userId = GlobalUtil.getLoginUserId();
        payload.setDocId(paymentSlipVO.getId().toString());
        //单据编号（付款申请单号）
        payload.setDocNo(paymentSlipVO.getPaymentApplyNo());
        //单据类型
        payload.setDocType(PurchasePaymentEnum.BankDocType.PAY.getCode());
        //业务号  业务参考号
        String paymentApplyNo = paymentSlipVO.getPaymentApplyNo();
        payload.setYurref(paymentSlipVO.getId().toString() + "PR" + paymentApplyNo.substring(paymentApplyNo.length() - 4));
        //
        //如果收款账号不存在 提示
        if (!StringUtils.hasText(paymentSlipVO.getReceivingAccount()) || !bookAccountVOMap.containsKey(Long.valueOf(paymentSlipVO.getReceivingAccount()))) {
            throw TwException.error("", "付款申请单为:" + paymentSlipVO.getPaymentApplyNo() + "的收款账号不正确");
        }
        if (!StringUtils.hasText(paymentSlipVO.getPaymentAccount()) || !bookAccountVOMap.containsKey(Long.valueOf(paymentSlipVO.getPaymentAccount()))) {
            throw TwException.error("", "付款申请单为:" + paymentSlipVO.getPaymentApplyNo() + "的付款账号不正确");
        }

        //收款账户信息
        Long receivingAccount = Long.valueOf(paymentSlipVO.getReceivingAccount());
        BookAccountVO receivingAccountVO = bookAccountVOMap.get(receivingAccount).get(0);
        //付款账户信息
        Long paymentAccount = Long.valueOf(paymentSlipVO.getPaymentAccount());
        BookAccountVO paymentAccountVO = bookAccountVOMap.get(paymentAccount).get(0);

        //收款户名
        payload.setAccnam(receivingAccountVO.getAccountName());
        //收款银行名称
        payload.setAccnamBankName(paymentSlipVO.getReceivingBank());
        //收款账号
        payload.setAccnbr(receivingAccountVO.getAccountNo());
        // 开户行网点
        payload.setCrtbnk(receivingAccountVO.getDepositBankOutlet());
        //收方开户行地址
        payload.setCrtadr(receivingAccountVO.getDepositCity());
        //付款户名
        payload.setPayaccnam(paymentAccountVO.getAccountName());
        //付款账号
        payload.setPayaccnbr(paymentAccountVO.getAccountNo());
        //付款金额
        payload.setTrsamt(paymentSlipVO.getPaymentAmt());
        //网银用途（支付用途）
        payload.setNusage(paymentSlipVO.getPayPurpose());
        payload.setUsrId(userId.toString());
        payload.setUsrName(cacheUtil.getUserName(userId));
        payload.setCcynbr(10);//币种人民币默认10
        //普通汇兑
        payload.setTrstyp("100001");
        //普通汇兑
        payload.setStlchn(PurchasePaymentEnum.PayStlchnEnum.ORDINARY.getCode());//G 普通Q 快速  R 实时-超网

        //系统内标识 Y 收方为招商银行账号      N收方为他行账户
        payload.setBnkflg(PurchasePaymentEnum.PaySystemEnum.YES.getCode());
        if (StringUtils.hasText(paymentSlipVO.getReceivingBank()) && paymentSlipVO.getReceivingBank().indexOf("招商") == -1) {
            payload.setBnkflg(PurchasePaymentEnum.PaySystemEnum.NO.getCode());
        } else {
            payload.setCrtadr("");
            payload.setCrtbnk("");
        }
        payload.setOrderType(PurchasePaymentEnum.OrderTypeEnum.CONTANT.getCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void createPaySerialNum(List<PaymentSlipPayload> payloadList) {
        String generateSeqNum = generateSeqNum(GenerateSeqNumConstants.PAY_SERIAL_NUM);
        if (!CollectionUtils.isEmpty(payloadList)) {
            payloadList.stream().forEach(p -> {
                p.setPaySerialNum(generateSeqNum);
                update(p);
            });
        }

    }

    @Override
    public List<String> queryPaySerialNum() {

        return paymentSlipDAO.queryPaySerialNum();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePayPurpose(Long id, String payPurpose, String paymentApplyNo) {
        //插入支付履历 修改网银用途 确认线下支付 提交网银 修改账户
        Long userId = GlobalUtil.getLoginUserId();
        //支付履历插入数据
        TDocHistPayload tDocHistPayload = new TDocHistPayload();
        tDocHistPayload.setDocId(id);
        tDocHistPayload.setDocNo(paymentApplyNo);
        tDocHistPayload.setHistTime(LocalDateTime.now());
        tDocHistPayload.setOwnerId(userId);
        tDocHistPayload.setCreateUserId(userId);
        tDocHistPayload.setDocType(PurchasePaymentEnum.PayDocType.CONTANT.getCode());
        tDocHistPayload.setHistType(PurchasePaymentEnum.PayHistType.HISTTYPE5.getDesc());
        tDocHistPayload.setHistDtl(PurchasePaymentEnum.PayHistDtl.DTL5.getDesc());
        tDocHistService.insert(tDocHistPayload);
        paymentSlipDAO.updatePayPurpose(id, payPurpose);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateReceiveAccount(Long id, String receivingBank, String receivingAccount, String paymentApplyNo) {
        //插入支付履历 修改网银用途 确认线下支付 提交网银 修改账户
        Long userId = GlobalUtil.getLoginUserId();
        //支付履历插入数据
        TDocHistPayload tDocHistPayload = new TDocHistPayload();
        tDocHistPayload.setDocId(id);
        tDocHistPayload.setDocNo(paymentApplyNo);
        tDocHistPayload.setHistTime(LocalDateTime.now());
        tDocHistPayload.setOwnerId(userId);
        tDocHistPayload.setCreateUserId(userId);
        tDocHistPayload.setDocType(PurchasePaymentEnum.PayDocType.CONTANT.getCode());
        tDocHistPayload.setHistType(PurchasePaymentEnum.PayHistType.HISTTYPE2.getDesc());
        tDocHistPayload.setHistDtl(PurchasePaymentEnum.PayHistDtl.DTL2.getDesc());
        tDocHistService.insert(tDocHistPayload);
        paymentSlipDAO.updateReceiveAccount(id, receivingBank, receivingAccount);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void passOrReject(List<PaymentSlipPayload> paymentSlipPayloadList) {
        if(!CollectionUtils.isEmpty(paymentSlipPayloadList)){
            //薪酬支付的付款申请单 在财务经理提交的时候如果payFlag是已支付 会将状态改为已付款 这个时候需要去判断合同是否关闭
            List<Long> idList = paymentSlipPayloadList.stream().filter(p -> PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode().equals(p.getState())).map(s -> s.getId()).collect(Collectors.toList());
            if(!CollectionUtils.isEmpty(idList)){
                // 去查看是否需要关闭合同
                dealPaymentData(idList);
            }
            paymentSlipPayloadList.stream().forEach(p->{
                //如果支付标识是已支付 就直接变为已付款
                if(PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode().equals(p.getState())){
                    p.setState(PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode());
                    p.setPayMode(PurchasePaymentEnum.PayModeEnum.PAYMANUAL.getCode());
                    p.setPayStatus(PurchasePaymentEnum.PayStatusEnum.SUCCESS.getCode());
                    p.setPayTime(LocalDateTime.now());
                }
                update(p);
            });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void paymentCommit(List<PaymentSlipPayload> paymentSlipPayloadList) {
        if (!CollectionUtils.isEmpty(paymentSlipPayloadList)) {
            paymentSlipPayloadList.stream().forEach(p -> update(p));
        }
        // dib专用逻辑 相当于 财务经理通过， 因财务经理通过功能已不调用
        passOrReject(paymentSlipPayloadList);
    }

    @Override
    public BookAccountVO queryReceiveAccountInfo(Long id) {
        return bookAccountService.queryByKey(id);
    }

    @Override
    public Boolean isOtherStatusByPaymentApplyId(Long paymentApplyId) {
        return paymentSlipDAO.isOtherStatusByPaymentApplyId(paymentApplyId);
    }


    //某些字段翻译
    public void translate(PaymentSlipVO paymentSlipVO, Map<Long, String> resultMap, Map<Long, List<BookAccountVO>> accountMap) {
        // 收款公司
        if (paymentSlipVO.getReceivingCompanyBookId() != null) {
            paymentSlipVO.setReceivingCompanyBookIdDesc(resultMap.get(paymentSlipVO.getReceivingCompanyBookId()));
        }
        // 付款公司
        if (paymentSlipVO.getPayCompanyBookId() != null) {
            paymentSlipVO.setPayCompanyBookIdDesc(resultMap.get(paymentSlipVO.getPayCompanyBookId()));
        }
        //付款账号
        if (paymentSlipVO.getPaymentAccount() != null && accountMap.containsKey(Long.valueOf(paymentSlipVO.getPaymentAccount()))) {
            paymentSlipVO.setPaymentAccountDesc(accountMap.get(Long.valueOf(paymentSlipVO.getPaymentAccount())).get(0).getAccountNo());
        }
        //收款账号
        if (paymentSlipVO.getReceivingAccount() != null && accountMap.containsKey(Long.valueOf(paymentSlipVO.getReceivingAccount()))) {
            paymentSlipVO.setReceivingAccountDesc(accountMap.get(Long.valueOf(paymentSlipVO.getReceivingAccount())).get(0).getAccountNo());
        }
    }

    //关闭合同并且更新付款申请单的状态
    public void closePaymentContractAndUpdatePaymentApply(Long paymentApplyId, BigDecimal currPayment) {
        BigDecimal currentPaymentApplyAmt = currPayment;
        PurchasePaymentService purchasePaymentService = SpringUtil.getBean(PurchasePaymentService.class);
        //先查询付款申请单 通过付款申请但的合同/合同no 去查询所有部分付款和已付款的付款申请单 再通过付款申请单查询所有已付款的付款记录
        // 然后和合同/协议金额去对比 相等的话说明已付款 将合同/协议关闭
        PurchasePaymentVO purchasePaymentVO = purchasePaymentService.queryByKey(paymentApplyId);
        String docNo = purchasePaymentVO.getDocNo();
        // 关闭合同标志位 为了处理预付款申请单已付款未核销的情况
        Boolean closeFlag = true;
        // 如果当前付款申请单是预付款 只是付款没有核销 就不关闭合同
        if (PurchasePaymentEnum.PaymentType.ADVANCE_PAY.getCode().equals(purchasePaymentVO.getPaymentApplicationType())) {
            closeFlag = false;
        }
        //通过合同查询所有的付款申请单
        List<PurchasePaymentVO> purchasePaymentVOS = purchasePaymentService.queryByDocNo(docNo, Arrays.asList(PurchasePaymentEnum.PaymentStatus.PARTIAL_PAY.getCode(), PurchasePaymentEnum.PaymentStatus.PAID.getCode(), PurchasePaymentEnum.PaymentStatus.WRITTEN_OFF.getCode()));
        if (!CollectionUtils.isEmpty(purchasePaymentVOS)) {
            // 如果存在已付款但是未核销的预付款申请单 就不关闭
            List<PurchasePaymentVO> advancePayPurchasePaymentVOS = purchasePaymentVOS.stream().filter(p -> PurchasePaymentEnum.PaymentType.ADVANCE_PAY.getCode().equals(p.getPaymentApplicationType()) && PurchasePaymentEnum.PaymentStatus.PAID.getCode().equals(p.getState())).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(advancePayPurchasePaymentVOS)) {
                closeFlag = false;
            }
            //获取所有的付款申请单的id
            List<Long> paymentApplyIdList = purchasePaymentVOS.stream().map(p -> p.getId()).collect(Collectors.toList());
            //通过付款申请单id去查询合同下面所有的已付款的付款记录
            List<PaymentSlipVO> paymentSlipVOS = paymentSlipDAO.queryByPaymentApplyIdListAndState(paymentApplyIdList, PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode());
            if (!CollectionUtils.isEmpty(paymentSlipVOS)) {
                //把所有的已付款的付款记录的金额加起来 再加上本次的金额
                BigDecimal totalPaymentAmt = paymentSlipVOS.stream().map(p -> p.getPaymentAmt()).reduce(BigDecimal.ZERO, BigDecimal::add);
                //把本次的金额加上之前已付款的付款记录的金额
                currPayment = currPayment.add(totalPaymentAmt);
            }
            //当前付款申清单的已付款金额
            BigDecimal totalPamentApplyAmt = paymentSlipVOS.stream().filter(p -> p.getPaymentApplyId().equals(paymentApplyId)).map(s -> s.getPaymentAmt()).reduce(BigDecimal.ZERO, BigDecimal::add);
            currentPaymentApplyAmt = totalPamentApplyAmt.add(currentPaymentApplyAmt);
        }
        PurchasePaymentPayload purchasePaymentPayload = new PurchasePaymentPayload();
        purchasePaymentPayload.setId(purchasePaymentVO.getId());
        //更新当前付款申请单的状态 小于是部分付款 大于等于是全部付款
        if (currentPaymentApplyAmt.compareTo(purchasePaymentVO.getCurrPaymentAmt()) < 0) {
            purchasePaymentPayload.setState(PurchasePaymentEnum.PaymentStatus.PARTIAL_PAY.getCode());
            purchasePaymentService.updateByKeyDynamic(purchasePaymentPayload);
        } else {
            purchasePaymentPayload.setState(PurchasePaymentEnum.PaymentStatus.PAID.getCode());
            purchasePaymentService.updateByKeyDynamic(purchasePaymentPayload);
//            //如果是采购协议的提现申请单 就将提现申请单置为全部付款 且提现申请的付款申请单的付款记录只会有一条 需求胥超
//            if (PurchasePaymentEnum.PaymentDocType.AGREEMENT.getCode().equals(purchasePaymentVO.getDocType()) && PurchasePaymentEnum.AcceptanceMethod.WITHDRAW.getCode().equals(purchasePaymentVO.getPaymentApplicationType())) {
//                //将该付款申请单下的所有的提现列表 并更新为已付款
//                resWithdrawApplyService.updateWithdrawStatusByPaymentApplyId(purchasePaymentVO.getId(), PurchasePaymentEnum.WithdrawStatus.PAID.getCode());
//            }
        }
        //采购合同
        if (closeFlag && PurchasePaymentEnum.PaymentDocType.CONTRACT.getCode().equals(purchasePaymentVO.getDocType())) {
            PurchaseContractManagerVO purchaseContractManagerVO = purchaseContractManagerService.queryByNo(purchasePaymentVO.getDocNo());
            if (purchaseContractManagerVO.getAmt() != null && purchaseContractManagerVO.getAmt().compareTo(currPayment) == 0) {
                //关闭合同
                PurchaseContractManagerPayload payload = new PurchaseContractManagerPayload();
                payload.setId(purchaseContractManagerVO.getId());
                payload.setContractStatus(PurchaseContractStatusEnum.CLOSE.getCode());
                purchaseContractManagerService.update(payload);
            }
        }
    }


    //关闭采购合同
    @Override
    public void closePaymentContract(String contractNo) {
        PurchasePaymentService purchasePaymentService = SpringUtil.getBean(PurchasePaymentService.class);
        //通过合同查询所有的付款申请单
        List<PurchasePaymentVO> purchasePaymentVOS = purchasePaymentService.queryByDocNo(contractNo, Arrays.asList(PurchasePaymentEnum.PaymentStatus.PARTIAL_PAY.getCode(), PurchasePaymentEnum.PaymentStatus.PAID.getCode(), PurchasePaymentEnum.PaymentStatus.WRITTEN_OFF.getCode()));
        if (!CollectionUtils.isEmpty(purchasePaymentVOS)) {
            List<PurchasePaymentVO> advancePayPurchasePaymentVOS = purchasePaymentVOS.stream().filter(p -> PurchasePaymentEnum.PaymentType.ADVANCE_PAY.getCode().equals(p.getPaymentApplicationType()) && PurchasePaymentEnum.PaymentStatus.PAID.getCode().equals(p.getState())).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(advancePayPurchasePaymentVOS)) {
                return;
            }
            //获取所有的付款申请单的id
            List<Long> paymentApplyIdList = purchasePaymentVOS.stream().map(PurchasePaymentVO::getId).collect(Collectors.toList());
            //通过付款申请单id去查询合同下面所有的已付款的付款记录
            List<PaymentSlipVO> paymentSlipVOS = paymentSlipDAO.queryByPaymentApplyIdListAndState(paymentApplyIdList, PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode());
            if (!CollectionUtils.isEmpty(paymentSlipVOS)) {
                //把所有的已付款的付款记录的金额加起来 再加上本次的金额
                BigDecimal totalPaymentAmt = paymentSlipVOS.stream().map(PaymentSlipVO::getPaymentAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
                PurchaseContractManagerVO purchaseContractManagerVO = purchaseContractManagerService.queryByNo(contractNo);
                if (purchaseContractManagerVO.getAmt() != null && purchaseContractManagerVO.getAmt().compareTo(totalPaymentAmt) == 0) {
                    //关闭合同
                    PurchaseContractManagerPayload payload = new PurchaseContractManagerPayload();
                    payload.setId(purchaseContractManagerVO.getId());
                    payload.setContractStatus(PurchaseContractStatusEnum.CLOSE.getCode());
                    purchaseContractManagerService.update(payload);
                }
            }
        }
    }

    @Override
    public Map<Long, BigDecimal> findNoPayAmtByPaymentApplyIdIn(Set<Long> paymentApplyIdSet) {
        List<Tuple> noPayAmtByPaymentApplyIdIn = paymentSlipDAO.findNoPayAmtByPaymentApplyIdIn(paymentApplyIdSet);
        return noPayAmtByPaymentApplyIdIn
                .stream()
                .collect(Collectors.toMap(
                        tuple -> tuple.get(0, Long.class),
                        tuple -> Objects.requireNonNullElse(tuple.get(1, BigDecimal.class), BigDecimal.ZERO)
                ));
    }
}
