package com.elitesland.tw.tw5.server.prd.schedule.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.schedule.payload.PrdActivityCalendarPayload;
import com.elitesland.tw.tw5.api.prd.schedule.query.PrdActivityCalendarQuery;
import com.elitesland.tw.tw5.api.prd.schedule.vo.PrdActivityCalendarVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.schedule.entity.PrdActivityCalendarDO;
import com.elitesland.tw.tw5.server.prd.schedule.entity.QPrdActivityCalendarDO;
import com.elitesland.tw.tw5.server.prd.schedule.repo.PrdActivityCalendarRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.List;

@Repository
@RequiredArgsConstructor
public class PrdActivityCalendarDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PrdActivityCalendarRepo repo;
    private final QPrdActivityCalendarDO qdo = QPrdActivityCalendarDO.prdActivityCalendarDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdActivityCalendarVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdActivityCalendarVO.class,
                qdo.id,
                //qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 活动标题
                qdo.title,
                // 活动类型
                qdo.type,
                // 开始时间
                qdo.startTime,
                // 结束时间
                qdo.endTime,
                // 活动城市
                qdo.city,
                // 活动地点
                qdo.location,
                // 活动规模
                qdo.scale,
                // 活动负责人
                qdo.manageUserId,
                // 参与人
                qdo.participants,
                // 直播地址
                qdo.liveUrl,
                qdo.scheduleId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdActivityCalendarVO> getJpaQueryWhere(PrdActivityCalendarQuery query) {
        JPAQuery<PrdActivityCalendarVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PrdActivityCalendarQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PrdActivityCalendarQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 活动标题 精确 */
        if (!ObjectUtils.isEmpty(query.getTitle())) {
            list.add(qdo.title.eq(query.getTitle()));
        }
        /** 活动类型 精确 */
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 开始时间 精确 */
        if (!ObjectUtils.isEmpty(query.getStartTime())) {
            list.add(qdo.startTime.eq(query.getStartTime()));
        }
        /** 结束时间 精确 */
        if (!ObjectUtils.isEmpty(query.getEndTime())) {
            list.add(qdo.endTime.eq(query.getEndTime()));
        }
        /** 活动城市 精确 */
        if (!ObjectUtils.isEmpty(query.getCity())) {
            list.add(qdo.city.eq(query.getCity()));
        }
        /** 活动地点 精确 */
        if (!ObjectUtils.isEmpty(query.getLocation())) {
            list.add(qdo.location.eq(query.getLocation()));
        }
        /** 活动负责人 精确 */
        if (!ObjectUtils.isEmpty(query.getManageUserId())) {
            list.add(qdo.manageUserId.eq(query.getManageUserId()));
        }
        /** 参与人 精确 */
        if (!ObjectUtils.isEmpty(query.getParticipants())) {
            list.add(qdo.participants.eq(query.getParticipants()));
        }
        /** 直播地址 精确 */
        if (!ObjectUtils.isEmpty(query.getLiveUrl())) {
            list.add(qdo.liveUrl.eq(query.getLiveUrl()));
        }
        if (!ObjectUtils.isEmpty(query.getProjectTime())) {
            list.add(qdo.startTime.between(query.getProjectTime().atTime(0, 0, 0), query.getProjectTime().atTime(23, 59, 59)).
                    or(qdo.endTime.between(query.getProjectTime().atTime(0, 0, 0), query.getProjectTime().atTime(23, 59, 59))));
        }
        if (!ObjectUtils.isEmpty(query.getYearMonthDate())) {
            list.add(qdo.startTime.loe(query.getYearMonthDate().with(TemporalAdjusters.lastDayOfMonth()).atTime(23, 59, 59)));
            list.add(qdo.endTime.goe(query.getYearMonthDate().with(TemporalAdjusters.firstDayOfMonth()).atTime(0, 0, 0)));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdActivityCalendarVO queryByKey(Long id) {
        JPAQuery<PrdActivityCalendarVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdActivityCalendarVO> queryListDynamic(PrdActivityCalendarQuery query) {
        JPAQuery<PrdActivityCalendarVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdActivityCalendarVO> queryPaging(PrdActivityCalendarQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PrdActivityCalendarVO> jpaQuery = getJpaQueryWhere(query);
        List<PrdActivityCalendarVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PrdActivityCalendarVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdActivityCalendarDO save(PrdActivityCalendarDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdActivityCalendarDO> saveAll(List<PrdActivityCalendarDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PrdActivityCalendarPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 活动标题
        if (payload.getTitle() != null) {
            update.set(qdo.title, payload.getTitle());
        }
        // 活动类型
        if (payload.getType() != null) {
            update.set(qdo.type, payload.getType());
        }
        // 开始时间
        if (payload.getStartTime() != null) {
            update.set(qdo.startTime, payload.getStartTime());
        }
        // 结束时间
        if (payload.getEndTime() != null) {
            update.set(qdo.endTime, payload.getEndTime());
        }
        // 活动城市
        if (payload.getCity() != null) {
            update.set(qdo.city, payload.getCity());
        }
        // 活动地点
        if (payload.getLocation() != null) {
            update.set(qdo.location, payload.getLocation());
        }
        // 活动规模
        if (payload.getScale() != null) {
            update.set(qdo.scale, payload.getScale());
        }
        // 活动负责人
        if (payload.getManageUserId() != null) {
            update.set(qdo.manageUserId, payload.getManageUserId());
        }
        // 参与人
        if (payload.getParticipants() != null) {
            update.set(qdo.participants, payload.getParticipants());
        }
        // 直播地址
        if (payload.getLiveUrl() != null) {
            update.set(qdo.liveUrl, payload.getLiveUrl());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 活动标题
            if (nullFields.contains("title")) {
                update.setNull(qdo.title);
            }
            // 活动类型
            if (nullFields.contains("type")) {
                update.setNull(qdo.type);
            }
            // 开始时间
            if (nullFields.contains("startTime")) {
                update.setNull(qdo.startTime);
            }
            // 结束时间
            if (nullFields.contains("endTime")) {
                update.setNull(qdo.endTime);
            }
            // 活动城市
            if (nullFields.contains("city")) {
                update.setNull(qdo.city);
            }
            // 活动地点
            if (nullFields.contains("location")) {
                update.setNull(qdo.location);
            }
            // 活动规模
            if (nullFields.contains("scale")) {
                update.setNull(qdo.scale);
            }
            // 活动负责人
            if (nullFields.contains("manageUserId")) {
                update.setNull(qdo.manageUserId);
            }
            // 参与人
            if (nullFields.contains("participants")) {
                update.setNull(qdo.participants);
            }
            // 直播地址
            if (nullFields.contains("liveUrl")) {
                update.setNull(qdo.liveUrl);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}
