package com.elitesland.tw.tw5.server.prd.system.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemSelectionPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemSelectionQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemRemindService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemSelectionService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemSelectionConvert;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemSelectionDAO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemSelectionDO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 系统选择项service
 *
 * @author wangding
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdSystemSelectionServiceImpl implements PrdSystemSelectionService {
    private final PrdSystemSelectionDAO dao;
    private final CacheUtil cacheUtil;
    private final PrdSystemRemindService remindService;

    @Transactional
    @Override
    public PrdSystemSelectionVO insert(PrdSystemSelectionPayload payload) {
        payload.setSelectionKey(payload.getSelectionKey().trim());
        payload.setSelectionValue(payload.getSelectionValue().trim());
        // 新增的时候给默认值 默认是不禁用的
        String remindCode = "SYS:E:SYSTEM_SELECTION_KEY_NOT_REPEAT";
        boolean allowReturn = remindService.allowReturn(remindCode, payload);
        if (allowReturn) {
            // 检查编码不能重复
            PrdSystemSelectionQuery query = new PrdSystemSelectionQuery();
            query.setSelectionKey(payload.getSelectionKey());
            int count = dao.queryListDynamic(query).size();
            if (count > 0) {
                throw TwException.error("", "编号不可重复");
            }
        }
        PrdSystemSelectionDO ado = PrdSystemSelectionConvert.INSTANCE.toDo(payload);
        dao.save(ado);
        cacheUtil.loadSystemSelectionCache();
        return PrdSystemSelectionConvert.INSTANCE.toVo(ado);
    }

    @Transactional
    @Override
    public Long update(PrdSystemSelectionPayload payload) {
        payload.setSelectionKey(payload.getSelectionKey().trim());
        payload.setSelectionValue(payload.getSelectionValue().trim());
        String remindCode = "SYS:E:SYSTEM_SELECTION_KEY_NOT_REPEAT";
        boolean allowReturn = remindService.allowReturn(remindCode, payload);
        if (allowReturn) {
            // 检查编码不能重复
            PrdSystemSelectionQuery query = new PrdSystemSelectionQuery();
            query.setSelectionKey(payload.getSelectionKey());
            query.setId(payload.getId());
            int count = dao.queryListDynamic(query).size();
            if (count > 0) {
                throw TwException.error("", "编号不可重复");
            }
        }
        dao.updateByKeyDynamic(payload);
        cacheUtil.loadSystemSelectionCache();
        return 0L;
    }

    @Transactional
    @Override
    public boolean delete(List<Long> keys) {
        dao.delete(keys);
        cacheUtil.loadSystemSelectionCache();
        return true;
    }

    @Transactional
    @Override
    public boolean deleteSoft(List<Long> keys) {
        dao.deleteSoft(keys);
        cacheUtil.loadSystemSelectionCache();
        return true;
    }

    @Override
    public PrdSystemSelectionVO queryByKey(Long key) {
        return dao.queryByKey(key);
    }

    @Override
    public PagingVO<PrdSystemSelectionVO> paging(PrdSystemSelectionQuery query) {
        return dao.queryPaging(query);
    }


    @Override
    public List<PrdSystemSelectionVO> selectByCondition(String key) {
        PrdSystemSelectionVO view = cacheUtil.getSystemSelection(key);
        if (view == null || view.getChildren() == null || view.getChildren().isEmpty()) {
            return new ArrayList<>();
        }
        List<PrdSystemSelectionVO> children = view.getChildren();
        List<PrdSystemSelectionVO> list = new ArrayList<>(children.size());
        children.forEach(selectionView -> list.add(selectionView.clone()));
        list.forEach(selectionView -> selectionView.setChildren(null));
        List<PrdSystemSelectionVO> collect = list.stream().sorted(Comparator.comparing(selectionView -> selectionView.getSortNo() == null ? Integer.MAX_VALUE : selectionView.getSortNo())).collect(Collectors.toList());

        return collect;
    }

    //    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public List<PrdSystemSelectionVO> queryList(PrdSystemSelectionQuery query) {
        return dao.queryListDynamic(query);
    }

    @Override
    public List<PrdSystemSelectionVO> getSystemSelectionChildrenList(PrdSystemSelectionQuery query) {
        PrdSystemSelectionVO view = cacheUtil.getSystemSelection(query.getParentSelectionKey());
        if (view == null || view.getChildren() == null || view.getChildren().isEmpty()) {
            return Collections.emptyList();
        }
        List<PrdSystemSelectionVO> allChildren = new ArrayList<>();
        cacheUtil.getAllChildren(view, allChildren);
        var filtedList = filte(allChildren, query);
        var cloneList = filtedList.stream().map(PrdSystemSelectionVO::clone).collect(Collectors.toList());
        cloneList.forEach(prdSystemSelectionVO -> prdSystemSelectionVO.setChildren(null));//非树
        return cloneList;
    }

    List<PrdSystemSelectionVO> filte(List<PrdSystemSelectionVO> list, PrdSystemSelectionQuery query) {
        var stream = list.stream();
        if (StringUtils.isNotBlank(query.getSelectionName())) {
            stream = stream.filter(prdSystemSelectionVO -> query.getSelectionName().equals(prdSystemSelectionVO.getSelectionName()));
        }
        if (StringUtils.isNotBlank(query.getSelectionValue())) {
            stream = stream.filter(prdSystemSelectionVO -> query.getSelectionValue().equals(prdSystemSelectionVO.getSelectionValue()));
        }
        if (query.getParentId() != null) {
            stream = stream.filter(prdSystemSelectionVO -> query.getParentId().equals(prdSystemSelectionVO.getParentId()));
        }
        if (StringUtils.isNotBlank(query.getExtString10())) {
            stream = stream.filter(prdSystemSelectionVO -> query.getExtString10().equals(prdSystemSelectionVO.getExtString10()));
        }
        return stream.collect(Collectors.toList());
    }
}
