package com.elitesland.tw.tw5.server.prd.file.service;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.tw.tw5.api.prd.file.payload.PrdFilePayload;
import com.elitesland.tw.tw5.api.prd.file.query.PrdFileQuery;
import com.elitesland.tw.tw5.api.prd.file.service.PrdFileService;
import com.elitesland.tw.tw5.api.prd.file.vo.PrdFileVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemLogService;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.FileUtil;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.file.config.FileProperties;
import com.elitesland.tw.tw5.server.prd.file.convert.PrdFileConvert;
import com.elitesland.tw.tw5.server.prd.file.entity.PrdFileDO;
import com.elitesland.tw.tw5.server.prd.file.repo.PrdFileRepo;
import com.elitesland.tw.tw5.server.prd.office.storage.FileStorageMutator;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Optional;

/**
 * 附件-文件本地存储实现
 *
 * @author duwh
 * @date 2022/09/22
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdFileLocalServiceImpl implements PrdFileService {

    private final FileProperties fileProperties;
    private final PrdSystemLogService logService;
    private final PrdFileRepo prdFileRepo;
    private final FileStorageMutator storageMutator;

    /**
     * 上传
     *
     * @param name          名字
     * @param folderId      文件夹id
     * @param multipartFile 附件
     * @return {@link PrdFileVO}
     */
    @Override
    public PrdFileVO upload(String name, Long folderId, MultipartFile multipartFile) {
        // TODO  校验文件夹是否存在
        if (null == folderId) {
            folderId = 1L;
        }
        // 原始上传的文件名
        final String originalFilename = multipartFile.getOriginalFilename();
        final long fileSize = multipartFile.getSize();
        FileUtil.checkSize(fileProperties.getMaxSize(), fileSize);
        String suffix = FileUtil.getExtensionName(multipartFile.getOriginalFilename());
        String type = FileUtil.getFileType(suffix);
        File file = FileUtil.upload(multipartFile, fileProperties.getPath().getPath() + type + File.separator);
        if (ObjectUtil.isNull(file)) {
            throw TwException.error("0802", "上传失败");
        }
        try {
            name = StringUtils.isBlank(name) ? FileUtil.getFileNameNoEx(multipartFile.getOriginalFilename()) : name;
            PrdFileDO entityDo = new PrdFileDO();
            entityDo.setName(name);
            entityDo.setFolderId(folderId);
            entityDo.setFileName(originalFilename);
            entityDo.setRealName(file.getName());
            entityDo.setServerPath(file.getPath());
            entityDo.setSuffix(suffix);
            entityDo.setFileType(multipartFile.getContentType());
            entityDo.setFileTypeDesc(type);
            entityDo.setFileSize(fileSize);
            entityDo.setFileSizeDesc(FileUtil.getSize(fileSize));
            return PrdFileConvert.INSTANCE.toVo(prdFileRepo.save(entityDo));
        } catch (Exception e) {
            FileUtil.del(file);
            throw e;
        }
    }

    /**
     * 根据合同模板名称创建空文件
     *
     * @param name     名字
     * @param folderId 文件夹id
     * @return {@link PrdFileVO}
     */
    @Override
    public PrdFileVO createByContractTemp(String name, Long folderId) throws IOException {
        if (StringUtils.isEmpty(name)) {
            throw TwException.error("", "name is empty");
        }
        Date date = new Date();
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMddhhmmssS");
        String nowStr = "-" + format.format(date);

        String pathStr = fileProperties.getPath().getPath() + FileUtil.TXT + File.separator;
        final String suffix = ".docx";
        String fileName = name + suffix;
        String fileNamePro = name + nowStr + suffix;
        String fileNamePath = pathStr + fileNamePro;
        log.info("[file] fileNamePath:{}", fileNamePath);
        if (null == folderId) {
            folderId = 1L;
        }
        String demoPath = "assets" + File.separator + "new.docx";
        // get the input file stream
        InputStream stream = Thread.currentThread()
            .getContextClassLoader()
            .getResourceAsStream(demoPath);

        if (stream == null) {
            log.error("[file] 模板new.docx文件不存在");
            return null;
        }
        // create a file in the specified directory
        final Path path = Path.of(fileNamePath);
        storageMutator.createFile(path, stream);
        //storageMutator.createMeta(fileNamePath, uid, uname);  // create meta information of the demo file
        // 原始上传的文件名
        final String originalFilename = fileName;
        PrdFileDO entityDo = new PrdFileDO();
        entityDo.setName(name);
        entityDo.setFolderId(folderId);
        entityDo.setFileName(originalFilename);
        entityDo.setRealName(fileName);
        entityDo.setServerPath(fileNamePath);
        entityDo.setSuffix("docx");
        final String fileType = Files.probeContentType(path);
        //entityDo.setFileType("application/vnd.openxmlformats-officedocument.wordprocessingml.document");
        entityDo.setFileType(fileType);
        entityDo.setFileTypeDesc(FileUtil.TXT);
        final long fileSize = Files.size(path);
        entityDo.setFileSize(fileSize);
        entityDo.setFileSizeDesc(FileUtil.getSize(fileSize));
        return PrdFileConvert.INSTANCE.toVo(prdFileRepo.save(entityDo));

    }

    /**
     * 获取文件路径
     *
     * @param type 类型
     * @return {@link String}
     */
    @Override
    public String getFilePath(String type) {
        return fileProperties.getPath().getPath() + type + File.separator;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdFileVO insert(PrdFilePayload payload) {
        PrdFileDO entityDo = PrdFileConvert.INSTANCE.toDo(payload);
        prdFileRepo.save(entityDo);
        return PrdFileConvert.INSTANCE.toVo(entityDo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdFileVO update(PrdFilePayload payload) {
        PrdFileDO entity = prdFileRepo.findById(payload.getId()).orElseGet(PrdFileDO::new);
        Assert.notNull(entity.getId(), "附件不存在");
        PrdFileDO entityDo = PrdFileConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return PrdFileConvert.INSTANCE.toVo(prdFileRepo.save(entity));
    }

    @Override
    public PrdFileVO queryByKey(Long key) {
        PrdFileDO entity = prdFileRepo.findById(key).orElseGet(PrdFileDO::new);
        Assert.notNull(entity.getId(), "附件不存在");
        return PrdFileConvert.INSTANCE.toVo(entity);
    }

    @Override
    public List<PrdFileVO> queryList(PrdFileQuery query) {
        return PrdFileConvert.INSTANCE.toVoList(prdFileRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    @Override
    public PagingVO<PrdFileVO> paging(PrdFileQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.desc("createTime");
        query.defaultOrder(orderItem);
        final Specification<PrdFileDO> specification = (root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder);
        Page<PrdFileDO> page = prdFileRepo.findAll(specification, query.getPageRequest());
        return PageUtil.toPageVo(page.map(PrdFileConvert.INSTANCE::toVo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            prdFileRepo.deleteSoft(keys);
        }
    }


}
