package com.elitesland.tw.tw5.server.prd.my.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.TAttendanceRecordPayload;
import com.elitesland.tw.tw5.api.prd.my.query.TAttendanceRecordQuery;
import com.elitesland.tw.tw5.api.prd.my.vo.TAttendanceRecordVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.QTAttendanceRecordDO;
import com.elitesland.tw.tw5.server.prd.my.entity.TAttendanceRecordDO;
import com.elitesland.tw.tw5.server.prd.my.repo.TAttendanceRecordRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

@Repository
@RequiredArgsConstructor
public class TAttendanceRecordDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TAttendanceRecordRepo repo;
    private final QTAttendanceRecordDO qdo = QTAttendanceRecordDO.tAttendanceRecordDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TAttendanceRecordVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TAttendanceRecordVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 打卡人员ID
                qdo.attendanceResId,
                // 打卡日期
                qdo.attendanceDate,
                // 规则ID
                qdo.attendanceRuleId,
                // 上班时间
                qdo.attendanceTimeStart,
                // 下班时间
                qdo.attendanceTimeEnd,
                // 打卡结果 YSE/NO
                qdo.attendanceResult,
                // 正常、迟到或早退、矿工
                qdo.attendanceResultDetail,
                // 特殊打卡原因
                qdo.specialReason,
                // 打卡设备
                qdo.attendanceDevice
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TAttendanceRecordVO> getJpaQueryWhere(TAttendanceRecordQuery query) {
        JPAQuery<TAttendanceRecordVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TAttendanceRecordQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TAttendanceRecordQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 打卡人员ID 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceResId())) {
            list.add(qdo.attendanceResId.eq(query.getAttendanceResId()));
        }
        /** 打卡日期 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceDate())) {
            list.add(qdo.attendanceDate.eq(query.getAttendanceDate()));
        }
        /** 规则ID 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceRuleId())) {
            list.add(qdo.attendanceRuleId.eq(query.getAttendanceRuleId()));
        }
        /** 上班时间 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceTimeStart())) {
            list.add(qdo.attendanceTimeStart.eq(query.getAttendanceTimeStart()));
        }
        /** 下班时间 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceTimeEnd())) {
            list.add(qdo.attendanceTimeEnd.eq(query.getAttendanceTimeEnd()));
        }
        /** 打卡结果 YSE/NO 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceResult())) {
            list.add(qdo.attendanceResult.eq(query.getAttendanceResult()));
        }
        /** 正常、迟到或早退、矿工 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceResultDetail())) {
            list.add(qdo.attendanceResultDetail.eq(query.getAttendanceResultDetail()));
        }
        /** 特殊打卡原因 精确 */
        if (!ObjectUtils.isEmpty(query.getSpecialReason())) {
            list.add(qdo.specialReason.eq(query.getSpecialReason()));
        }
        /** 打卡设备 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceDevice())) {
            list.add(qdo.attendanceDevice.eq(query.getAttendanceDevice()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TAttendanceRecordVO queryByKey(Long id) {
        JPAQuery<TAttendanceRecordVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TAttendanceRecordVO> queryListDynamic(TAttendanceRecordQuery query) {
        JPAQuery<TAttendanceRecordVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TAttendanceRecordVO> queryPaging(TAttendanceRecordQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<TAttendanceRecordVO> jpaQuery = getJpaQueryWhere(query);
        List<TAttendanceRecordVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TAttendanceRecordVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TAttendanceRecordDO save(TAttendanceRecordDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TAttendanceRecordDO> saveAll(List<TAttendanceRecordDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TAttendanceRecordPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 打卡人员ID
        if (payload.getAttendanceResId() != null) {
            update.set(qdo.attendanceResId, payload.getAttendanceResId());
        }
        // 打卡日期
        if (payload.getAttendanceDate() != null) {
            update.set(qdo.attendanceDate, payload.getAttendanceDate());
        }
        // 规则ID
        if (payload.getAttendanceRuleId() != null) {
            update.set(qdo.attendanceRuleId, payload.getAttendanceRuleId());
        }
        // 上班时间
        if (payload.getAttendanceTimeStart() != null) {
            update.set(qdo.attendanceTimeStart, payload.getAttendanceTimeStart());
        }
        // 下班时间
        if (payload.getAttendanceTimeEnd() != null) {
            update.set(qdo.attendanceTimeEnd, payload.getAttendanceTimeEnd());
        }
        // 打卡结果 YSE/NO
        if (payload.getAttendanceResult() != null) {
            update.set(qdo.attendanceResult, payload.getAttendanceResult());
        }
        // 正常、迟到或早退、矿工
        if (payload.getAttendanceResultDetail() != null) {
            update.set(qdo.attendanceResultDetail, payload.getAttendanceResultDetail());
        }
        // 特殊打卡原因
        if (payload.getSpecialReason() != null) {
            update.set(qdo.specialReason, payload.getSpecialReason());
        }
        // 打卡设备
        if (payload.getAttendanceDevice() != null) {
            update.set(qdo.attendanceDevice, payload.getAttendanceDevice());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 打卡人员ID
            if (nullFields.contains("attendanceResId")) {
                update.setNull(qdo.attendanceResId);
            }
            // 打卡日期
            if (nullFields.contains("attendanceDate")) {
                update.setNull(qdo.attendanceDate);
            }
            // 规则ID
            if (nullFields.contains("attendanceRuleId")) {
                update.setNull(qdo.attendanceRuleId);
            }
            // 上班时间
            if (nullFields.contains("attendanceTimeStart")) {
                update.setNull(qdo.attendanceTimeStart);
            }
            // 下班时间
            if (nullFields.contains("attendanceTimeEnd")) {
                update.setNull(qdo.attendanceTimeEnd);
            }
            // 打卡结果 YSE/NO
            if (nullFields.contains("attendanceResult")) {
                update.setNull(qdo.attendanceResult);
            }
            // 正常、迟到或早退、矿工
            if (nullFields.contains("attendanceResultDetail")) {
                update.setNull(qdo.attendanceResultDetail);
            }
            // 特殊打卡原因
            if (nullFields.contains("specialReason")) {
                update.setNull(qdo.specialReason);
            }
            // 打卡设备
            if (nullFields.contains("attendanceDevice")) {
                update.setNull(qdo.attendanceDevice);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    public List<Long> findResAttendanceLog(Long resId, LocalDate attendanceDate) {
        return jpaQueryFactory.select(qdo.id)
                .from(qdo)
                .where(qdo.attendanceResId.eq(resId).and(qdo.attendanceDate.eq(attendanceDate)))
                .fetch();
    }

    public String findStartTime(Long id) {
        return jpaQueryFactory.select(qdo.attendanceTimeStart)
                .from(qdo)
                .where(qdo.id.eq(id))
                .fetchOne();
    }

    public String findResAttendanceStatus(Long id) {
        return jpaQueryFactory.select(qdo.attendanceResultDetail)
                .from(qdo)
                .where(qdo.id.eq(id))
                .fetchOne();
    }

    public String findResAttendanceResult(Long id) {
        return jpaQueryFactory.select(qdo.attendanceResult)
                .from(qdo)
                .where(qdo.id.eq(id))
                .fetchOne();
    }

}

