package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectSummarySettingPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectSummarySettingQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectSummarySettingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectSummarySettingDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectSummarySettingDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectSummarySettingRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 会议纪要配置
 *
 * @author zoey
 * @date 2024-05-28
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectSummarySettingDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectSummarySettingRepo repo;
    private final QPmsProjectSummarySettingDO qdo = QPmsProjectSummarySettingDO.pmsProjectSummarySettingDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectSummarySettingVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectSummarySettingVO.class,
                qdo.id,
                //qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 交付BU
                qdo.deliOrgId,
                // 纪要人员
                qdo.summaryUsers
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectSummarySettingVO> getJpaQueryWhere(PmsProjectSummarySettingQuery query) {
        JPAQuery<PmsProjectSummarySettingVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectSummarySettingQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectSummarySettingQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 交付BU 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliOrgId())) {
            list.add(qdo.deliOrgId.eq(query.getDeliOrgId()));
        }
        /** 纪要人员 精确 */
        if (!ObjectUtils.isEmpty(query.getSummaryUsers())) {
            list.add(qdo.summaryUsers.like(SqlUtil.toSqlLikeString(query.getSummaryUsers())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectSummarySettingVO queryByKey(Long id) {
        JPAQuery<PmsProjectSummarySettingVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectSummarySettingVO> queryListDynamic(PmsProjectSummarySettingQuery query) {
        JPAQuery<PmsProjectSummarySettingVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectSummarySettingVO> queryPaging(PmsProjectSummarySettingQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectSummarySettingVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectSummarySettingVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectSummarySettingVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectSummarySettingDO save(PmsProjectSummarySettingDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectSummarySettingDO> saveAll(List<PmsProjectSummarySettingDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectSummarySettingPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 交付BU
        if (payload.getDeliOrgId() != null) {
            update.set(qdo.deliOrgId, payload.getDeliOrgId());
        }
        // 纪要人员
        if (payload.getSummaryUsers() != null) {
            update.set(qdo.summaryUsers, payload.getSummaryUsers());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 交付BU
            if (nullFields.contains("deliOrgId")) {
                update.setNull(qdo.deliOrgId);
            }
            // 纪要人员
            if (nullFields.contains("summaryUsers")) {
                update.setNull(qdo.summaryUsers);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

