package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchaseAgreementAssociationPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseAgreementAssociationQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseAgreementAssociationVO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PurchaseAgreementAssociationDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseAgreementAssociationDO;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PurchaseAgreementAssociationRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 采购协议关联协议
 *
 * @author Echo
 * @date 2023-06-08
 */
@Repository
@RequiredArgsConstructor
public class PurchaseAgreementAssociationDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PurchaseAgreementAssociationRepo repo;
    private final QPurchaseAgreementAssociationDO qdo = QPurchaseAgreementAssociationDO.purchaseAgreementAssociationDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementAssociationVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PurchaseAgreementAssociationVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 序号
                qdo.no,
                qdo.agreementId,
                // 关联协议
                qdo.associationAgreementId,
                // 供应商地址簿Id
                qdo.supplierBookId,
                // 协议编码
                qdo.agreementCode,
                // 协议名称
                qdo.agreementName,
                // 协议内容说明
                qdo.associationContentDescription
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementAssociationVO> getJpaQueryWhere(PurchaseAgreementAssociationQuery query) {
        JPAQuery<PurchaseAgreementAssociationVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchaseAgreementAssociationQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchaseAgreementAssociationQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 序号 精确 */
        if (!ObjectUtils.isEmpty(query.getNo())) {
            list.add(qdo.no.eq(query.getNo()));
        }
        if (!ObjectUtils.isEmpty(query.getAgreementId())) {
            list.add(qdo.agreementId.eq(query.getAgreementId()));
        }
        /** 关联协议 精确 */
        if (!ObjectUtils.isEmpty(query.getAssociationAgreementId())) {
            list.add(qdo.associationAgreementId.eq(query.getAssociationAgreementId()));
        }
        /** 供应商地址簿Id 精确 */
        if (!ObjectUtils.isEmpty(query.getSupplierBookId())) {
            list.add(qdo.supplierBookId.eq(query.getSupplierBookId()));
        }
        /** 协议编码 精确 */
        if (!ObjectUtils.isEmpty(query.getAgreementCode())) {
            list.add(qdo.agreementCode.eq(query.getAgreementCode()));
        }
        /** 协议名称 精确 */
        if (!ObjectUtils.isEmpty(query.getAgreementName())) {
            list.add(qdo.agreementName.eq(query.getAgreementName()));
        }
        /** 协议内容说明 精确 */
        if (!ObjectUtils.isEmpty(query.getAssociationContentDescription())) {
            list.add(qdo.associationContentDescription.eq(query.getAssociationContentDescription()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchaseAgreementAssociationVO queryByKey(Long id) {
        JPAQuery<PurchaseAgreementAssociationVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchaseAgreementAssociationVO> queryListDynamic(PurchaseAgreementAssociationQuery query) {
        JPAQuery<PurchaseAgreementAssociationVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchaseAgreementAssociationVO> queryPaging(PurchaseAgreementAssociationQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PurchaseAgreementAssociationVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchaseAgreementAssociationVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchaseAgreementAssociationVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PurchaseAgreementAssociationDO save(PurchaseAgreementAssociationDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PurchaseAgreementAssociationDO> saveAll(List<PurchaseAgreementAssociationDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PurchaseAgreementAssociationPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 序号
        if (payload.getNo() != null) {
            update.set(qdo.no, payload.getNo());
        }
        // 关联协议
        if (payload.getAssociationAgreementId() != null) {
            update.set(qdo.associationAgreementId, payload.getAssociationAgreementId());
        }
        // 供应商
        if (payload.getSupplierBookId() != null) {
            update.set(qdo.supplierBookId, payload.getSupplierBookId());
        }
        // 协议编码
        if (payload.getAgreementCode() != null) {
            update.set(qdo.agreementCode, payload.getAgreementCode());
        }
        // 协议名称
        if (payload.getAgreementName() != null) {
            update.set(qdo.agreementName, payload.getAgreementName());
        }
        // 协议内容说明
        if (payload.getAssociationContentDescription() != null) {
            update.set(qdo.associationContentDescription, payload.getAssociationContentDescription());
        }
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 序号
            if (nullFields.contains("no")) {
                update.setNull(qdo.no);
            }
            if (nullFields.contains("agreementId")) {
                update.setNull(qdo.agreementId);
            }
            // 关联协议
            if (nullFields.contains("associationAgreementId")) {
                update.setNull(qdo.associationAgreementId);
            }
            // 供应商
            if (nullFields.contains("supplierBookId")) {
                update.setNull(qdo.supplierBookId);
            }
            // 协议编码
            if (nullFields.contains("agreementCode")) {
                update.setNull(qdo.agreementCode);
            }
            // 协议名称
            if (nullFields.contains("agreementName")) {
                update.setNull(qdo.agreementName);
            }
            // 协议内容说明
            if (nullFields.contains("associationContentDescription")) {
                update.setNull(qdo.associationContentDescription);
            }
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
            if (nullFields.contains("deleteFlag")) {
                update.setNull(qdo.deleteFlag);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 删除关联协议
     * @param documentIdList 协议IdList
     */
    public void deleteSoftByDocumentIdList(List<Long> documentIdList) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.agreementId.in(documentIdList));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }
}

