package com.elitesland.tw.tw5.server.prd.budget.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.budget.payload.PmsBudgetDocRelatePayload;
import com.elitesland.tw.tw5.api.prd.budget.query.PmsBudgetDocRelateQuery;
import com.elitesland.tw.tw5.api.prd.budget.vo.PmsBudgetDocRelateVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.budget.entity.PmsBudgetDocRelateDO;
import com.elitesland.tw.tw5.server.prd.budget.entity.QPmsBudgetDocRelateDO;
import com.elitesland.tw.tw5.server.prd.budget.repo.PmsBudgetDocRelateRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPADeleteClause;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算单据关联管理
 *
 * @author sunxw
 * @date 2024-09-05
 */
@Repository
@RequiredArgsConstructor
public class PmsBudgetDocRelateDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsBudgetDocRelateRepo repo;
    private final QPmsBudgetDocRelateDO qdo = QPmsBudgetDocRelateDO.pmsBudgetDocRelateDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetDocRelateVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsBudgetDocRelateVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 来源单据ID
                qdo.relateDocId,
                // 来源单据类型
                qdo.relateDocType,
                // 事由ID
                qdo.reasonId,
                // 事由类型
                qdo.reasonType,
                // 核算项目ID
                qdo.busItemId,
                // 占用金额
                qdo.occupyAmt,
                // 已使用金额
                qdo.usedAmt,
                // 扩展字段1
                qdo.extString1,
                // 扩展字段2
                qdo.extString2,
                // 扩展字段3
                qdo.extString3
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetDocRelateVO> getJpaQueryWhere(PmsBudgetDocRelateQuery query) {
        JPAQuery<PmsBudgetDocRelateVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsBudgetDocRelateQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsBudgetDocRelateQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 来源单据ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRelateDocId())) {
            list.add(qdo.relateDocId.eq(query.getRelateDocId()));
        }
        /** 来源单据类型 精确 */
        if (!ObjectUtils.isEmpty(query.getRelateDocType())) {
            list.add(qdo.relateDocType.eq(query.getRelateDocType()));
        }
        /** 事由ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReasonId())) {
            list.add(qdo.reasonId.eq(query.getReasonId()));
        }
        /** 事由类型 精确 */
        if (!ObjectUtils.isEmpty(query.getReasonType())) {
            list.add(qdo.reasonType.eq(query.getReasonType()));
        }
        /** 核算项目ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBusItemId())) {
            list.add(qdo.busItemId.eq(query.getBusItemId()));
        }
        /** 占用金额 精确 */
        if (!ObjectUtils.isEmpty(query.getOccupyAmt())) {
            list.add(qdo.occupyAmt.eq(query.getOccupyAmt()));
        }
        /** 已使用金额 精确 */
        if (!ObjectUtils.isEmpty(query.getUsedAmt())) {
            list.add(qdo.usedAmt.eq(query.getUsedAmt()));
        }
        /** 扩展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString1())) {
            list.add(qdo.extString1.eq(query.getExtString1()));
        }
        /** 扩展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString2())) {
            list.add(qdo.extString2.eq(query.getExtString2()));
        }
        /** 扩展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExtString3())) {
            list.add(qdo.extString3.eq(query.getExtString3()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsBudgetDocRelateVO queryByKey(Long id) {
        JPAQuery<PmsBudgetDocRelateVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsBudgetDocRelateVO> queryListDynamic(PmsBudgetDocRelateQuery query) {
        JPAQuery<PmsBudgetDocRelateVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsBudgetDocRelateVO> queryPaging(PmsBudgetDocRelateQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsBudgetDocRelateVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsBudgetDocRelateVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsBudgetDocRelateVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsBudgetDocRelateDO save(PmsBudgetDocRelateDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsBudgetDocRelateDO> saveAll(List<PmsBudgetDocRelateDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsBudgetDocRelatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 来源单据ID
        if (payload.getRelateDocId() != null) {
            update.set(qdo.relateDocId, payload.getRelateDocId());
        }
        // 来源单据类型
        if (payload.getRelateDocType() != null) {
            update.set(qdo.relateDocType, payload.getRelateDocType());
        }
        // 事由ID
        if (payload.getReasonId() != null) {
            update.set(qdo.reasonId, payload.getReasonId());
        }
        // 事由类型
        if (payload.getReasonType() != null) {
            update.set(qdo.reasonType, payload.getReasonType());
        }
        // 核算项目ID
        if (payload.getBusItemId() != null) {
            update.set(qdo.busItemId, payload.getBusItemId());
        }
        // 占用金额
        if (payload.getOccupyAmt() != null) {
            update.set(qdo.occupyAmt, payload.getOccupyAmt());
        }
        // 已使用金额
        if (payload.getUsedAmt() != null) {
            update.set(qdo.usedAmt, payload.getUsedAmt());
        }
        // 扩展字段1
        if (payload.getExtString1() != null) {
            update.set(qdo.extString1, payload.getExtString1());
        }
        // 扩展字段2
        if (payload.getExtString2() != null) {
            update.set(qdo.extString2, payload.getExtString2());
        }
        // 扩展字段3
        if (payload.getExtString3() != null) {
            update.set(qdo.extString3, payload.getExtString3());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 来源单据ID
            if (nullFields.contains("relateDocId")) {
                update.setNull(qdo.relateDocId);
            }
            // 来源单据类型
            if (nullFields.contains("relateDocType")) {
                update.setNull(qdo.relateDocType);
            }
            // 事由ID
            if (nullFields.contains("reasonId")) {
                update.setNull(qdo.reasonId);
            }
            // 事由类型
            if (nullFields.contains("reasonType")) {
                update.setNull(qdo.reasonType);
            }
            // 核算项目ID
            if (nullFields.contains("busItemId")) {
                update.setNull(qdo.busItemId);
            }
            // 占用金额
            if (nullFields.contains("occupyAmt")) {
                update.setNull(qdo.occupyAmt);
            }
            // 已使用金额
            if (nullFields.contains("usedAmt")) {
                update.setNull(qdo.usedAmt);
            }
            // 扩展字段1
            if (nullFields.contains("extString1")) {
                update.setNull(qdo.extString1);
            }
            // 扩展字段2
            if (nullFields.contains("extString2")) {
                update.setNull(qdo.extString2);
            }
            // 扩展字段3
            if (nullFields.contains("extString3")) {
                update.setNull(qdo.extString3);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据主键查询
     *
     * @param relateDocId 相关单据ID
     * @return 结果
     */
    public PmsBudgetDocRelateVO queryByRelateDocId(Long relateDocId) {
        JPAQuery<PmsBudgetDocRelateVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.relateDocId.eq(relateDocId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    @Transactional
    public long updateByRelateDocId(PmsBudgetDocRelatePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.relateDocId.eq(payload.getRelateDocId()));
        // 来源单据ID
        if (payload.getRelateDocId() != null) {
            update.set(qdo.relateDocId, payload.getRelateDocId());
        }
        // 来源单据类型
        if (payload.getRelateDocType() != null) {
            update.set(qdo.relateDocType, payload.getRelateDocType());
        }
        // 事由ID
        if (payload.getReasonId() != null) {
            update.set(qdo.reasonId, payload.getReasonId());
        }
        // 事由类型
        if (payload.getReasonType() != null) {
            update.set(qdo.reasonType, payload.getReasonType());
        }
        // 核算项目ID
        if (payload.getBusItemId() != null) {
            update.set(qdo.busItemId, payload.getBusItemId());
        }
        // 占用金额
        if (payload.getOccupyAmt() != null) {
            update.set(qdo.occupyAmt, payload.getOccupyAmt());
        }
        // 已使用金额
        if (payload.getUsedAmt() != null) {
            update.set(qdo.usedAmt, payload.getUsedAmt());
        }
        // 扩展字段1
        if (payload.getExtString1() != null) {
            update.set(qdo.extString1, payload.getExtString1());
        }
        // 扩展字段2
        if (payload.getExtString2() != null) {
            update.set(qdo.extString2, payload.getExtString2());
        }
        // 扩展字段3
        if (payload.getExtString3() != null) {
            update.set(qdo.extString3, payload.getExtString3());
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param relateDocIds 相关单据ID
     * @return 删除的行数
     */
    public long deleteByRelateDocId(List<Long> relateDocIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.relateDocId.in(relateDocIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param relateDocTypes 相关单据类型
     * @return 删除的行数
     */
    public long deleteByRelateDocType(List<String> relateDocTypes) {
        //不传当为全删
        JPADeleteClause deleter = jpaQueryFactory.delete(qdo);
        if (!ObjectUtils.isEmpty(relateDocTypes)) {
            deleter.where(qdo.relateDocType.in(relateDocTypes));
        }
        return deleter.execute();
    }

}

