package com.elitesland.tw.tw5.server.prd.pms.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalAccountPayload;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalWideSettlePayload;
import com.elitesland.tw.tw5.api.prd.cal.service.CalAccountService;
import com.elitesland.tw.tw5.api.prd.cal.service.CalWideSettleService;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgOrganizationService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.BuProjectPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.BuProjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.service.BuProjectService;
import com.elitesland.tw.tw5.api.prd.pms.vo.BuProjectVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSettingVO;
import com.elitesland.tw.tw5.api.prd.task.service.TaskCommonService;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CalAccTurTypeEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CalAccTypeEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CalSettleStatusEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.PmsReasonTypeEnum;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.ProjectStatusEnum;
import com.elitesland.tw.tw5.server.prd.pms.convert.BuProjectConvert;
import com.elitesland.tw.tw5.server.prd.pms.dao.BuProjectDAO;
import com.elitesland.tw.tw5.server.prd.pms.dao.BuProjectMembersDAO;
import com.elitesland.tw.tw5.server.prd.pms.entity.BuProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.BuProjectMembersDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.BuProjectRepo;
import com.elitesland.tw.tw5.server.prd.system.constant.SystemSettingsItemEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * BU项目管理
 *
 * @author carl
 * @date 2023-11-29
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class BuProjectServiceImpl extends BaseServiceImpl implements BuProjectService {

    private final BuProjectRepo buProjectRepo;
    private final BuProjectDAO buProjectDAO;
    private final CalAccountService calAccountService;
    private final CacheUtil cacheUtil;
    private final CalWideSettleService calWideSettleService;
    private final TransactionUtilService transactionUtilService;
    private final PrdOrgOrganizationService serviceOrg;
    private final BuProjectMembersDAO buProjectMembersDAO;
    private final TaskCommonService taskCommonService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BuProjectVO insertOrUpdate(BuProjectPayload payload) {
        //数据校验
        checkData(payload);

        BuProjectVO buProjectVO = buProjectDAO.queryByBuIdAndYear(payload.getBuId(), payload.getProjYear());
        if (buProjectVO != null) {
            if (payload.getId() == null || !payload.getId().equals(buProjectVO.getId())) {
                throw TwException.error("", "BU已存在" + payload.getProjYear() + "年经营项目");
            }
        }
        if (payload.getId() == null) {
            //生成项目编码
            String code = generateSeqNum("PMS_PROJECT");
            payload.setProjNo(code);
        }
        payload.setAutoReportFlag(0);
        payload.setDeliBuId(payload.getBuId());
        payload.setProjStatus(ProjectStatusEnum.ACTIVE.getCode());
        payload.setHwhtSyncFlag(0);
        BuProjectDO entityDo = BuProjectConvert.INSTANCE.toDo(payload);
        BuProjectDO save = buProjectRepo.save(entityDo);

        if (payload.getId() == null) {
            /**
             * 新建BU项目同时
             * 1.把bu内的所有人员添加为项目成员
             * 2.新建项目账户
             * 3.创建账户流水
             */
            //添加项目成员
            createMembers(save.getId(), payload.getBuId());
            //创建项目账户
            createAccount(save);
            //开启事务执行泛用当量结算单
            transactionUtilService.executeWithRunnable(() -> {
                createProjInitWideSettle(save);
            });
        }
        return BuProjectConvert.INSTANCE.toVo(save);
    }

    /**
     * 创建项目成员
     *
     * @param projId
     * @param orgId
     */
    void createMembers(Long projId, Long orgId) {
        List<PrdOrgEmployeeRefVO> prdOrgEmployeeRefVOS = serviceOrg.queryEmployeeRefList(orgId);
        List<BuProjectMembersDO> buProjectMembersDOs = new ArrayList<>();
        prdOrgEmployeeRefVOS.forEach(emp -> {
            BuProjectMembersDO membersDO = new BuProjectMembersDO();
            membersDO.setProjId(projId);
            membersDO.setResId(emp.getUserId());
            membersDO.setWorkbenchFlag(1);
            buProjectMembersDOs.add(membersDO);
        });
        if (buProjectMembersDOs.size() > 0) {
            buProjectMembersDAO.saveAll(buProjectMembersDOs);
            List<Long> userIds = buProjectMembersDOs.stream().map(BuProjectMembersDO::getResId).collect(Collectors.toList());
            //判断是否创建初始化任务包
            PrdSystemSettingVO settingVO = cacheUtil.getSystemSetting(SystemSettingsItemEnum.prj_member_init_task.getCode());
            if (settingVO != null && StringUtils.hasText(settingVO.getSettingValue()) && "true".equals(settingVO.getSettingValue())) {
                taskCommonService.batchCreatTaskPackage(PmsReasonTypeEnum.PROJ_BU.getCode(), projId, userIds, null);
            }
        }


    }

    /**
     * 创建项目账户
     *
     * @param entityDo
     */
    void createAccount(BuProjectDO entityDo) {
        CalAccountPayload accountPayload = new CalAccountPayload();
        accountPayload.setAuType(CalAccTypeEnum.BU_PROJ.getCode());
        accountPayload.setAuId(entityDo.getId());
        accountPayload.setLedgerName(entityDo.getProjName());
        accountPayload.setLedgerNo(entityDo.getProjNo());
        calAccountService.insert(accountPayload);
    }

    /**
     * 项目立项创建泛用结算单
     *
     * @param projectDO
     */
    void createProjInitWideSettle(BuProjectDO projectDO) {
        CalWideSettlePayload settlePayload = new CalWideSettlePayload();
        settlePayload.setSettleStatus(CalSettleStatusEnum.FINISH.getCode());
        settlePayload.setSettleType(CalAccTurTypeEnum.PROJ_INIT.getCode());
        settlePayload.setSettleDate(LocalDate.now());
        settlePayload.setFromSourceType(CalAccTypeEnum.BU.getCode());
        settlePayload.setFromSourceId(projectDO.getDeliBuId());
        settlePayload.setFromSourceName(cacheUtil.getOrgName(projectDO.getDeliBuId()));
        settlePayload.setToSourceType(CalAccTypeEnum.BU_PROJ.getCode());
        settlePayload.setToSourceId(projectDO.getId());
        settlePayload.setToSourceName(projectDO.getProjName());
        settlePayload.setApplySettleEqva(projectDO.getTotalEqva());
        settlePayload.setSettlePrice(projectDO.getEqvaPrice());
        settlePayload.setApplySettleAmt(projectDO.getTotalEqva().multiply(projectDO.getEqvaPrice()));
        settlePayload.setIsSubmit(1);
        calWideSettleService.insertOrUpdate(settlePayload);
    }

    /**
     * 数据校验
     */
    void checkData(BuProjectPayload payload) {
        if (payload.getBuId() == null) {
            throw TwException.error("", "BU不可为空");
        }

        if (payload.getProjYear() == null) {
            throw TwException.error("", "年度不可为空");
        }
        if (payload.getPmResId() == null) {
            throw TwException.error("", "项目经理不可为空");
        }

        if (payload.getDeliUserId() == null) {
            throw TwException.error("", "交付负责人不可为空");
        }
        if (payload.getEqvaPrice() == null) {
            throw TwException.error("", "当量预估单价不可为空");
        }
        if (payload.getTotalEqva() == null || payload.getTotalEqva().compareTo(BigDecimal.ZERO) < 0) {
            throw TwException.error("", "预计总当量需大于等于0");
        }
        if (payload.getTotalReimbursement() == null || payload.getTotalReimbursement().compareTo(BigDecimal.ZERO) < 0) {
            throw TwException.error("", "预计总费用需大于等于0");
        }
    }

    @Override
    public PagingVO<BuProjectVO> queryPaging(BuProjectQuery query) {
        operPermissionFlag(query);
        PagingVO<BuProjectVO> buProjectVOPagingVO = buProjectDAO.queryPaging(query);
        if (buProjectVOPagingVO.getTotal() > 0) {
            buProjectVOPagingVO.getRecords().forEach(vo -> vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId())));
        }
        return buProjectVOPagingVO;
    }

    /**
     * 权限处理
     *
     * @param query
     */
    void operPermissionFlag(BuProjectQuery query) {
        Long loginUserId = GlobalUtil.getLoginUserId();
        //需要处理权限
        query.setLoginUserId(loginUserId);
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.OPS.getCode()));
        query.setPermissionFlag(!rolePermission);
//        if (!rolePermission) {
//            Set<Long> longs = employeeService.querySubordinatesIdsByUserId(loginUserId);
//            query.setLowerUserIds(longs);
//
//        }
    }

    void operPermissionFlagForTimesheet(BuProjectQuery query) {
        Long loginUserId = GlobalUtil.getLoginUserId();
        //需要处理权限
        query.setLoginUserId(loginUserId);
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.OPS.getCode()));
        query.setPermissionFlagForTimesheet(!rolePermission);
//        if (!rolePermission) {
//            Set<Long> longs = employeeService.querySubordinatesIdsByUserId(loginUserId);
//            query.setLowerUserIds(longs);
//
//        }
    }

    @Override
    public List<BuProjectVO> queryListDynamic(BuProjectQuery query) {
        if (!StringUtils.hasText(query.getProjStatus())) {
            query.setProjStatus(ProjectStatusEnum.ACTIVE.getCode());
        }
        Long loginUserId = GlobalUtil.getLoginUserId();
        //需要处理权限
        query.setLoginUserId(loginUserId);
//        query.setProjYear(LocalDate.now().getYear());
//        operPermissionFlag(query);
//        query.setPermissionFlag(false);
        return buProjectDAO.queryListDynamic(query);
    }

    @Override
    public List<BuProjectVO> queryListDynamicSimple(BuProjectQuery query) {
        return buProjectDAO.queryListDynamicSimple(query);
    }

    @Override
    public List<BuProjectVO> listByMember(BuProjectQuery query) {
        if (!StringUtils.hasText(query.getProjStatus())) {
            query.setProjStatus(ProjectStatusEnum.ACTIVE.getCode());
        }
        Long loginUserId = GlobalUtil.getLoginUserId();
        //需要处理权限
        query.setLoginUserId(loginUserId);
        query.setProjYear(LocalDate.now().getYear());
        operPermissionFlagForTimesheet(query);
        return buProjectDAO.queryListDynamic(query);
    }

    @Override
    public BuProjectVO queryByKey(Long key) {
        return buProjectDAO.queryByKey(key);
    }

    @Transactional
    @Override
    public Long updateHwhtSyncFlag(List<Long> ids) {
        if (ObjectUtils.isEmpty(ids)) {
            log.warn("待更新数据id为空！");
            return -1L;
        }
        return buProjectDAO.updateHwhtSyncFlag(ids);
    }

//
//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public BuProjectVO update(BuProjectPayload payload) {
//        BuProjectDO entity = buProjectRepo.findById(payload.getId()).orElseGet(BuProjectDO::new);
//        Assert.notNull(entity.getId(), "不存在");
//        BuProjectDO entityDo = BuProjectConvert.INSTANCE.toDo(payload);
//        entity.copy(entityDo);
//        return BuProjectConvert.INSTANCE.toVo(buProjectRepo.save(entity));
//    }
//
//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public long updateByKeyDynamic(BuProjectPayload payload) {
//        BuProjectDO entity = buProjectRepo.findById(payload.getId()).orElseGet(BuProjectDO::new);
//        Assert.notNull(entity.getId(), "不存在");
//        long result = buProjectDAO.updateByKeyDynamic(payload);
//        return result;
//    }
//
//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public void deleteSoft(List<Long> keys) {
//        if (!keys.isEmpty()) {
//            buProjectDAO.deleteSoft(keys);
//        }
//    }

}
