package com.elitesland.tw.tw5.server.prd.salecon.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgOrganizationService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationRefVO;
import com.elitesland.tw.tw5.api.prd.personplan.payload.PersonPlanDtlPayload;
import com.elitesland.tw.tw5.api.prd.personplan.payload.PersonPlanPayload;
import com.elitesland.tw.tw5.api.prd.personplan.service.PersonPlanService;
import com.elitesland.tw.tw5.api.prd.personplan.vo.PersonPlanDtlVO;
import com.elitesland.tw.tw5.api.prd.personplan.vo.PersonPlanVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.SaleConExecConditionPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.OrgGrossProfitRateSettingQuery;
import com.elitesland.tw.tw5.api.prd.salecon.query.SaleConExecConditionQuery;
import com.elitesland.tw.tw5.api.prd.salecon.service.ConPurchaseDemandService;
import com.elitesland.tw.tw5.api.prd.salecon.service.OrgGrossProfitRateSettingService;
import com.elitesland.tw.tw5.api.prd.salecon.service.SaleConContractService;
import com.elitesland.tw.tw5.api.prd.salecon.service.SaleConExecConditionService;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConPurchaseDemandVO;
import com.elitesland.tw.tw5.api.prd.salecon.vo.OrgGrossProfitRateSettingVO;
import com.elitesland.tw.tw5.api.prd.salecon.vo.SaleConContractVO;
import com.elitesland.tw.tw5.api.prd.salecon.vo.SaleConExecConditionVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemRoleService;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.FunctionSelectionEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.personplan.constants.PersonPlanTypeEnum;
import com.elitesland.tw.tw5.server.prd.salecon.convert.SaleConExecConditionConvert;
import com.elitesland.tw.tw5.server.prd.salecon.dao.SaleConExecConditionDAO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.SaleConExecConditionDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.SaleConExecConditionRepo;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 销售合同执行情况管理
 *
 * @author sunxw
 * @date 2023-11-24
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class SaleConExecConditionServiceImpl extends BaseServiceImpl implements SaleConExecConditionService {

    private final ConPurchaseDemandService purchaseDemandService;
    private final SaleConContractService saleConContractService;
    private final CacheUtil cacheUtil;
    private final SaleConExecConditionRepo saleConExecConditionRepo;
    private final SaleConExecConditionDAO saleConExecConditionDAO;
    @Value("${tw5.workflow.enabled}")
    private Boolean workflow_enabled;

    private final WorkflowUtil workflowUtil;

    private final TransactionUtilService transactionUtilService;
    private final OrgGrossProfitRateSettingService orgGrossProfitRateSettingService;
    private final PrdSystemRoleService roleService;
    private final PrdOrgOrganizationService prdOrgOrganizationService;
    private final PersonPlanService personPlanService;

    @Override
    public PagingVO<SaleConExecConditionVO> queryPaging(SaleConExecConditionQuery query) {

        PagingVO<SaleConExecConditionVO> saleConExecConditionPaging = saleConExecConditionDAO.queryPaging(query);
        if (!CollectionUtils.isEmpty(saleConExecConditionPaging.getRecords())) {
            translateList(saleConExecConditionPaging.getRecords());
        }
        return saleConExecConditionPaging;
    }

    @Override
    public List<SaleConExecConditionVO> queryListDynamic(SaleConExecConditionQuery query) {
        List<SaleConExecConditionVO> saleConExecConditionVOS = saleConExecConditionDAO.queryListDynamic(query);
        if (!CollectionUtils.isEmpty(saleConExecConditionVOS)) {
            translateList(saleConExecConditionVOS);
        }
        return saleConExecConditionVOS;
    }

    @Override
    public SaleConExecConditionVO queryByKey(Long key) {
        SaleConExecConditionVO saleConExecConditionVO = saleConExecConditionDAO.queryByKey(key);
        if (saleConExecConditionVO != null) {

     //       Map<String, BigDecimal> stringBigDecimalMap = saleConContractService.extraAndDemandTotalAmt(saleConExecConditionVO.getContractId());
      //      saleConExecConditionVO.setSaleExpense(stringBigDecimalMap.get("ExtraAmt"));
            BigDecimal amt = saleConExecConditionVO.getAmt() == null ? BigDecimal.ZERO : saleConExecConditionVO.getAmt();   //合同总金额
            BigDecimal taxRate = saleConExecConditionVO.getTaxRate() == null ? BigDecimal.ZERO : saleConExecConditionVO.getTaxRate();   //税率
            BigDecimal notTaxAmt = amt.divide(BigDecimal.ONE.add(taxRate), 2, RoundingMode.HALF_UP);    //不含税总金额
            saleConExecConditionVO.setNotTaxAmt(notTaxAmt);
            translateList(Collections.singletonList(saleConExecConditionVO));
        }
        return saleConExecConditionVO;
    }

    @Override
    @Transactional
    public SaleConExecConditionVO insertOrUpdate(SaleConExecConditionPayload payload) {
        this.checkConExecCondition(payload);
        this.buildConExecCondition(payload);
        SaleConExecConditionDO entityDo = SaleConExecConditionConvert.INSTANCE.toDo(payload);
        if (ObjectUtils.isEmpty(payload.getId())) {
            SaleConExecConditionDO save = saleConExecConditionRepo.save(entityDo);
            //新增的时候增加合同预算资源规划
            SaleConContractVO saleConContractVO = saleConContractService.queryByKey(payload.getContractId());
            Long oppoId = saleConContractVO.getOppoId();
            if (oppoId != null) {
                PersonPlanVO personPlanVO = personPlanService.getByObjIdAndPlanType(oppoId, Arrays.asList(PersonPlanTypeEnum.DELIVERY.getCode(), PersonPlanTypeEnum.PRE_SALES.getCode()));
                if (personPlanVO != null) {
                    PersonPlanPayload personPlanPayload = new PersonPlanPayload();
                    BeanUtils.copyProperties(personPlanVO, personPlanPayload);
                    personPlanPayload.setId(null);
                    personPlanPayload.setVersion("0");
                    personPlanPayload.setPlanType(PersonPlanTypeEnum.BUDGET.getCode());
                    personPlanPayload.setObjId(saleConContractVO.getId());
                    personPlanPayload.setObjName(saleConContractVO.getName());
                    List<PersonPlanDtlVO> personPlanDtlVOList = personPlanVO.getPersonPlanDtlVOList();
                    if (!CollectionUtils.isEmpty(personPlanDtlVOList)) {
                        List<PersonPlanDtlPayload> personPlanDtlPayloadList = new ArrayList<>();
                        personPlanDtlVOList.stream().forEach(personPlanDtlVO -> {
                            PersonPlanDtlPayload personPlanDtlPayload = new PersonPlanDtlPayload();
                            BeanUtils.copyProperties(personPlanDtlVO, personPlanDtlPayload);
                            personPlanDtlPayload.setId(null);
                            personPlanDtlPayload.setPlanId(null);
                            personPlanDtlPayloadList.add(personPlanDtlPayload);
                        });
                        personPlanPayload.setPersonPlanDtlPayloadList(personPlanDtlPayloadList);
                    }
                    personPlanService.save(personPlanPayload);
                }
            }
            return SaleConExecConditionConvert.INSTANCE.toVo(save);
        } else {
            saleConExecConditionDAO.updateByKeyDynamic(payload);
            return SaleConExecConditionConvert.INSTANCE.toVo(entityDo);
        }
    }

    @Override
    @Transactional
    public SaleConExecConditionVO update(SaleConExecConditionPayload payload) {
        SaleConExecConditionDO entity = saleConExecConditionRepo.findById(payload.getId()).orElseGet(SaleConExecConditionDO::new);
        Assert.notNull(entity.getId(), "不存在");
        SaleConExecConditionDO entityDo = SaleConExecConditionConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return SaleConExecConditionConvert.INSTANCE.toVo(saleConExecConditionRepo.save(entity));
    }

    @Override
    @Transactional
    public long updateByKeyDynamic(SaleConExecConditionPayload payload) {
        SaleConExecConditionDO entity = saleConExecConditionRepo.findById(payload.getId()).orElseGet(SaleConExecConditionDO::new);
        Assert.notNull(entity.getId(), "不存在");
        return saleConExecConditionDAO.updateByKeyDynamic(payload);
    }

    @Override
    @Transactional
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            saleConExecConditionDAO.deleteSoft(keys);
        }
    }


    /**
     * 必填值校验
     *
     * @param payload
     * @return
     */
    private void checkConExecCondition(SaleConExecConditionPayload payload) {
        Assert.notNull(payload, "未传payload对象");
        // Assert.notNull(payload.getProjExpense(), "项目费用未填写");
        // Assert.notNull(payload.getConPriceLevel(), "合同价格评级未填写");
        // Assert.notNull(payload.getBuGross(), "部门毛利未填写");
        // Assert.notNull(payload.getEconomicEquiv(), "最经济交付当量未填写");
        // Assert.notNull(payload.getCalcEffectAmt(), "未传核算后有效额");
        Assert.notNull(payload.getContractId(), "合同id为空");
    }

    /**
     * 构建合同执行情况数据
     *
     * @param payload
     * @return
     */
    @Override
    public SaleConExecConditionPayload buildConExecCondition(SaleConExecConditionPayload payload) {
        SaleConContractVO conContractVO = saleConContractService.queryByKey(payload.getContractId());
        Long contractId = payload.getContractId();
        BigDecimal demandTotalAmt = this.getConPurchaseAmt(contractId);
        payload.setContractId(contractId);
        payload.setSignAmt(conContractVO.getAmt());
        payload.setTaxRate(conContractVO.getTaxRate());
        payload.setPurchaseAmt(demandTotalAmt);
        payload.setEffectConAmt(conContractVO.getEffectiveAmt());
        String conPriceLevel = payload.getConPriceLevel();
        if (StringUtils.hasText(conPriceLevel)) {
            String selectionValue = cacheUtil.transferSystemSelection(FunctionSelectionEnum.SaleConPriceLevel.getCode(), conPriceLevel);
            payload.setConPriceLevel(selectionValue);
        }
        payload.setStatisticDate(LocalDate.now());
        // 新增的时候计算毛利率 更新不处理
        if (payload.getId() == null) {
            //查询部门毛利率配置
            List<OrgGrossProfitRateSettingVO> orgGrossProfitRateSettingVOS = queryOrgGrossProfitRateSetting(conContractVO);

//            OrgGrossProfitRateSettingQuery orgGrossProfitRateSettingQuery = new OrgGrossProfitRateSettingQuery();
//            // 查询部门的所有上级
//            Long deliBuId = conContractVO.getDeliBuId();
//            List<PrdOrgOrganizationRefVO> prdOrgOrganizationRefVOS = prdOrgOrganizationService.queryAllFatherOrgByOrgId(deliBuId);
//            List<Long> orgIdList = prdOrgOrganizationRefVOS.stream().map(e -> e.getOrgId()).collect(Collectors.toList());
//            orgGrossProfitRateSettingQuery.setOrgIdList(orgIdList);
//            orgGrossProfitRateSettingQuery.setWorkType(conContractVO.getWorkType());
//            // 合同的创建时间>=配置的开始时间 <=配置的结束时间
//            orgGrossProfitRateSettingQuery.setStartDate(LocalDate.parse(conContractVO.getCreateTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"))));
//            orgGrossProfitRateSettingQuery.setEndDate(LocalDate.parse(conContractVO.getCreateTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"))));
//            List<OrgGrossProfitRateSettingVO> orgGrossProfitRateSettingVOS = orgGrossProfitRateSettingService.queryListDynamic(orgGrossProfitRateSettingQuery);
            if (!CollectionUtils.isEmpty(orgGrossProfitRateSettingVOS)) {
                payload.setBuGross(orgGrossProfitRateSettingVOS.get(0).getOrgGrossProfitRate() == null ? BigDecimal.ZERO : orgGrossProfitRateSettingVOS.get(0).getOrgGrossProfitRate());
                payload.setOverOrderGross(orgGrossProfitRateSettingVOS.get(0).getOrderGrossProfitRate() == null ? BigDecimal.ZERO : orgGrossProfitRateSettingVOS.get(0).getOrderGrossProfitRate());
            }
        }
        return payload;
    }

    /**
     * 查询合同过单毛利率配置
     *
     * @return
     */
    List<OrgGrossProfitRateSettingVO> queryOrgGrossProfitRateSetting(SaleConContractVO conContractVO) {
        //查询部门毛利率配置
        OrgGrossProfitRateSettingQuery orgGrossProfitRateSettingQuery = new OrgGrossProfitRateSettingQuery();
        // 查询部门的所有上级
        Long deliBuId = conContractVO.getDeliBuId();
        List<PrdOrgOrganizationRefVO> prdOrgOrganizationRefVOS = prdOrgOrganizationService.queryAllFatherOrgByOrgId(deliBuId);
        List<Long> orgIdList = prdOrgOrganizationRefVOS.stream().map(e -> e.getOrgId()).collect(Collectors.toList());
        orgGrossProfitRateSettingQuery.setOrgIdList(orgIdList);
        orgGrossProfitRateSettingQuery.setWorkType(conContractVO.getWorkType());
        // 合同的创建时间>=配置的开始时间 <=配置的结束时间
        orgGrossProfitRateSettingQuery.setStartDate(LocalDate.parse(conContractVO.getCreateTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"))));
        orgGrossProfitRateSettingQuery.setEndDate(LocalDate.parse(conContractVO.getCreateTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"))));
        List<OrgGrossProfitRateSettingVO> orgGrossProfitRateSettingVOS = orgGrossProfitRateSettingService.queryListDynamic(orgGrossProfitRateSettingQuery);

        return orgGrossProfitRateSettingVOS;
    }

    @Override
    public SaleConExecConditionVO queryByContractId(Long contractId) {
        SaleConExecConditionVO vo = saleConExecConditionDAO.queryByContractId(contractId);
        if (ObjectUtils.isEmpty(vo)) {
            return new SaleConExecConditionVO();
        }
        String conPriceLevel = vo.getConPriceLevel();
        String selectionValue = cacheUtil.getSystemSelectionValueByName(FunctionSelectionEnum.SaleConPriceLevel.getCode(), conPriceLevel);
        vo.setConPriceLevel(selectionValue);
        vo.setConPriceLevelDesc(conPriceLevel);
        return vo;
    }

    @Override
    public BigDecimal calcEffectAmt(SaleConExecConditionPayload payload) {
        SaleConContractVO conContractVO = saleConContractService.queryByKey(payload.getContractId());
        BigDecimal calcTaxRate = new BigDecimal(1).add(conContractVO.getTaxRate());
        BigDecimal demandTotalAmt = this.getConPurchaseAmt(payload.getContractId());
        //核算后有效额=含税总金额/（1+税率）-销售费用-项目费用-合同采购金额/(1+税率)
        if (payload.getSaleExpense() == null) {
            payload.setSaleExpense(BigDecimal.ZERO);
        }
        BigDecimal calcEffectAmt = conContractVO.getAmt().divide(calcTaxRate, 2, RoundingMode.HALF_UP).subtract(payload.getSaleExpense()).subtract(payload.getProjExpense());
        if (demandTotalAmt.compareTo(BigDecimal.ZERO) > 0) {
            calcEffectAmt = calcEffectAmt.subtract((demandTotalAmt.divide(calcTaxRate, 2, RoundingMode.HALF_UP)));
        }
        return calcEffectAmt;
    }

    @Override
    @Transactional
    public void saleConExecConditionApply(SaleConExecConditionVO vo, SaleConContractVO saleConContractVO) {
        Long id = null;
        if (vo == null || vo.getId() == null) {
            SaleConExecConditionPayload saleConExecConditionPayload = new SaleConExecConditionPayload();
            saleConExecConditionPayload.setContractId(saleConContractVO.getId());
            vo = insertOrUpdate(saleConExecConditionPayload);
            id = vo.getId();
        } else {
            id = vo.getId();
        }
        // 合同可能会撤回 撤回后合同预算如果走过流程 就不发起了
        if (vo.getProcInstId() == null) {
            ProcessInfo processInfo = new ProcessInfo();
            if (workflow_enabled) {
                HashMap variables = new HashMap();
                //流程参数
                // 交付BULeader
                Long deliBuLeader = prdOrgOrganizationService.queryManageIdById(saleConContractVO.getDeliBuId());
                variables.put("Activity_030x3nk", deliBuLeader);


                // 合同预算流程中【平台资源总监】逻辑调整，如下，其中需要注意资源经理的权限（能在详情中看到合同预算）需求来源：#8693
                // 资源经理
                List<OrgGrossProfitRateSettingVO> orgGrossProfitRateSettingVOS = queryOrgGrossProfitRateSetting(saleConContractVO);
                List<Long> collect;
                if (ObjectUtils.isEmpty(orgGrossProfitRateSettingVOS)) {
                    collect = roleService.queryUserIdByRoleCode(RoleEnum.PLAT_RESOURCE_DIRECTOR.getCode());
                } else {
                    collect = orgGrossProfitRateSettingVOS.stream().map(OrgGrossProfitRateSettingVO::getResManagerId).collect(Collectors.toList());
                }
                //  List<Long> planResourcelist = roleService.queryUserIdByRoleCode(RoleEnum.PLAT_RESOURCE_DIRECTOR.getCode());
                variables.put("Activity_1klbasn", collect);

                // 财务核定
                List<Long> finclist = roleService.queryUserIdByRoleCode(RoleEnum.PLAT_FIN_CHECK.getCode());
                variables.put("Activity_1qkxzp9", finclist);
                // CFO
                List<Long> cfolist = roleService.queryUserIdByRoleCode(RoleEnum.PLAT_CFO.getCode());
                variables.put("Activity_1lhuzn9", cfolist);

                // 合同金额是否大于50w
                Boolean amtFlag = false;
                if (saleConContractVO.getAmt() != null) {
                    if (saleConContractVO.getAmt().compareTo(new BigDecimal(500000)) > 0) {
                        amtFlag = true;
                    }
                }
                variables.put("amtFlag", amtFlag);

                //平台财务负责人
                List<Long> planFinlist = roleService.queryUserIdByRoleCode(RoleEnum.PLAT_FIN_PIC.getCode());
                variables.put("Activity_18pk6yw", planFinlist);


                //发起流程审批
                processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                        ProcDefKey.SALE_CON_BUDGET.name(),
                        saleConContractVO.getName() + "-合同预算流程",
                        id + "",
                        variables), deliBuLeader
                );
            }
            //流程启动成功后，回写业务表数据
            SaleConExecConditionPayload payload = new SaleConExecConditionPayload();
            payload.setProcInstId(processInfo.getProcInstId());
            payload.setId(id);
            payload.setProcInstStatus(processInfo.getProcInstStatus());
            payload.setSubmitTime(LocalDateTime.now());
            //开启事务执行修改，主要是修改审批状态
            transactionUtilService.executeWithRunnable(() -> {
                saleConExecConditionDAO.updateByKeyDynamic(payload);
            });
        }

    }

    @Override
    public PagingVO<SaleConExecConditionVO> queryWorkAgentFlagPaging(SaleConExecConditionQuery query) {
        Long loginUserId = GlobalUtil.getLoginUserId();
        query.setLoginUserId(loginUserId);
        query.setWorkAgentFlag(true);
        return saleConExecConditionDAO.queryWorkAgentFlagPaging(query);
    }

    @Override
    public Long queryWorkAgentFlagCount(Long userId) {
        SaleConExecConditionQuery query = new SaleConExecConditionQuery();
        query.setLoginUserId(userId);
        query.setWorkAgentFlag(true);
        return saleConExecConditionDAO.queryWorkAgentFlagCount(query);
    }

    /**
     * 获取合同采购金额
     *
     * @param contractId
     * @return
     */
    private BigDecimal getConPurchaseAmt(Long contractId) {
        //获取销售合同对应的采购(不含税)
        BigDecimal demandTotalAmt = BigDecimal.ZERO;
        ConPurchaseDemandVO conPurchaseDemandVO = purchaseDemandService.queryBySaleConId(contractId);
        if (!ObjectUtils.isEmpty(conPurchaseDemandVO) && !ObjectUtils.isEmpty(conPurchaseDemandVO.getConPurchaseDemandDVOS())) {
            demandTotalAmt = conPurchaseDemandVO.getConPurchaseDemandDVOS().stream()
                    .map(obj -> Optional.ofNullable(obj.getNotTaxAmt()).orElse(BigDecimal.ZERO))
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return demandTotalAmt;
    }

    // 翻译某些字段
    private void translateList(List<SaleConExecConditionVO> voList) {
        voList.stream().forEach(e -> {
            String conPriceLevel = e.getConPriceLevel();
            if (StringUtils.hasText(conPriceLevel)) {
                String selectionValue = cacheUtil.getSystemSelectionValueByName(FunctionSelectionEnum.SaleConPriceLevel.getCode(), conPriceLevel);
                e.setConPriceLevelDesc(selectionValue);
            }

        });
    }
}
