package com.elitesland.tw.tw5.server.yeedocref;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.elitesland.tw.tw5.api.common.funConfig.vo.BusinessTableFieldsVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.funConfig.dao.BusinessTableFieldsDAO;
import com.elitesland.tw.tw5.server.yeedoc.service.YeedocService;
import com.elitesland.tw.tw5.server.yeedocref.functions.YeedocDeleteListVO;
import com.google.common.collect.Lists;
import freemarker.cache.StringTemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

@Component
@RequiredArgsConstructor
@Slf4j
public class YeedocUtils {

    private Configuration cfg;

    private final Map<String, CommonFormulaFunction> functionMap;
    private final YeedocService yeedocService;
    private final BusinessTableFieldsDAO businessTableFieldsDAO;

    @PostConstruct
    public void initCfg() {
        // 创建FreeMarker配置实例
        this.cfg = new Configuration(Configuration.VERSION_2_3_31);

        // 设置模板加载器
        StringTemplateLoader stringLoader = new StringTemplateLoader();
        this.cfg.setTemplateLoader(stringLoader);

        // 设置默认编码
        this.cfg.setDefaultEncoding("UTF-8");

        // 注册自定义函数
        functionMap.forEach((key, value) -> this.cfg.setSharedVariable(key.replace(CommonFormulaFunction.SERVICE_NAME_PREFIX, ""), value));
//        this.cfg.setSharedVariable(TwYearFunction.FUNCTION_CODE, new TwYearFunction());
//        this.cfg.setSharedVariable("getYearMonth", new YeedocConfig.YeedocFormulaYearMonthFunction());
//        this.cfg.setSharedVariable("getBuName", new YeedocConfig.YeedocFormulaBuNameFunction());
    }

    /**
     * 计算存储到易稻壳的文件夹目录路径
     *
     * @param context
     * @param formula
     * @return
     * @throws IOException
     * @throws TemplateException
     */
    public String calculatePath(Map<String, Object> context, String formula) throws IOException, TemplateException {
        // 加载模板
        StringTemplateLoader stringLoader = (StringTemplateLoader) cfg.getTemplateLoader();
        stringLoader.putTemplate("myTemplate", formula);
        Template template = cfg.getTemplate("myTemplate");

        return FreeMarkerTemplateUtils.processTemplateIntoString(template, context);
    }

    /**
     * 保存易稻壳附件
     *
     * @param context 上下文信息
     * @param info    要处理的易稻壳文件信息
     * @return 最终存好的文件id字符串
     */
    public String saveYeedocFileByFieldConfig(Map<String, Object> context, String info) {
        if (info == null || info.isEmpty()) {
            return null;
        }
        JSONObject jsonObject = JSONObject.parseObject(info);
        JSONArray fileList = jsonObject.getJSONArray("fileList");
        String deleteList = jsonObject.getString("deleteList");
        //JSONArray deleteList = jsonObject.getJSONArray("deleteList");
        String tableName = jsonObject.getString("tableName");
        String fieldName = jsonObject.getString("fieldName");
        List<YeedocDeleteListVO> yeedocDeleteListVOS = JSONArray.parseArray(deleteList, YeedocDeleteListVO.class);
        // 已经新建过不需要处理的文件
        List<String> fileCodes = new ArrayList<>(fileList.stream().map(Object::toString).filter(it -> !it.contains("Ptcent_YiDoc_Item_WebApi_TempLoadFile")).toList());

        // 要去新建的文件
        List<String> tempFileCodes = fileList.stream().map(Object::toString).filter(it -> it.contains("Ptcent_YiDoc_Item_WebApi_TempLoadFile")).toList();
        if (!tempFileCodes.isEmpty()) {
            JSONObject configJsonObject = getYeedocFieldConfig(tableName, fieldName);
            // 保存易稻壳附件
            Map resultMap = saveYeedocFile(context,
                    configJsonObject.getString("formula"), configJsonObject.getString("libraryId"), tempFileCodes, yeedocDeleteListVOS);
            // 创建的新文件的id
            if(resultMap.containsKey("add")){
                Map<String, String> tempFileMap = (Map<String, String>) resultMap.get("add");
                List<String> newFileCodes = new ArrayList<>();
                tempFileCodes.stream().forEach(tmp->{
                    if(tempFileMap.containsKey(tmp)){
                        newFileCodes.add(tempFileMap.get(tmp));
                    }
                });
                // 新增的附件
                if (!CollectionUtils.isEmpty(newFileCodes)) {
                    fileCodes.addAll(newFileCodes);
                }
            }
            // 覆盖的附件
            if(resultMap.containsKey("cover")){
                List coverItemIdList = (List) resultMap.get("cover");
                if (!CollectionUtils.isEmpty(coverItemIdList)) {
                    fileCodes.addAll(coverItemIdList);
                }
            }
        }

        // 要去删除的易稻壳文件列表
        if (!CollectionUtils.isEmpty(yeedocDeleteListVOS)) {
            List<String> toDeleteList = yeedocDeleteListVOS.stream().map(YeedocDeleteListVO::getFileCode).filter(it -> !it.contains("Ptcent_YiDoc_Item_WebApi_TempLoadFile")).toList();
            //   List<String> toDeleteList = deleteList.stream().map(Object::toString).filter(it -> !it.contains("Ptcent_YiDoc_Item_WebApi_TempLoadFile")).toList();
            yeedocService.deleteItem(String.join(",", toDeleteList));
        }
        return String.join(",", fileCodes);
    }

    /**
     * 获取表属性配置中配置的字段的易稻壳存储配置
     *
     * @param tablename 表名
     * @param fieldName 字段名
     * @return 易稻壳存储配置，JSONObject(libraryId,formula)
     */
    public JSONObject getYeedocFieldConfig(String tablename, String fieldName) {
        BusinessTableFieldsVO businessTableFieldsVO = businessTableFieldsDAO.queryByTableNameAndFieldName(tablename, fieldName);
        if (ObjectUtils.isEmpty(businessTableFieldsVO)) {
            throw TwException.error("", "保存易稻壳附件时，businessTableFieldsVO不能为空!");
        }
        String showType = businessTableFieldsVO.getShowType();
        if (!"ConfigYeedocFile".equals(showType)) {
            throw TwException.error("", "保存易稻壳附件失败，【" + fieldName + "】 字段类型不是易稻壳附件类型!");
        }
        String showAttr = businessTableFieldsVO.getShowAttr();
        return JSONObject.parseObject(showAttr);
    }

    /**
     * 保存易稻壳附件
     *
     * @param context   上下文信息
     * @param formula   易稻壳路径表达式
     * @param libraryId 易稻壳文档库ID
     * @param cacheKeys 易稻壳临时上传文件ID集合
     * @return
     */
    public Map saveYeedocFile(Map<String, Object> context, String formula, String libraryId, List<String> cacheKeys, List<YeedocDeleteListVO> yeedocDeleteListVOS) {
        if (CollectionUtils.isEmpty(cacheKeys)) {
            throw TwException.error("", "保存易稻壳附件时，cacheKeys不能为空!");
        }
        // 获取当前登陆人易稻壳的token
        String authTokenByYdk = yeedocService.getYeedocJwt();
        String calculatePath = "";
        // 先将表达式解析为一个具体的路径
        try {
            // 解析表达式
            calculatePath = calculatePath(context, formula);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw TwException.error("", "保存易稻壳附件时，formula表达式解析失败!");
        }
        //通过路径去创建易稻壳文件夹
        String ydkFolderId = createYdkFolder(Collections.singletonList(calculatePath), authTokenByYdk, libraryId);
        //保存易稻壳文件
        Map resultMap = saveYdkFile(ydkFolderId, new HashSet<>(cacheKeys), authTokenByYdk, libraryId, yeedocDeleteListVOS);
        return resultMap;
    }

    /**
     * 创建文件夹（一个或者多个）
     *
     * @param pathArry
     * @param authTokenByYdk
     * @return 易道壳的文件夹的itemId
     */
    private String createYdkFolder(List<String> pathArry, String authTokenByYdk, String libraryId) {
        Map<String, Object> map = new HashMap<>();
        map.put("PathArry", pathArry);
        map.put("LibraryId", libraryId);
        log.info("[创建文件夹-易道壳开始]-map:{}", JSONObject.toJSONString(map));
        String resultStr = yeedocService.createFolder(map, authTokenByYdk);
        if (org.apache.commons.lang3.StringUtils.isEmpty(resultStr)) {
            log.info("[创建文件夹-易道壳]-map:{}", JSONObject.toJSONString(map));
            log.info("[创建文件夹-易道壳]-resultStr:{}", resultStr);
            throw TwException.error("", "创建文件夹失败!result = {" + resultStr + "}");
        }
        JSONObject resultJson = JSONObject.parseObject(resultStr);
        log.info("[创建文件夹-易道壳返回值]-resultJson:{}", resultJson);
        Object data = resultJson.get("Data");
        if (ObjectUtils.isEmpty(data)) {
            throw TwException.error("", "创建文件夹失败!result = {" + resultJson + "}");
        }
        JSONArray dataArray = JSONArray.parseArray(data.toString());
        String folderId = (String) JSONObject.parseObject(String.valueOf(dataArray.get(dataArray.size() - 1))).get("FolderId");
        if (!StringUtils.hasText(folderId)) {
            throw TwException.error("", "创建文件夹返回值folderId为空!");
        }
        return folderId;
    }


    /**
     * 最终保存附件
     *
     * @param ydkFolderId
     * @param cacheKeys
     * @return map(cacheKey, itemId)
     */
    private Map saveYdkFile(String ydkFolderId, Set<String> cacheKeys, String authTokenByYdk, String libraryId, List<YeedocDeleteListVO> yeedocDeleteListVOS) {
        Map<String, Object> map = new HashMap<>();
        Map resultMap = new HashMap();
        if (CollectionUtils.isEmpty(cacheKeys)) {
            return null;
        }
        map.put("FolderId", ydkFolderId);
        map.put("LibraryId", libraryId);
        map.put("CacheKey", cacheKeys);
        map.put("ItemList", Lists.newArrayList());
        map.put("AssociatedItemID", Lists.newArrayList());
        map.put("Description", "");
        log.info("[最终保存附件-易道壳开始]-map:{}", JSONObject.toJSONString(map));
        log.info("[最终保存附件-易道壳开始]-authTokenByYdk:{}", authTokenByYdk);
        String resultStr = yeedocService.newUploadFilesSave(map, authTokenByYdk);
        if (org.apache.commons.lang3.StringUtils.isEmpty(resultStr)) {
            log.info("[最终保存附件-易道壳]-resultStr:{}", resultStr);
            throw TwException.error("", "最终保存附件失败!result = {" + resultStr + "}");
        }
        JSONObject resultJson = JSONObject.parseObject(resultStr);
        log.info("[最终保存附件-易道壳返回值]-resultJson:{}", resultJson);
        Boolean isSuccess = resultJson.getBoolean("IsSuccess");
        String message = resultJson.getString("Message");
        // todo 易道壳返回的是重复文件名称的，目前重复名称的这种不处理
//        if ((!isSuccess) || (!"上传成功！".equals(message))) {
//            throw TwException.error("", "最终保存附件失败!result = {" + resultJson + "}");
//            if("存在同名文件或文件夹！".equals(message)){
//                coverYdkFile(map,authTokenByYdk,resultJson.getString("Data"),true);
//            }else{
//                throw TwException.error("", "最终保存附件失败!result = {" + resultJson + "}");
//            }
//        }
        if (!isSuccess) {
            throw TwException.error("", "最终保存附件失败!result = {" + resultJson + "}");
        }
        // 判断有没有重复的文件
        JSONArray jsonArray = resultJson.getJSONArray("Data");
        List<String> itemIdList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(jsonArray)) {
            // 调用易稻壳接口执行覆盖操作
            Map<String, Set> coverYdkFileMap = coverYdkFile(map, authTokenByYdk, resultJson.getString("Data"), true);
            // 删除覆盖的cacheKey
            Set cacheKeysSet = coverYdkFileMap.get("cacheKeySet");
            if (!CollectionUtils.isEmpty(cacheKeysSet)) {
                Iterator<String> iterator = cacheKeys.iterator();
                while (iterator.hasNext()) {
                    String cacheKey = iterator.next();
                    if (cacheKeysSet.contains(cacheKey)) {
                        iterator.remove();
                    }
                }
            }
            //去删除的deleteList中去查找覆盖的fileName
            Set fileNameSet = coverYdkFileMap.get("fileNameSet");
            List<YeedocDeleteListVO> notDeleteist = new ArrayList<>();
            yeedocDeleteListVOS.stream().forEach(yeedocDeleteListVO -> {
                if (fileNameSet.contains(yeedocDeleteListVO.getFileName())) {
                    itemIdList.add(yeedocDeleteListVO.getFileCode());
                    notDeleteist.add(yeedocDeleteListVO);
                }
            });
            // 将不需要删除的itemIdList从yeedocDeleteListVOS中剔除
            yeedocDeleteListVOS.removeAll(notDeleteist);
        }

        // 获取易道壳附件id
        if (!CollectionUtils.isEmpty(cacheKeys)) {
            Map<String, Object> getItemMap = new HashMap<>();
            getItemMap.put("CacheKeys", cacheKeys);
            log.info("[获取易道壳附件id-易道壳开始]-map:{}", JSONObject.toJSONString(getItemMap));
            log.info("[获取易道壳附件id-易道壳开始]-authTokenByYdk:{}", authTokenByYdk);
            String itemIdResult = yeedocService.getCaCheKeyItemId(getItemMap, authTokenByYdk);
            if (org.apache.commons.lang3.StringUtils.isEmpty(itemIdResult)) {
                log.info("[获取易道壳附件id-易道壳]-resultStr:{}", itemIdResult);
            }
            JSONObject itemIdResultJson = JSONObject.parseObject(itemIdResult);
            Object itemIdData = itemIdResultJson.get("Data");
            if (ObjectUtils.isEmpty(itemIdData)) {
                throw TwException.error("", "获取易道壳附件id失败!result = {" + itemIdResultJson + "}");
            }
            List<JSONObject> itemIdJsonObjectList = JSONArray.parseArray(itemIdData.toString()).toJavaList(JSONObject.class);
            Map<String, String> resultAddMap = itemIdJsonObjectList.stream().filter(Objects::nonNull).collect(Collectors.toMap(item -> item.getString("CacheKey"), item -> item.getString("ItemId")));
            // 新增的文件
            resultMap.put("add", resultAddMap);
        }
        // 覆盖的文件 因为覆盖的文件的itemId不变
        resultMap.put("cover", itemIdList);
        return resultMap;
    }

    /**
     * 如果重复覆盖原有的附件
     *
     * @param map            易稻壳首次上传附件的参数
     * @param authTokenByYdk 易稻壳的token
     * @param data           上一次保存返回的数据(上传附件如果附件重复会返回存在同名文件
     *                       并且在data中会将重复附件显示
     *                       如果要覆盖则将返回的数据处理一下重新给到易稻壳)
     * @param coverFlag      是否覆盖
     * @return map(cacheKey, itemId)
     */
    private Map<String, Set> coverYdkFile(Map map, String authTokenByYdk, String data, Boolean coverFlag) {
        if (!coverFlag) {
            return null;
        }
        Set fileNameSet = new HashSet();
        Set cacheKeySet = new HashSet();
        JSONArray jsonArray = JSONArray.parseArray(data);
        List<Map> itemList = new ArrayList();
        jsonArray.stream().forEach(item -> {
            Map itemMap = new HashMap();
            JSONObject jsonObject = JSONObject.parseObject(item.toString());
            itemMap.put("CacheKey", jsonObject.get("CacheKey"));
            itemMap.put("Id", jsonObject.get("Id"));
            itemMap.put("OperationType", 2);
            itemList.add(itemMap);
            cacheKeySet.add(jsonObject.get("CacheKey"));
            fileNameSet.add(jsonObject.get("Name"));
        });
        map.put("ItemList", itemList);
        log.info("[最终保存附件-易道壳开始]-map:{}", JSONObject.toJSONString(map));
        log.info("[最终保存附件-易道壳开始]-authTokenByYdk:{}", authTokenByYdk);
        String resultStr = yeedocService.newUploadFilesSave(map, authTokenByYdk);
        if (org.apache.commons.lang3.StringUtils.isEmpty(resultStr)) {
            log.info("[最终保存附件-易道壳]-resultStr:{}", resultStr);
            throw TwException.error("", "最终保存附件失败!result = {" + resultStr + "}");
        }
        JSONObject resultJson = JSONObject.parseObject(resultStr);
        log.info("[最终保存附件-易道壳返回值]-resultJson:{}", resultJson);
        Boolean isSuccess = resultJson.getBoolean("IsSuccess");
        // todo 易道壳返回的是重复文件名称的，目前重复名称的这种不处理
        if ((!isSuccess)) {
            throw TwException.error("", "最终保存附件失败!result = {" + resultJson + "}");
        }
        Map<String, Set> cacheKeyAndItemIdMap = new HashMap<>();
        cacheKeyAndItemIdMap.put("cacheKeySet", cacheKeySet);
        cacheKeyAndItemIdMap.put("fileNameSet", fileNameSet);
        return cacheKeyAndItemIdMap;
    }

    /**
     * 获取易稻壳附件详情 (以后前端不调用 改为后端调用)
     *
     * @param fileCodes 易稻壳文件ID集合
     * @return List<YeedocFileVO>
     */
    public List<YeedocFileVO> getYeedocFile(List<String> fileCodes) {
        String allItemsInfo = yeedocService.getAllItemsInfo(fileCodes);
        if (!StringUtils.hasText(allItemsInfo)) {
            log.info("[获取易道壳附件-易道壳]-resultStr:{}", allItemsInfo);
        }
        JSONObject allItemsInfotJson = JSONObject.parseObject(allItemsInfo);
        Object data = allItemsInfotJson.get("Data");
        if (ObjectUtils.isEmpty(data)) {
            throw TwException.error("", "获取易道壳附件失败!result = {" + allItemsInfotJson + "}");
        }
        List<YeedocFileVO> yeedocFileVO = JSONArray.parseArray(String.valueOf(data), YeedocFileVO.class);
        return yeedocFileVO;
    }


}
