package com.elitesland.tw.tw5.server.prd.crm.service;

import com.elitesland.tw.tw5.api.prd.crm.payload.CrmOperationPlanDetailMemberListPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmOperationPlanDetailMemberPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmOperationPlanDetailMemberQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOperationPlanDetailMemberService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmOperationPlanDetailMemberVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemLogService;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.ChangeFieldLogUtil;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.PrdSystemObjectEnum;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmOperationPlanDetailMemberConvert;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmOperationPlanDetailDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmOperationPlanDetailMemberDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmOperationPlanDetailMemberDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmOperationPlanDetailMemberRepo;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmOperationPlanDetailRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * 客户经营 -经营计划参与者
 *
 * @author duwh
 * @date 2022/11/21
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CrmOperationPlanDetailMemberServiceImpl implements CrmOperationPlanDetailMemberService {

    private final CrmOperationPlanDetailMemberRepo repo;
    private final CrmOperationPlanDetailRepo crmOperationPlanDetailRepo;
    private final PrdSystemLogService logService;
    private final CacheUtil cacheUtil;
    private final ChangeFieldLogUtil changeFieldLogUtil;
    private final JPAQueryFactory jpaQueryFactory;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmOperationPlanDetailMemberVO insert(CrmOperationPlanDetailMemberPayload payload) {
        check(payload);
        initDefaultValue(payload);
        CrmOperationPlanDetailMemberDO entityDo = CrmOperationPlanDetailMemberConvert.INSTANCE.toDo(payload);
        repo.save(entityDo);

        // 参与者的添加：（张三）添加了参与者（字段新值）
        final String createLog = changeFieldLogUtil.getFieldsCreateLog(entityDo, "添加了");
        logService.saveNewLog(entityDo.getPlanDetailId(),
            PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getCode(),
            createLog);
        return CrmOperationPlanDetailMemberConvert.INSTANCE.toVo(entityDo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<CrmOperationPlanDetailMemberVO> savaAll(CrmOperationPlanDetailMemberListPayload payload) {
        final List<CrmOperationPlanDetailMemberPayload> memberList = payload.getMemberList();
        List<CrmOperationPlanDetailMemberVO> list = new ArrayList<>();
        if (!CollectionUtils.isEmpty(memberList)) {
            memberList.forEach(crmOperationPlanDetailMemberPayload -> {
                list.add(insert(crmOperationPlanDetailMemberPayload));
            });
        }
        return list;
    }

    /**
     * 初始化默认值
     *
     * @param payload 有效载荷
     */
    private void initDefaultValue(CrmOperationPlanDetailMemberPayload payload) {
        if (!StringUtils.hasText(payload.getUserName())) {
            final String userName = cacheUtil.getUserName(payload.getUserId());
            payload.setUserName(userName);
            // TODO 用户头像
        }
        if (null == payload.getOperId()) {
            final CrmOperationPlanDetailDO operationPlanDetailDO = crmOperationPlanDetailRepo.getById(payload.getPlanDetailId());
            final Long operId = operationPlanDetailDO.getOperId();
            payload.setOperId(operId);
        }
    }

    /**
     * 检查
     *
     * @param payload
     */
    private void check(CrmOperationPlanDetailMemberPayload payload) {
        if (null == payload.getPlanDetailId()) {
            throw TwException.error("", "planDetailId不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmOperationPlanDetailMemberVO update(CrmOperationPlanDetailMemberPayload payload) {
        CrmOperationPlanDetailMemberDO entity = repo.findById(payload.getId()).orElseGet(CrmOperationPlanDetailMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");
        CrmOperationPlanDetailMemberDO entityLog = new CrmOperationPlanDetailMemberDO();
        BeanUtils.copyProperties(entity, entityLog);

        CrmOperationPlanDetailMemberDO entityDo = CrmOperationPlanDetailMemberConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        // 修改NULL值特殊处理
        final StringBuilder fieldsUpdateLog = changeFieldLogUtil.nullFieldsProcess(payload, entityLog, entity);
        final CrmOperationPlanDetailMemberDO save = repo.save(entity);


        fieldsUpdateLog.append(changeFieldLogUtil.getFieldsUpdateLog(entityDo, entityLog));
        if (StringUtils.hasText(fieldsUpdateLog)) {
            logService.saveNewLog(entityDo.getPlanDetailId(),
                PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getCode(),
                fieldsUpdateLog.toString());
        }
        return CrmOperationPlanDetailMemberConvert.INSTANCE.toVo(save);
    }

    @Override
    public CrmOperationPlanDetailMemberVO queryByKey(Long key) {
        CrmOperationPlanDetailMemberDO entity = repo.findById(key).orElseGet(CrmOperationPlanDetailMemberDO::new);
        Assert.notNull(entity.getId(), "不存在");
        final CrmOperationPlanDetailMemberVO crmCustomerOperationVO = CrmOperationPlanDetailMemberConvert.INSTANCE.toVo(entity);
        return crmCustomerOperationVO;
    }

    @Override
    public List<CrmOperationPlanDetailMemberVO> queryList(CrmOperationPlanDetailMemberQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.desc("createTime");
        query.defaultOrder(orderItem);
        return CrmOperationPlanDetailMemberConvert.INSTANCE.toVoList(repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder)));
    }

    @Override
    public PagingVO<CrmOperationPlanDetailMemberVO> paging(CrmOperationPlanDetailMemberQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.desc("createTime");
        query.defaultOrder(orderItem);
        Page<CrmOperationPlanDetailMemberDO> page = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        return PageUtil.toPageVo(page.map(CrmOperationPlanDetailMemberConvert.INSTANCE::toVo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {

            keys.stream().forEach(id -> {
                Optional<CrmOperationPlanDetailMemberDO> optional = repo.findById(id);
                if (!optional.isEmpty()) {
                    CrmOperationPlanDetailMemberDO entity = optional.get();
                    logService.saveNewLog(entity.getPlanDetailId(),
                        PrdSystemObjectEnum.CUSTOMER_OPERATION_PLAN_DETAIL.getCode(),
                        "删除了参与者 " + entity.getUserName());
                }
            });
            repo.deleteSoft(keys);
        }
    }

    /**
     * 根据经营计划主键id集合 逻辑删除数据
     *
     * @param planDetailIds 经营计划主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoftByPlanDetailIds(List<Long> planDetailIds) {
        if (!planDetailIds.isEmpty()) {
            QCrmOperationPlanDetailMemberDO qdo = QCrmOperationPlanDetailMemberDO.crmOperationPlanDetailMemberDO;
            JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.planDetailId.in(planDetailIds));
            update.execute();
        }
    }

    /**
     * 根据客户经营主键id集合 逻辑删除数据
     *
     * @param operIds 客户经营主键
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoftByOperIds(List<Long> operIds) {
        if (!operIds.isEmpty()) {
            QCrmOperationPlanDetailMemberDO qdo = QCrmOperationPlanDetailMemberDO.crmOperationPlanDetailMemberDO;
            JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.operId.in(operIds));
            update.execute();
        }
    }

    @Override
    public void saveAll(List<CrmOperationPlanDetailMemberPayload> planDetailPayloadS) {
        List<CrmOperationPlanDetailMemberDO> list = CrmOperationPlanDetailMemberConvert.INSTANCE.toDoList(planDetailPayloadS);
        repo.saveAll(list);
    }

}
