package com.elitesland.tw.tw5.server.prd.inv.service;


import com.baiwang.open.entity.request.ImageInvoicesQueryRequest;
import com.baiwang.open.entity.response.ImageInvoicesQueryResponse;
import com.baiwang.open.entity.response.node.ImageInvoicesQuery;
import com.baiwang.open.entity.response.node.ImageInvoicesQueryMediaInvoiceAttachedISP;
import com.baiwang.open.entity.response.node.ImageInvoicesQueryMediaInvoiceDetailISP;
import com.baiwang.open.entity.response.node.ImageInvoicesQueryMediaInvoiceTravelISP;
import com.baiwang.open.exception.BWOpenException;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.tw.tw5.api.prd.inv.payload.InvInvoicePayload;
import com.elitesland.tw.tw5.api.prd.inv.query.InvInvoiceQuery;
import com.elitesland.tw.tw5.api.prd.inv.service.InvInvoiceService;
import com.elitesland.tw.tw5.api.prd.inv.service.TwInvoiceSendMsgService;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvInvoiceVO;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgEmployeeService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.inv.common.InvoiceStateEnum;
import com.elitesland.tw.tw5.server.prd.inv.common.InvoiceTypeEnum;
import com.elitesland.tw.tw5.server.prd.inv.convert.InvInvoiceAttachedConvert;
import com.elitesland.tw.tw5.server.prd.inv.convert.InvInvoiceConvert;
import com.elitesland.tw.tw5.server.prd.inv.convert.InvInvoiceDetailConvert;
import com.elitesland.tw.tw5.server.prd.inv.convert.InvInvoiceTravelItineraryConvert;
import com.elitesland.tw.tw5.server.prd.inv.dao.InvInvoiceAttachedDAO;
import com.elitesland.tw.tw5.server.prd.inv.dao.InvInvoiceDAO;
import com.elitesland.tw.tw5.server.prd.inv.dao.InvInvoiceDetailDAO;
import com.elitesland.tw.tw5.server.prd.inv.dao.InvInvoiceTravelItineraryDAO;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvInvoiceAttachedDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvInvoiceDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvInvoiceDetailDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvInvoiceTravelItineraryDO;
import com.elitesland.tw.tw5.server.prd.inv.repo.InvInvoiceRepo;
import com.querydsl.core.Tuple;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.DoubleStream;

/**
 * 发票主表
 *
 * @author zoey
 * @date 2023-12-05
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class InvInvoiceServiceImpl extends BaseServiceImpl implements InvInvoiceService {

    private final InvInvoiceRepo invInvoiceRepo;
    private final InvInvoiceDAO invInvoiceDAO;
    private final InvInvoiceDetailDAO invInvoiceDetailDAO;
    private final InvInvoiceAttachedDAO invInvoiceAttachedDAO;
    private final InvInvoiceTravelItineraryDAO invInvoiceTravelItineraryDAO;
    private final CacheUtil cacheUtil;
    private final PrdOrgEmployeeService employeeService;
    @Autowired
    private TwInvoiceSendMsgService invoiceSendMsgService;

    @Override
    public PagingVO<InvInvoiceVO> queryPaging(InvInvoiceQuery query) {
        //处理权限
//        handlePermission(query);
        PagingVO<InvInvoiceVO> invInvoiceVOPagingVO = invInvoiceDAO.queryPaging(query);
        List<InvInvoiceVO> records = invInvoiceVOPagingVO.getRecords();
        for (InvInvoiceVO record : records) {
            if (record.getInspectionStatus() == 0 && record.getInspectionErrorDesc().equals("暂不支持查验")) {
                record.setInspectionStatus(1);
            }
        }
        return invInvoiceVOPagingVO;
    }

    /**
     * 发票权限：只考虑全局权限和个人权限
     *
     * @param query
     */
    private void handlePermission(InvInvoiceQuery query) {
        //需要处理权限
        Long userId = GlobalUtil.getLoginUserId();
//        Long userId = 579786372043377650L;
        if (query.getPersonalInv() != null && query.getPersonalInv() == 1) {
            query.setInvOwnerId(userId);
            query.setCreateUserId(userId);
            query.setInvStateNot("INVALID");
        } else {
            // 通过前端参数判断权限
            Boolean isPermission = query.getIsPermission();
            if (isPermission) {
                //判断是否需要权限验证
                Boolean rolePermission = false;
                rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode(), RoleEnum.PLATFORM_RES.getCode()));
                if (!rolePermission) {
                    //需要处理权限
                    query.setInvOwnerId(userId);
                }
            } else {
                // 不限制权限
            }
        }
    }


    @Override
    public List<InvInvoiceVO> queryListDynamic(InvInvoiceQuery query) {

        List<InvInvoiceVO> invInvoiceVOS = invInvoiceDAO.queryListDynamic(query);
        for (InvInvoiceVO record : invInvoiceVOS) {
            if (record.getInspectionStatus() != null && record.getInspectionStatus() == 0 && record.getInspectionErrorDesc() != null && record.getInspectionErrorDesc().equals("暂不支持查验")) {
                record.setInspectionStatus(1);
            }
        }
        return invInvoiceVOS;
    }

    @Override
    public InvInvoiceVO queryByKey(Long key) {
        InvInvoiceDO entity = invInvoiceRepo.findById(key).orElseGet(InvInvoiceDO::new);
        Assert.notNull(entity.getId(), "不存在");
        InvInvoiceVO vo = InvInvoiceConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public InvInvoiceVO insert(InvInvoicePayload payload) {
        // 校验重复 发票代码+发票号码重复提示：“该发票已存在！”
        check(payload);
        InvInvoiceDO entityDo = InvInvoiceConvert.INSTANCE.toDo(payload);
        return InvInvoiceConvert.INSTANCE.toVo(invInvoiceRepo.save(entityDo));
    }

    private void check(InvInvoicePayload payload) {
        InvInvoiceQuery invInvoiceQuery = new InvInvoiceQuery();
        invInvoiceQuery.setInvoiceNo(payload.getInvoiceNo());

        invInvoiceQuery.setInvoiceCode(payload.getInvoiceCode());
        invInvoiceQuery.setInvType(payload.getInvType());
        // 按同一发票类型，发票号码，发票编码查重
        long invInvoiceCount = invInvoiceDAO.countCommon(invInvoiceQuery);
        if (invInvoiceCount > 0) {
            throw TwException.error("", "该发票已存在！");
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public InvInvoiceVO update(InvInvoicePayload payload) {
        InvInvoiceDO entity = invInvoiceRepo.findById(payload.getId()).orElseGet(InvInvoiceDO::new);
        Assert.notNull(entity.getId(), "不存在");
        InvInvoiceDO entityDo = InvInvoiceConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return InvInvoiceConvert.INSTANCE.toVo(invInvoiceRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(InvInvoicePayload payload) {
        InvInvoiceDO entity = invInvoiceRepo.findById(payload.getId()).orElseGet(InvInvoiceDO::new);
        Assert.notNull(entity.getId(), "不存在");
        long result = invInvoiceDAO.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            invInvoiceDAO.deleteSoft(keys);
        }
    }

    @Override
    public void getInvoicesFromBaiwang(Long userId) {
        SysUserDTO loginUser = GlobalUtil.getLoginUser();
        PrdOrgEmployeeVO employeeVO = employeeService.queryByUserId(loginUser.getId());
        String scanUserAccount = employeeVO.getBaiwangMobile() == null ? employeeVO.getMobile() : employeeVO.getBaiwangMobile();
        if (!StringUtils.hasText(scanUserAccount)) {
            throw TwException.error("", "登陆人手机号为空,拉取发票信息失败!");
        } else {
            if (ObjectUtils.isEmpty(employeeVO)) {
                throw TwException.error("", "无法查询到发票所有人的用户信息,拉取发票信息失败!");
            } else {
                String userAccount = employeeVO.getBaiwangMobile() == null ? employeeVO.getMobile() : employeeVO.getBaiwangMobile();
                if (!StringUtils.hasText(userAccount)) {
                    throw TwException.error("", "发票所有人手机号为空,拉取发票信息失败!");
                } else {
                    getInvoicesFromBaiwang("EL" + scanUserAccount, "EL" + userAccount, userId);
                }
            }
        }
    }

    @Override
    @Transactional
    public void addRelateReimByIds(List<InvInvoicePayload> payloads) {
        if (ObjectUtils.isEmpty(payloads)) {
            log.warn("报销单与发票待关联数据为空");
            return;
        }
        payloads.forEach(invInvoiceDAO::updateByKeyDynamic);
    }

    @Override
    @Transactional
    public void updateReimStatusByReimId(Long reimId, String reimStatus) {
        invInvoiceDAO.updateReimStatusByReimId(reimId, reimStatus);
    }

    @Override
    public Map<String, BigDecimal> findDeductTaxDeviceAmountTaxByNoIn(List<String> invoiceNos) {
        List<Tuple> tuples = invInvoiceDAO.findDeductTaxDeviceAmountTaxByNoIn(invoiceNos);
        if (CollectionUtils.isEmpty(tuples)) {
            return Collections.emptyMap();
        }
        Map<String, BigDecimal> resultMap = new HashMap<>(tuples.size());
        tuples.forEach(tuple -> {
            String invoiceCode = tuple.get(1, String.class) == null ? "" : tuple.get(1, String.class);
            String key = tuple.get(0, String.class) + "-" + invoiceCode;
            BigDecimal value = tuple.get(2, BigDecimal.class);
            value = (value != null) ? value : BigDecimal.ZERO;
            resultMap.put(key, value);
        });
        return resultMap;

    }

    @Override
    public void updateWrittenOffAmtByNo(Map<String, BigDecimal> newTheAmtMap, Map<String, BigDecimal> oldTheAmtMap) {
        // 要修改结果的Map（最终执行SQL的Map）
        Map<String, BigDecimal> resultTheAmtMap = new HashMap<>();
        // 提取删除的发票核销明细（删除的需要减去）
        Map<String, BigDecimal> deleteTheAmtMap = oldTheAmtMap.entrySet().stream()
            .filter(entry -> !newTheAmtMap.containsKey(entry.getKey()))
            .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        // 将新的核销明细添加到resultMap中
        newTheAmtMap.forEach((key, value) -> {
            value = value == null ? BigDecimal.ZERO : value;
            BigDecimal subTheAmt = oldTheAmtMap.get(key) == null ? BigDecimal.ZERO : oldTheAmtMap.get(key);
            resultTheAmtMap.put(key, value.subtract(subTheAmt));
        });
        // 将删除的核销明细添加到resultMap中
        if (!deleteTheAmtMap.isEmpty()) {
            deleteTheAmtMap.forEach((key, value) -> {
                value = value == null ? BigDecimal.ZERO : value;
                resultTheAmtMap.put(key, BigDecimal.ZERO.subtract(value));
            });
        }
        // 执行修改语句
        resultTheAmtMap.forEach(invInvoiceDAO::addWrittenOffAmtByInvoiceNo);
    }

    @Override
    public void subtractWrittenOffAmt(String invoiceNoAndCode, BigDecimal subTheAmt) {
        invInvoiceDAO.subWrittenOffAmtByInvoiceNo(invoiceNoAndCode, subTheAmt);
    }

    @Transactional(rollbackFor = Exception.class)
    int getInvoicesFromBaiwang(String scanUserAccount, String userAccount, Long userId) {
//        在新建报销单选择发票 弹出的窗体上增加一个 同步百望云发票 按钮，用户点了之后，先到Tw数据库获取该用户 之前最近的发票时间，然后到百望获取这个时间之后创建的 该用户的所有发票
//        获取之后 保存到TW发票池 如果单号已存在的发票 不执行保存即可
//        因为TW的发票可能已经被使用过了
//        每个报销的界面都维护一个 按钮

        ImageInvoicesQueryRequest request = new ImageInvoicesQueryRequest();
        request.setScanUserAccount(scanUserAccount);
        request.setUserAccount(userAccount);
        request.setIsDel(0);
        // 按照创建人查询，如果是发票所属人发生变更了呢？
        // TODO: 想办法把lastCteateTime查出为最早校验失败的发票
        LocalDateTime searchTime = LocalDateTime.now().minusDays(15);
        LocalDateTime lastCreateTime = invInvoiceRepo.getLastCreateTime(userId);
        LocalDateTime firstCheckFailTime = invInvoiceRepo.getFirstCheckFailTime(userId);
        if (firstCheckFailTime != null) {
            // 判断最新同步时间是不是大于了15天
            if (lastCreateTime.isBefore(searchTime)) {
                searchTime = lastCreateTime;
                // 判断最早查验失败时间是不是大于15天
            } else if (firstCheckFailTime.isBefore(searchTime)) {
                // 同步时间为15天
            } else {
                // 最后查验时间小于15天，同步时间是最早查验失败时间
                searchTime = firstCheckFailTime.minusMinutes(1);
            }
        } else {
            // 最后更新时间作为同步时间
            searchTime = lastCreateTime;
        }
        if (searchTime != null) {
            request.setCreateTimeStart(searchTime.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        }
        try {
            ImageInvoicesQueryResponse response = invoiceSendMsgService.imageInvoicesQuery(request);
            if (response != null) {
                List<ImageInvoicesQuery> reInvs = response.getResponse();
                if (!CollectionUtils.isEmpty(reInvs)) {
                    // 保存发票
                    for (ImageInvoicesQuery mainInfo : reInvs) {
                        // 将查询结果映射成发票
                        mainInfo.setCreateTime(null);
                        InvInvoiceDO invoiceDO = InvInvoiceConvert.INSTANCE.baiwangDoToDo(mainInfo);
                        InvInvoiceQuery invInvoiceQuery = new InvInvoiceQuery();
                        invInvoiceQuery.setInvoiceNo(mainInfo.getInvoiceNo());
                        if (mainInfo.getInvoiceCode() != null) {
                            invInvoiceQuery.setInvoiceCode(mainInfo.getInvoiceCode());
                        }
                        if (mainInfo.getInvoiceType() != null) {
                            invInvoiceQuery.setInvoiceType(mainInfo.getInvoiceType());
                        }
                        // 按同一发票类型，发票号码，发票编码查重
                        List<InvInvoiceVO> invInvoiceVOS = invInvoiceDAO.queryListDynamic(invInvoiceQuery);
//                        List<Long> existsItem = invInvoiceRepo.existsItem(mainInfo.getInvoiceNo(), mainInfo.getInvoiceCode());
                        // 有重复且是新建状态的发票才同步更新，否则不更新
                        if (!CollectionUtils.isEmpty(invInvoiceVOS)) {
                            InvInvoiceVO invInvoiceVO = invInvoiceVOS.get(0);
                            // 只针对查无此票的发票更新！！！！
                            if (invInvoiceVO.getInspectionStatus() == 2 && invInvoiceVO.getInvReimStatus().equals("NEW") && invInvoiceVO.getWrittenOffAmt() == null && invInvoiceVO.getReimDId() == null) {
                                invoiceDO.setId(invInvoiceVO.getId());
                                invoiceDO.setReimId(invInvoiceVO.getReimDId());//防止遗漏丢失发票数据
                                invoiceDO.setWrittenOffAmt(invInvoiceVO.getWrittenOffAmt());
                            } else {
                                continue;
                            }
                        }
                        invoiceDO.setBaiwangCreateTime(mainInfo.getCreateTime());
                        invoiceDO.setBaiwangUpdateTime(mainInfo.getUpdateTime());
                        String invType = mainInfo.getInvoiceType();//百旺发票类型的编码
                        InvoiceTypeEnum it = InvoiceTypeEnum.of(invType);
                        if (it != null && !it.getNeedInspect()) {
                            //定额发票等增值税专用发票不需要核验，核验状态设置为无需核验
                            invoiceDO.setInspectionStatus(5);
                        }
                        invoiceDO.setInvType(invType); // 发票类型
                        setInvoiceType(invoiceDO, mainInfo.getInvoiceType()); // 发票类型名称
                        setInvState(invoiceDO, mainInfo.getInvoiceState()); // 发票状态
                        invoiceDO.setInvReimStatus("NEW");
                        invoiceDO.setInvOwnerId(userId); // 归属人
                        //附件
                        invoiceDO.setImgContent(mainInfo.getFileAddress());
                        // 明细表
                        List<ImageInvoicesQueryMediaInvoiceDetailISP> invoiceDetailList = mainInfo.getInvoiceDetailList();
                        // 设置 最小-最大税率
                        if (invoiceDetailList != null) {
                            DoubleStream distinct = invoiceDetailList.stream().filter(i -> i.getTaxRate() != null)
                                .mapToDouble(i -> i.getTaxRate().doubleValue())
                                .distinct();
                            if (distinct.count() > 0) {
                                Double min = invoiceDetailList.stream().filter(i -> i.getTaxRate() != null)
                                    .mapToDouble(i -> i.getTaxRate().doubleValue())
                                    .distinct().min().getAsDouble();
                                Double max = invoiceDetailList.stream().filter(i -> i.getTaxRate() != null)
                                    .mapToDouble(i -> i.getTaxRate().doubleValue())
                                    .distinct().max().getAsDouble();
                                if (min.compareTo(max) == 0) {
                                    invoiceDO.setTaxRate(String.valueOf(((int) (min * 100)) + "%"));
                                } else {
                                    invoiceDO.setTaxRate(String.valueOf(((int) (min * 100)) + "%~" + ((int) (max * 100)) + "%"));
                                }
                            }
                        }
                        InvInvoiceDO save = invInvoiceDAO.save(invoiceDO);
                        // 明细表
                        if (invoiceDetailList != null) {
                            for (ImageInvoicesQueryMediaInvoiceDetailISP invoiceDetail : invoiceDetailList) {
                                // 将查询结果映射成明细发票
                                InvInvoiceDetailDO invInvoiceDetailDO = InvInvoiceDetailConvert.INSTANCE.baiwangDoToDo(invoiceDetail);
                                invInvoiceDetailDO.setInvId(save.getId());
                                invInvoiceDetailDO.setBaiwangId(invoiceDetail.getId());
                                invInvoiceDetailDO.setBaiwangInvId(mainInfo.getId());
                                invInvoiceDetailDAO.save(invInvoiceDetailDO);
                            }
                        }

                        // 扩展表
                        ImageInvoicesQueryMediaInvoiceAttachedISP invoiceAttached = mainInfo.getInvoiceAttached();
                        if (invoiceAttached != null) {
                            // 将查询结果映射成发票
                            InvInvoiceAttachedDO invoiceAttachedDO = InvInvoiceAttachedConvert.INSTANCE.baiwangDoToDo(invoiceAttached);
                            invoiceAttachedDO.setInvId(save.getId());
                            invoiceAttachedDO.setBaiwangId(invoiceAttached.getId());
                            invoiceAttachedDO.setBaiwangInvId(mainInfo.getId());
                            invInvoiceAttachedDAO.save(invoiceAttachedDO);
                        }
                        // 行程单表
                        List<ImageInvoicesQueryMediaInvoiceTravelISP> invoiceTravelList = mainInfo.getInvoiceTravelList();
                        if (invoiceTravelList != null) {
                            for (ImageInvoicesQueryMediaInvoiceTravelISP flightEntity : invoiceTravelList) {
                                InvInvoiceTravelItineraryDO invoiceAttachedDO = InvInvoiceTravelItineraryConvert.INSTANCE.baiwangDoToDo(flightEntity);
                                invoiceAttachedDO.setInvId(save.getId());
                                invoiceAttachedDO.setBaiwangId(flightEntity.getId());
                                invoiceAttachedDO.setBaiwangInvId(mainInfo.getId());
                                invInvoiceTravelItineraryDAO.save(invoiceAttachedDO);
                            }
                        }
                    }
                }
            }
        } catch (BWOpenException e) {
            log.error("[从百望获取影像-票据列表信息失败(入发票池)],账号:", scanUserAccount, ",时间：", searchTime, ",response=", e);
            if (e.getSubCode().equals("9999")) {
                throw TwException.error("", e.getSubMessage() + ",请检查用户手机号是否与百旺账号一致！");
            } else {
                throw TwException.error("", e.getSubMessage());
            }
        }
        return 0;
    }

    /**
     * 根据百望返回的信息，修改发票的类型
     *
     * @param entity
     * @param invoiceType
     */
    private void setInvoiceType(InvInvoiceDO entity, String invoiceType) {
        String text = null;
        InvoiceTypeEnum[] values = InvoiceTypeEnum.values();
        for (InvoiceTypeEnum invoiceTypeEnum : values) {
            if (invoiceTypeEnum.getCode().equals(invoiceType)) {
                text = invoiceTypeEnum.getDesc();
            }
        }
        entity.setInvoiceType(text);
    }

    /**
     * 根据百望返回的信息，修改发票的状态
     *
     * @param entity
     * @param getInvoiceState
     */
    private void setInvState(InvInvoiceDO entity, Integer getInvoiceState) {
        String text = null;
        InvoiceStateEnum[] values = InvoiceStateEnum.values();
        for (InvoiceStateEnum invoiceStateEnum : values) {
            if (invoiceStateEnum.getCode().equals(getInvoiceState + "")) {
                text = invoiceStateEnum.getUdcValue();
            }
        }
        entity.setInvState(text);
    }

    @Override
    public List<InvInvoiceVO> queryInvoiceNoByKey(String invocieNo) {
        return invInvoiceDAO.queryInvoiceNoByKey(invocieNo);
    }
}
