package com.elitesland.tw.tw5.server.prd.my.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.TBusinessRulesPayload;
import com.elitesland.tw.tw5.api.prd.my.query.TBusinessRulesQuery;
import com.elitesland.tw.tw5.api.prd.my.vo.TBusinessRulesVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.QTBusinessRulesDO;
import com.elitesland.tw.tw5.server.prd.my.entity.TBusinessRulesDO;
import com.elitesland.tw.tw5.server.prd.my.repo.TBusinessRulesRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

@Repository
@RequiredArgsConstructor
public class TBusinessRulesDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TBusinessRulesRepo repo;
    private final QTBusinessRulesDO qdo = QTBusinessRulesDO.tBusinessRulesDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TBusinessRulesVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TBusinessRulesVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 表达式类型
                qdo.proType,
                // 算法名称
                qdo.proName,
                // 算法编码
                qdo.proCode,
                // 算法规则表达式
                qdo.proExpression,
                // 算法是否启用
                qdo.enable
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TBusinessRulesVO> getJpaQueryWhere(TBusinessRulesQuery query) {
        JPAQuery<TBusinessRulesVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TBusinessRulesQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TBusinessRulesQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 表达式类型 精确 */
        if (!ObjectUtils.isEmpty(query.getProType())) {
            list.add(qdo.proType.eq(query.getProType()));
        }
        /** 算法名称 精确 */
        if (!ObjectUtils.isEmpty(query.getProName())) {
            list.add(qdo.proName.eq(query.getProName()));
        }
        /** 算法编码 精确 */
        if (!ObjectUtils.isEmpty(query.getProCode())) {
            list.add(qdo.proCode.eq(query.getProCode()));
        }
        /** 算法规则表达式 精确 */
        if (!ObjectUtils.isEmpty(query.getProExpression())) {
            list.add(qdo.proExpression.eq(query.getProExpression()));
        }
        /** 算法是否启用 精确 */
        if (!ObjectUtils.isEmpty(query.getEnable())) {
            list.add(qdo.enable.eq(query.getEnable()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TBusinessRulesVO queryByKey(Long id) {
        JPAQuery<TBusinessRulesVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TBusinessRulesVO> queryListDynamic(TBusinessRulesQuery query) {
        JPAQuery<TBusinessRulesVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TBusinessRulesVO> queryPaging(TBusinessRulesQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<TBusinessRulesVO> jpaQuery = getJpaQueryWhere(query);
        List<TBusinessRulesVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TBusinessRulesVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TBusinessRulesDO save(TBusinessRulesDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TBusinessRulesDO> saveAll(List<TBusinessRulesDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TBusinessRulesPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 表达式类型
        if (payload.getProType() != null) {
            update.set(qdo.proType, payload.getProType());
        }
        // 算法名称
        if (payload.getProName() != null) {
            update.set(qdo.proName, payload.getProName());
        }
        // 算法编码
        if (payload.getProCode() != null) {
            update.set(qdo.proCode, payload.getProCode());
        }
        // 算法规则表达式
        if (payload.getProExpression() != null) {
            update.set(qdo.proExpression, payload.getProExpression());
        }
        // 算法是否启用
        if (payload.getEnable() != null) {
            update.set(qdo.enable, payload.getEnable());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 表达式类型
            if (nullFields.contains("proType")) {
                update.setNull(qdo.proType);
            }
            // 算法名称
            if (nullFields.contains("proName")) {
                update.setNull(qdo.proName);
            }
            // 算法编码
            if (nullFields.contains("proCode")) {
                update.setNull(qdo.proCode);
            }
            // 算法规则表达式
            if (nullFields.contains("proExpression")) {
                update.setNull(qdo.proExpression);
            }
            // 算法是否启用
            if (nullFields.contains("enable")) {
                update.setNull(qdo.enable);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}