package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectReportPlanPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectReportPlanQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectReportPlanVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectReportPlanDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectReportPlanDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectReportPlanRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目汇报计划管理
 *
 * @author carl
 * @date 2023-09-05
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectReportPlanDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectReportPlanRepo repo;
    private final QPmsProjectReportPlanDO qdo = QPmsProjectReportPlanDO.pmsProjectReportPlanDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectReportPlanVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectReportPlanVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 项目id
                qdo.projId,
                // 期间
                qdo.periodDate,
                // 金额
                qdo.amt,
                // 项目汇报id
                qdo.briefId,
                //汇报编号
                qdo.briefNo
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectReportPlanVO> getJpaQueryWhere(PmsProjectReportPlanQuery query) {
        JPAQuery<PmsProjectReportPlanVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectReportPlanQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectReportPlanQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        }
        /** 项目ids 精确 */
        if (!ObjectUtils.isEmpty(query.getProjIds())) {
            list.add(qdo.projId.in(query.getProjIds()));
        }
        /** 期间 精确 */
        if (!ObjectUtils.isEmpty(query.getPeriodDate())) {
            list.add(qdo.periodDate.eq(query.getPeriodDate()));
        }
        /** 金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAmt())) {
            list.add(qdo.amt.eq(query.getAmt()));
        }
        /** 项目汇报id 精确 */
        if (!ObjectUtils.isEmpty(query.getBriefId())) {
            list.add(qdo.briefId.eq(query.getBriefId()));
        }
        if (query.getStartDate() != null && query.getEndDate() != null) {
            list.add(qdo.periodDate.between(query.getStartDate(), query.getEndDate()));
        } else if (query.getEndDate() != null) {
            list.add(qdo.periodDate.loe(query.getEndDate()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectReportPlanVO queryByKey(Long id) {
        JPAQuery<PmsProjectReportPlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectReportPlanVO> queryListDynamic(PmsProjectReportPlanQuery query) {
        JPAQuery<PmsProjectReportPlanVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectReportPlanVO> queryPaging(PmsProjectReportPlanQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectReportPlanVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectReportPlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectReportPlanVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectReportPlanDO save(PmsProjectReportPlanDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectReportPlanDO> saveAll(List<PmsProjectReportPlanDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectReportPlanPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 期间
        if (payload.getPeriodDate() != null) {
            update.set(qdo.periodDate, payload.getPeriodDate());
        }
        // 金额
        if (payload.getAmt() != null) {
            update.set(qdo.amt, payload.getAmt());
        }
        // 项目汇报id
        if (payload.getBriefId() != null) {
            update.set(qdo.briefId, payload.getBriefId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 期间
            if (nullFields.contains("periodDate")) {
                update.setNull(qdo.periodDate);
            }
            // 金额
            if (nullFields.contains("amt")) {
                update.setNull(qdo.amt);
            }
            // 项目汇报id
            if (nullFields.contains("briefId")) {
                update.setNull(qdo.briefId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

