package com.elitesland.tw.tw5.server.prd.system.dao;

import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemShortcutPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemShortcutQuery;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemShortcutVO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemShortcutDO;
import com.elitesland.tw.tw5.server.prd.system.entity.QPrdSystemShortcutDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemShortcutRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 系统快捷入口
 *
 * @author wangly
 * @date 2024-03-28
 */
@Repository
@RequiredArgsConstructor
public class PrdSystemShortcutDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PrdSystemShortcutRepo repo;
    private final QPrdSystemShortcutDO qdo = QPrdSystemShortcutDO.prdSystemShortcutDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemShortcutVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemShortcutVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 快捷入口名称
                qdo.shortcutName,
                // 快捷入口前端路由
                qdo.portalRoute,
                // 快捷入口图标
                qdo.shortcutIcon,
                // 快捷入口排序
                qdo.sortIndex,
                // 快捷入口状态
                qdo.shortcutStatus,
                // 类型
                qdo.type,
                // web类型 pc app
                qdo.webType,
                // 是否固定(固定后app页面不能移动)
                qdo.fixFlag
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemShortcutVO> getJpaQueryWhere(PrdSystemShortcutQuery query) {
        JPAQuery<PrdSystemShortcutVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, SqlUtil.getOrderse()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PrdSystemShortcutQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PrdSystemShortcutQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 快捷入口名称 精确 */
        if (!ObjectUtils.isEmpty(query.getShortcutName())) {
            list.add(qdo.shortcutName.like(SqlUtil.toSqlLikeString(query.getShortcutName())));
        }
        /** 快捷入口前端路由 精确 */
        if (!ObjectUtils.isEmpty(query.getPortalRoute())) {
            list.add(qdo.portalRoute.eq(query.getPortalRoute()));
        }
        /** 快捷入口图标 精确 */
        if (!ObjectUtils.isEmpty(query.getShortcutIcon())) {
            list.add(qdo.shortcutIcon.eq(query.getShortcutIcon()));
        }
        /** 快捷入口排序 精确 */
        if (!ObjectUtils.isEmpty(query.getSortIndex())) {
            list.add(qdo.sortIndex.eq(query.getSortIndex()));
        }
        /** 快捷入口状态 精确 */
        if (!ObjectUtils.isEmpty(query.getShortcutStatus())) {
            list.add(qdo.shortcutStatus.eq(query.getShortcutStatus()));
        }
        /** 类型 精确 */
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 类型 精确 */
        if (!ObjectUtils.isEmpty(query.getTypeList())) {
            list.add(qdo.type.in(query.getTypeList()));
        }
        /** web类型 pc app 精确 */
        if (!ObjectUtils.isEmpty(query.getWebType())) {
            list.add(qdo.webType.eq(query.getWebType()));
        }
        /** 是否固定(固定后app页面不能移动) 精确 */
        if (!ObjectUtils.isEmpty(query.getFixFlag())) {
            list.add(qdo.fixFlag.eq(query.getFixFlag()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdSystemShortcutVO queryByKey(Long id) {
        JPAQuery<PrdSystemShortcutVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdSystemShortcutVO> queryListDynamic(PrdSystemShortcutQuery query) {
        JPAQuery<PrdSystemShortcutVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdSystemShortcutVO> queryPaging(PrdSystemShortcutQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PrdSystemShortcutVO> jpaQuery = getJpaQueryWhere(query);
        List<PrdSystemShortcutVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PrdSystemShortcutVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdSystemShortcutDO save(PrdSystemShortcutDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdSystemShortcutDO> saveAll(List<PrdSystemShortcutDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PrdSystemShortcutPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 快捷入口名称
        if (payload.getShortcutName() != null) {
            update.set(qdo.shortcutName, payload.getShortcutName());
        }
        // 快捷入口前端路由
        if (payload.getPortalRoute() != null) {
            update.set(qdo.portalRoute, payload.getPortalRoute());
        }
        // 快捷入口图标
        if (payload.getShortcutIcon() != null) {
            update.set(qdo.shortcutIcon, payload.getShortcutIcon());
        }
        // 快捷入口排序
        if (payload.getSortIndex() != null) {
            update.set(qdo.sortIndex, payload.getSortIndex());
        }
        // 快捷入口状态
        if (payload.getShortcutStatus() != null) {
            update.set(qdo.shortcutStatus, payload.getShortcutStatus());
        }
        // 类型
        if (payload.getType() != null) {
            update.set(qdo.type, payload.getType());
        }
        // web类型 pc app
        if (payload.getWebType() != null) {
            update.set(qdo.webType, payload.getWebType());
        }
        // 是否固定(固定后app页面不能移动)
        if (payload.getFixFlag() != null) {
            update.set(qdo.fixFlag, payload.getFixFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 快捷入口名称
            if (nullFields.contains("shortcutName")) {
                update.setNull(qdo.shortcutName);
            }
            // 快捷入口前端路由
            if (nullFields.contains("portalRoute")) {
                update.setNull(qdo.portalRoute);
            }
            // 快捷入口图标
            if (nullFields.contains("shortcutIcon")) {
                update.setNull(qdo.shortcutIcon);
            }
            // 快捷入口排序
            if (nullFields.contains("sortIndex")) {
                update.setNull(qdo.sortIndex);
            }
            // 快捷入口状态
            if (nullFields.contains("shortcutStatus")) {
                update.setNull(qdo.shortcutStatus);
            }
            // 类型
            if (nullFields.contains("type")) {
                update.setNull(qdo.type);
            }
            // web类型 pc app
            if (nullFields.contains("webType")) {
                update.setNull(qdo.webType);
            }
            // 是否固定(固定后app页面不能移动)
            if (nullFields.contains("fixFlag")) {
                update.setNull(qdo.fixFlag);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

