/*******************************************************************************
 * $Header$
 * $Revision$
 * $Date$
 *
 *==============================================================================
 *
 * Copyright (c) 2001-2006 Primeton Technologies, Ltd.
 * All rights reserved.
 *
 * Created on 2010-11-15
 *******************************************************************************/


package com.elitesland.tw.tw5.server.common.util;

import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectCalendarPayload;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectCalendarService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectCalendarVO;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectWbsRelyVO;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectWbsVO;
import com.elitesland.tw.tw5.server.common.TwException;

import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.ProjectWbsTypeEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 自动排期计算
 *
 * @author carl
 */
@Slf4j
@RequiredArgsConstructor
public class AutoScheduling {
    private PmsProjectCalendarService calendarService;
    private List<PmsProjectWbsVO> pmsProjectWbsVOS;
    private List<PmsProjectWbsRelyVO> pmsProjectWbsRelyVOS;
    private LocalDate startDate;
    private Long projectId;

    public AutoScheduling(List<PmsProjectWbsVO> pmsProjectWbsVOS, List<PmsProjectWbsRelyVO> pmsProjectWbsRelyVOS, LocalDate startDate, PmsProjectCalendarService calendarService, Long projectId) {
        super();
        this.pmsProjectWbsVOS = pmsProjectWbsVOS;
        this.pmsProjectWbsRelyVOS = pmsProjectWbsRelyVOS;
        this.startDate = startDate;
        this.calendarService = calendarService;
        this.projectId = projectId;
    }

    /**
     * 1.先筛选出起始开口和末尾开口的wbs数据
     * 2.踢出不在网络中的wbs节点数据
     * 3.为开口节点设置最早开始和最早结束时间
     * 4.正推法为后置节点设置最早的开始和最晚时间
     * 5.逆推法计算节点最晚的开始和结束时间
     * 6.计算关键路径
     */

    public List<PmsProjectWbsVO> startScheduling() {
        if (ObjectUtils.isEmpty(this.pmsProjectWbsRelyVOS) || ObjectUtils.isEmpty(this.pmsProjectWbsVOS)) {
            throw TwException.error("wbs-101", "未识别到关系网络，请为活动、里程碑之间建立依赖关系以组成关系网络");
        }
        //基础wbs节点ids
        List<Long> wbsIds = this.pmsProjectWbsRelyVOS.stream().map(PmsProjectWbsRelyVO::getWbsId).distinct().collect(Collectors.toList());
        //前置依赖wbs节点ids
        List<Long> relyWbsIds = this.pmsProjectWbsRelyVOS.stream().map(PmsProjectWbsRelyVO::getWbsRelyId).distinct().collect(Collectors.toList());
        //踢出不在网络中的wbs节点数据
        this.pmsProjectWbsVOS = this.pmsProjectWbsVOS.stream().filter(wbsVO -> wbsIds.contains(wbsVO.getId()) || relyWbsIds.contains(wbsVO.getId())).collect(Collectors.toList());
        //初始化数据处理
        initializeData();
        //获取起始开口wbs数据
        List<PmsProjectWbsVO> startWbsVOS = this.pmsProjectWbsVOS.stream().filter(wbsVO -> !wbsIds.contains(wbsVO.getId())).collect(Collectors.toList());
        if (ObjectUtils.isEmpty(startWbsVOS)) {
            throw TwException.error("wbs-102", "无法获取起始开口数据，请核验！");
        }
        //获取末尾开口wbs数据
        List<PmsProjectWbsVO> endWbsVOS = this.pmsProjectWbsVOS.stream().filter(wbsVO -> !relyWbsIds.contains(wbsVO.getId())).collect(Collectors.toList());
        if (ObjectUtils.isEmpty(endWbsVOS)) {
            throw TwException.error("wbs-102", "无法获取结尾开口数据，请核验！");
        }
        //通知日期管理加载到缓存内
        List<Long> allWbsIds = this.pmsProjectWbsVOS.stream().map(PmsProjectWbsVO::getId).collect(Collectors.toList());
        calendarService.handleCalendarCache(allWbsIds, ProjectWbsTypeEnum.WBS.getCode(), "save", this.projectId);
        //为起始开口节点设置最早开始和最早结束时间
        startWbsVOS.forEach(startWbsVO -> {
            // 根据起始日期和工期获取结束日期
            PmsProjectCalendarVO byDurationDays = getCalendarPayload(startWbsVO.getId(), startWbsVO.getPreDurationDay().intValue(), this.startDate, null);

            if (byDurationDays != null) {
                startWbsVO.setEarlyStartDate(byDurationDays.getStartDate());
                startWbsVO.setEarlyEndDate(byDurationDays.getEndDate());

                if (startWbsVO.getWbsType().equals(ProjectWbsTypeEnum.MS.getCode())) {
                    startWbsVO.setEarlyEndDate(byDurationDays.getStartDate());
                }
                //正推法计算最早的开始和结束时间
                sequentialMethod(startWbsVO);
            } else {
                throw TwException.error("", "获取日期数据异常，请联系管理员！");
            }
        });
        //获取最晚的最早结束时间
        PmsProjectWbsVO pmsProjectWbsVO = endWbsVOS.stream().max(Comparator.comparing(PmsProjectWbsVO::getEarlyEndDate)).get();
        LocalDate earlyEndDate = pmsProjectWbsVO.getEarlyEndDate();

        //为末端开口节点设置最晚的开始和结束时间
        endWbsVOS.forEach(endWbsVO -> {
            // 根据最晚的结束时间和工期获取最晚的开始时间
            PmsProjectCalendarVO byDurationDays = getCalendarPayload(endWbsVO.getId(), endWbsVO.getPreDurationDay().intValue(), null, earlyEndDate);
            if (byDurationDays != null) {
                endWbsVO.setLateStartDate(byDurationDays.getStartDate());
                endWbsVO.setLateEndDate(byDurationDays.getEndDate());
                if (endWbsVO.getWbsType().equals(ProjectWbsTypeEnum.MS.getCode())) {
                    endWbsVO.setLateStartDate(byDurationDays.getEndDate());
                }
                //逆推法计算节点最晚的开始和结束时间
                inverseMethod(endWbsVO);
            } else {
                throw TwException.error("", "获取日期数据异常，请联系管理员！");
            }
        });

        //计算总浮时,关键活动,自由浮时
        this.pmsProjectWbsVOS.forEach(wbsVO1 -> {
            //TF = LS - ES = LF - EF
            PmsProjectCalendarVO byDurationDays = getCalendarPayload(wbsVO1.getId(), null, wbsVO1.getEarlyStartDate(), wbsVO1.getLateStartDate());
            //因为算的是工期，同一天返回工期也是1，所以这里要减去1
            int TF = byDurationDays.getDurationDays() - 1;
            PmsProjectCalendarVO byDurationDays0 = getCalendarPayload(wbsVO1.getId(), null, wbsVO1.getEarlyEndDate(), wbsVO1.getLateEndDate());
            int TF0 = byDurationDays0.getDurationDays() - 1;
//            long TF = wbsVO1.getEarlyStartDate().until(wbsVO1.getLateStartDate(), ChronoUnit.DAYS);
//            long TF0 = wbsVO1.getEarlyEndDate().until(wbsVO1.getLateEndDate(), ChronoUnit.DAYS);
            if (TF == TF0) {
                wbsVO1.setTotalFloat(new BigDecimal(TF));
                wbsVO1.setIsKeyNode(0);
                if (TF == 0) {
                    wbsVO1.setIsKeyNode(1);
                }
            } else {
                System.out.println("wbsVO:::" + wbsVO1.toString());
                throw TwException.error("", "日期控件计算异常，请联系管理员！");
            }
            //获取所有后置依赖
            Integer FF = null;
            List<PmsProjectWbsRelyVO> afterVOs = this.pmsProjectWbsRelyVOS.stream().filter(wbsRelyVO -> wbsRelyVO.getWbsRelyId().equals(wbsVO1.getId())).collect(Collectors.toList());
            for (PmsProjectWbsRelyVO afterVO : afterVOs) {
                PmsProjectWbsVO first = this.pmsProjectWbsVOS.stream().filter(wbsVO -> wbsVO.getId().equals(afterVO.getWbsId())).findFirst().get();
                int localDay = 999999;
                //因为算的是工期，同一天返回工期也是1，所以一下要多减去1
                //F-S类型时，FF = MIN{ES(suc)} - EF -1
                if (afterVO.getRelyType().equals("FS")) {
                    PmsProjectCalendarVO durationDays = getCalendarPayload(wbsVO1.getId(), null, wbsVO1.getEarlyEndDate(), first.getEarlyStartDate());
                    localDay = durationDays.getDurationDays() - 2;
                }
                //2. S-S类型时，FF = MIN{ES(suc)} - ES
                if (afterVO.getRelyType().equals("SS")) {
                    PmsProjectCalendarVO durationDays = getCalendarPayload(wbsVO1.getId(), null, wbsVO1.getEarlyStartDate(), first.getEarlyStartDate());
                    localDay = durationDays.getDurationDays() - 1;
                }
                //3. F-F类型时，FF = MIN{EF(suc)} - EF
                if (afterVO.getRelyType().equals("FF")) {
                    PmsProjectCalendarVO durationDays = getCalendarPayload(wbsVO1.getId(), null, wbsVO1.getEarlyEndDate(), first.getEarlyEndDate());
                    localDay = durationDays.getDurationDays() - 1;
                }
                if (localDay == 999999) {
                    throw TwException.error("", "自由浮时计算异常，请联系管理员！");
                }
                if (FF == null) {
                    FF = localDay;
                } else {
                    if (FF.intValue() > localDay) {
                        FF = localDay;
                    }
                }
            }

            wbsVO1.setFreeFloat(FF == null ? null : BigDecimal.valueOf(FF));
            wbsVO1.setAutoScheduling(1);
            wbsVO1.setPreEndDate(wbsVO1.getEarlyEndDate());
            wbsVO1.setPreStartDate(wbsVO1.getEarlyStartDate());
        });
        //通知日期管理清除缓存内容
        calendarService.handleCalendarCache(allWbsIds, ProjectWbsTypeEnum.WBS.getCode(), "clean", this.projectId);

        return this.pmsProjectWbsVOS;
    }

    /**
     * 逆推法计算节点最晚的开始和结束时间
     *
     * @param afterVO
     */
    void inverseMethod(PmsProjectWbsVO afterVO) {
        //获取当前节点所有前置依赖
        List<PmsProjectWbsRelyVO> frontVOs = this.pmsProjectWbsRelyVOS.stream().filter(wbsRelyVO -> wbsRelyVO.getWbsId().equals(afterVO.getId())).collect(Collectors.toList());
        if (!ObjectUtils.isEmpty(frontVOs)) {
            frontVOs.forEach(frontVO -> {
                Optional<PmsProjectWbsVO> first = this.pmsProjectWbsVOS.stream().filter(wbsVO -> wbsVO.getId().equals(frontVO.getWbsRelyId())).findFirst();
                if (first.isPresent()) {
                    PmsProjectWbsVO pmsProjectWbsVO = first.get();
                    LocalDate lateEndDate = pmsProjectWbsVO.getLateEndDate();
                    LocalDate localLateEndDate = null;
                    LocalDate localLateStartDate = null;
                    LocalDate lateStartDate = pmsProjectWbsVO.getLateStartDate();

                    if (pmsProjectWbsVO.getWbsType().equals(ProjectWbsTypeEnum.MS.getCode())) {
                        //如果前置依赖是里程碑
                        //里程碑的后置依赖关系类型只能是FF，FS
                        if (frontVO.getRelyType().equals("FS")) {
                            localLateEndDate = afterVO.getLateStartDate();
                        }
                        if (frontVO.getRelyType().equals("FF")) {
                            localLateEndDate = afterVO.getLateEndDate();
                        }
                    } else {
                        Boolean isStartDate = false;
                        //F-S 类型的前置活动 LF(pre) = MIN{LS(suc)} - 1
                        if (frontVO.getRelyType().equals("FS")) {
                            localLateEndDate = afterVO.getLateStartDate().minusDays(1);
                        }
                        //S-S 类型的前置活动 LF(pre) = MIN{LS(suc)} + AD + 休息天数 - 1
                        if (frontVO.getRelyType().equals("SS")) {
                            localLateStartDate = afterVO.getLateStartDate();
                            // 根据工期和最晚的的开始日期算出最晚的结束日期和休假天数
                            PmsProjectCalendarVO byDurationDays = getCalendarPayload(pmsProjectWbsVO.getId(), pmsProjectWbsVO.getPreDurationDay().intValue(), afterVO.getLateStartDate(), null);
                            localLateEndDate = byDurationDays.getEndDate();
                            isStartDate = true;
                        }
                        //F-F 类型的前置活动 LF(pre) = MIN{LF(suc)}
                        if (frontVO.getRelyType().equals("FF")) {
                            localLateEndDate = afterVO.getLateEndDate();
                        }

                        //赋值最晚开日期
                        if (!isStartDate) {
                            // 根据工期和最晚结束日期算出最晚的开始日期和休假天数
                            PmsProjectCalendarVO byDurationDays = getCalendarPayload(pmsProjectWbsVO.getId(), pmsProjectWbsVO.getPreDurationDay().intValue(), null, localLateEndDate);
                            localLateStartDate = byDurationDays.getStartDate();
                            //因为最晚结束时间可能是周么，所以要重新赋值
                            localLateEndDate = byDurationDays.getEndDate();
                        }
                    }

                    //赋值最晚结束日期
                    if (lateEndDate == null) {
                        lateEndDate = localLateEndDate;
                        lateStartDate = localLateStartDate;
                    } else {
                        if (localLateEndDate.isBefore(lateEndDate)) {
                            lateEndDate = localLateEndDate;
                            lateStartDate = localLateStartDate;
                        }
                    }
                    pmsProjectWbsVO.setLateEndDate(lateEndDate);
                    pmsProjectWbsVO.setLateStartDate(lateStartDate);
                    if (pmsProjectWbsVO.getWbsType().equals(ProjectWbsTypeEnum.MS.getCode())) {
                        //如果是里程碑，则结束时间等于开始时间
                        pmsProjectWbsVO.setLateStartDate(lateEndDate);
                    }
                    //递归执行
                    inverseMethod(pmsProjectWbsVO);
                } else {
                    throw TwException.error("", "存在无效的关联网络【" + frontVO.getWbsRelyName() + "-" + frontVO.getWbsName() + "，请联系管理员！");
                }
            });
        }
    }

    /**
     * 正推法计算节点最早的开始和结束时间
     *
     * @param frontVO
     */
    void sequentialMethod(PmsProjectWbsVO frontVO) {
        //获取当前节点所有后置依赖
        List<PmsProjectWbsRelyVO> afterRelyVOs = this.pmsProjectWbsRelyVOS.stream().filter(wbsRelyVO -> wbsRelyVO.getWbsRelyId().equals(frontVO.getId())).collect(Collectors.toList());
        if (!ObjectUtils.isEmpty(afterRelyVOs)) {
            afterRelyVOs.forEach(afterRelyVO -> {
                Optional<PmsProjectWbsVO> first = this.pmsProjectWbsVOS.stream().filter(wbsVO -> wbsVO.getId().equals(afterRelyVO.getWbsId())).findFirst();
                if (first.isPresent()) {
                    PmsProjectWbsVO pmsProjectWbsVO = first.get();
                    //校验是否存在网络回路
                    checkLoop(frontVO, pmsProjectWbsVO, afterRelyVO);

                    LocalDate earlyStartDate = pmsProjectWbsVO.getEarlyStartDate();
                    LocalDate localEarlyStartDate = null;
                    LocalDate localEarlyEndDate = null;
                    LocalDate earlyEndDate = pmsProjectWbsVO.getEarlyEndDate();

                    if (pmsProjectWbsVO.getWbsType().equals(ProjectWbsTypeEnum.MS.getCode())) {

                        //如果后置依赖是里程碑
                        //里程碑的前置依赖关系类型只能是FF
                        if (afterRelyVO.getRelyType().equals("FF")) {
                            localEarlyEndDate = localEarlyStartDate = frontVO.getEarlyEndDate();
                        } else {
                            throw TwException.error("", "存在无效的关联网络【" + afterRelyVO.getWbsRelyName() + "-" + afterRelyVO.getWbsName() + "，请核验！");
                        }
                    } else {
                        Boolean isEndDate = false;
                        //F-S 类型的接续活动 ES = MAX{EF(pre)} + 1
                        if (afterRelyVO.getRelyType().equals("FS")) {
                            localEarlyStartDate = frontVO.getEarlyEndDate().plusDays(1);
                        }
                        //S-S 类型的接续活动 ES = MAX{ES(pre)}
                        if (afterRelyVO.getRelyType().equals("SS")) {
                            localEarlyStartDate = frontVO.getEarlyStartDate();
                        }
                        //F-F 类型的接续活动 ES= MAX{EF(pre)} - AD - 休息天数 + 1
                        if (afterRelyVO.getRelyType().equals("FF")) {
                            localEarlyEndDate = frontVO.getEarlyEndDate();
                            // 根据工期和最早的结束日期算出最早开始日期和休假天数
                            PmsProjectCalendarVO byDurationDays = getCalendarPayload(pmsProjectWbsVO.getId(), pmsProjectWbsVO.getPreDurationDay().intValue(), null, frontVO.getEarlyEndDate());
                            localEarlyStartDate = byDurationDays.getStartDate();

                            isEndDate = true;
                        }

                        //赋值最早结束日期
                        if (!isEndDate) {
                            // 根据工期和开始日期算出结束日期和休假天数
                            PmsProjectCalendarVO byDurationDays = getCalendarPayload(pmsProjectWbsVO.getId(), pmsProjectWbsVO.getPreDurationDay().intValue(), localEarlyStartDate, null);
                            localEarlyEndDate = byDurationDays.getEndDate();
                            //因为最早的开始日期可能是休息日，所以要重新赋值查询结果
                            localEarlyStartDate = byDurationDays.getStartDate();
                        }
                    }

                    //赋值最早开始日期
                    if (earlyStartDate == null) {
                        earlyStartDate = localEarlyStartDate;
                        earlyEndDate = localEarlyEndDate;
                    } else {
                        if (earlyStartDate.isBefore(localEarlyStartDate)) {
                            earlyStartDate = localEarlyStartDate;
                            earlyEndDate = localEarlyEndDate;
                        }
                    }
                    pmsProjectWbsVO.setEarlyStartDate(earlyStartDate);
                    pmsProjectWbsVO.setEarlyEndDate(earlyEndDate);
                    if (pmsProjectWbsVO.getWbsType().equals(ProjectWbsTypeEnum.MS.getCode())) {
                        //如果是里程碑，则结束时间等于开始时间
                        pmsProjectWbsVO.setEarlyEndDate(earlyStartDate);
                    }
                    //递归执行
                    sequentialMethod(pmsProjectWbsVO);
                } else {
                    throw TwException.error("", "存在无效的关联网络【" + afterRelyVO.getWbsRelyName() + "-" + afterRelyVO.getWbsName() + "，请联系管理员！");
                }
            });
        }
    }

    /**
     * 检验是否存在网络回路
     *
     * @param frontVO         前置wbs
     * @param pmsProjectWbsVO 当前wbs
     * @param relyVO          前置关系
     */
    void checkLoop(PmsProjectWbsVO frontVO, PmsProjectWbsVO pmsProjectWbsVO, PmsProjectWbsRelyVO relyVO) {
        //先赋值所有的前置依赖frontIds
        String frontIds = "";
        if (StringUtils.hasText(frontVO.getFrontWbsIds())) {
            frontIds += frontVO.getFrontWbsIds() + ",";
        }
        frontIds += frontVO.getId();

        String frontRelyIds = "";
        if (StringUtils.hasText(frontVO.getFrontRelyIds())) {
            frontRelyIds += frontVO.getFrontRelyIds() + ",";
        }
        frontRelyIds += relyVO.getId();

        pmsProjectWbsVO.setFrontWbsIds(frontIds);
        pmsProjectWbsVO.setFrontRelyIds(frontRelyIds);
        //判断我的所有前置依赖关系中是否包含自己
        if (pmsProjectWbsVO.getFrontWbsIds().indexOf(pmsProjectWbsVO.getId() + "") >= 0) {

            //返回前端错误的回路数据
            String[] splitWbsIds = frontIds.split(",");
            int index = Arrays.binarySearch(splitWbsIds, pmsProjectWbsVO.getId() + "");
            String[] splitRelyIds = frontRelyIds.split(",");
            List<PmsProjectWbsRelyVO> relyVOS = new ArrayList<>();
            for (int i = index; i < splitRelyIds.length; i++) {
                Long relyId = Long.valueOf(splitRelyIds[i]);
                PmsProjectWbsRelyVO pmsProjectWbsRelyVO = this.pmsProjectWbsRelyVOS.stream().filter(relyVO1 -> relyVO1.getId().equals(relyId)).findFirst().get();
                PmsProjectWbsVO pmsProjectWbsVO1 = this.pmsProjectWbsVOS.stream().filter(wbsVO -> wbsVO.getId().equals(pmsProjectWbsRelyVO.getWbsId())).findFirst().get();
                PmsProjectWbsVO pmsProjectWbsVO2 = this.pmsProjectWbsVOS.stream().filter(wbsVO -> wbsVO.getId().equals(pmsProjectWbsRelyVO.getWbsRelyId())).findFirst().get();
                pmsProjectWbsRelyVO.setNodeCode(pmsProjectWbsVO1.getNodeCode());
                pmsProjectWbsRelyVO.setWbsName(pmsProjectWbsVO1.getWbsName());
                pmsProjectWbsRelyVO.setNodeRelyCode(pmsProjectWbsVO2.getNodeCode());
                pmsProjectWbsRelyVO.setWbsRelyName(pmsProjectWbsVO2.getWbsName());
                relyVOS.add(pmsProjectWbsRelyVO);
            }
            throw TwException.error("wbs-102", "", "存在网络回路，请核验！", relyVOS);
        }
    }

    /**
     * 获取日期控件
     *
     * @param wbsId        wbs组件
     * @param durationDays 工期
     * @param startDate    开始时间
     * @param endDate      结束时间
     * @return
     */
    PmsProjectCalendarVO getCalendarPayload(long wbsId, Integer durationDays, LocalDate startDate, LocalDate endDate) {
        PmsProjectCalendarPayload calendarPayload = new PmsProjectCalendarPayload();
        calendarPayload.setSourceId(wbsId);
        calendarPayload.setSourceType(ProjectWbsTypeEnum.WBS.getCode());
        calendarPayload.setStartDate(startDate);
        calendarPayload.setEndDate(endDate);
        calendarPayload.setDurationDays(durationDays);
        calendarPayload.setFromCacheFlag(true);//是true表示是从缓存中查询
        calendarPayload.setProjectId(this.projectId);
        PmsProjectCalendarVO byDurationDays = calendarService.findBySource(calendarPayload);
        if (byDurationDays == null) {
            throw TwException.error("", "日期控件不存在，请联系管理员！");
        }
        if (byDurationDays.getStartDate() == null || byDurationDays.getEndDate() == null) {
            throw TwException.error("", "日期控件数据返回异常，请联系管理员！");
        }
        return byDurationDays;
    }

    /**
     * 初始校验
     */
    void initializeData() {
        this.pmsProjectWbsVOS.forEach(pmsProjectWbsVO -> {
            if (pmsProjectWbsVO.getPreDurationDay() == null) {
                throw TwException.error("", "预计工期不存在，请核验！");
            }
            pmsProjectWbsVO.setEarlyStartDate(null);
            pmsProjectWbsVO.setEarlyEndDate(null);
            pmsProjectWbsVO.setLateStartDate(null);
            pmsProjectWbsVO.setLateEndDate(null);
            pmsProjectWbsVO.setFreeFloat(null);
            pmsProjectWbsVO.setTotalFloat(null);
            pmsProjectWbsVO.setDelayLag(null);
        });
    }

}  

