package com.elitesland.tw.tw5.server.prd.my.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.UserVacationPayload;
import com.elitesland.tw.tw5.api.prd.my.query.UserVacationQuery;
import com.elitesland.tw.tw5.api.prd.my.service.UserVacationService;
import com.elitesland.tw.tw5.api.prd.my.vo.UserVacationVO;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgEmployeeService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.server.common.ExcelUtil;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.my.convert.UserVacationConvert;
import com.elitesland.tw.tw5.server.prd.my.dao.UserVacationDAO;
import com.elitesland.tw.tw5.server.prd.my.entity.UserVacationDO;
import com.elitesland.tw.tw5.server.prd.my.repo.UserVacationRepo;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

@Service
@RequiredArgsConstructor
@Slf4j
public class UserVacationServiceImpl implements UserVacationService {

    private final UserVacationDAO userVacationDAO;

    private final UserVacationRepo repo;

    private final PrdOrgEmployeeService employeeService;

    private final CacheUtil cacheUtil;

    private final UdcUtil udcUtil;

    private final ExcelUtil excelUtil;

    @Override
    public void save(UserVacationPayload payload) {
        if (payload.getTotalDays().compareTo(payload.getUsedDays()) < 0) {
            throw TwException.error("", "可用天数不能大于总天数");
        }
        if (payload.getUserId() != null) {
            PrdOrgEmployeeVO employeeVO = cacheUtil.getEmployee(payload.getUserId());
            if (Objects.equals(employeeVO.getExtString6(), "EXTERNAL_RES")) {
                throw TwException.error("", "外部资源不可维护假期");
            }
        }
        userVacationDAO.save(UserVacationConvert.INSTANCE.payloadToDo(payload));
    }

    @Override
    public PagingVO<UserVacationVO> page(UserVacationQuery query) {
        PagingVO<UserVacationVO> pagingVO = userVacationDAO.queryPaging(query);
        if (pagingVO.getTotal() > 0) {
            List<UserVacationVO> userVacationList = pagingVO.getRecords();
            calculateAvailableDay(userVacationList);
            pagingVO.setRecords(userVacationList);
        }
        return pagingVO;
    }

    @Override
    public List<UserVacationVO> list(UserVacationQuery query) {
        List<UserVacationVO> userVacationList = userVacationDAO.list(query);
        if (!CollectionUtils.isEmpty(userVacationList)) {
            calculateAvailableDay(userVacationList);
        }
        return userVacationList;
    }

    @Override
    public void calculateAvailableDay(List<UserVacationVO> userVacationList) {
        userVacationList.forEach(userVacationVO -> {
            //年休需要计算冻结天数
            if ("ANNUAL".equals(userVacationVO.getVacationType()) || "ANNUAL_W".equals(userVacationVO.getVacationType())) {
                //入职日期

                BigDecimal totalDays = userVacationVO.getTotalDays();
                BigDecimal usedDays = userVacationVO.getUsedDays();
                LocalDate startDate = userVacationVO.getStartDate();
                LocalDate endDate = userVacationVO.getEndDate();

                BigDecimal workDays = new BigDecimal(0);
                LocalDate today = LocalDate.now();

                workDays = new BigDecimal(today.toEpochDay() - startDate.toEpochDay() + 1);
                if (today.isAfter(userVacationVO.getEndDate())) {
                    workDays = new BigDecimal(userVacationVO.getEndDate().toEpochDay() - startDate.toEpochDay() + 1);
                }
                BigDecimal actDays = new BigDecimal(endDate.toEpochDay() - startDate.toEpochDay() + 1);
                BigDecimal actYearDays = actDays.multiply(totalDays).divide(new BigDecimal(365), 0, RoundingMode.DOWN);
                BigDecimal availableDays = workDays.multiply(totalDays).divide(new BigDecimal(365), 0, RoundingMode.DOWN).subtract(usedDays);
                userVacationVO.setActYearDays(actYearDays);
                userVacationVO.setAvailableDays(availableDays);
                userVacationVO.setFrozenDay(actYearDays.subtract(availableDays).subtract(usedDays));
            } else {
                BigDecimal totalDays = userVacationVO.getTotalDays();
                BigDecimal usedDays = userVacationVO.getUsedDays();
                BigDecimal availableDays = totalDays.subtract(usedDays);
                userVacationVO.setAvailableDays(availableDays);
            }
        });
    }

    @Override
    public UserVacationVO queryById(Long id) {
        UserVacationDO userVacationDO = userVacationDAO.queryById(id);
        return UserVacationConvert.INSTANCE.doToVo(userVacationDO);
    }

    @Override
    @Transactional
    public void delete(Long[] ids) {
        userVacationDAO.deleteByIds(Arrays.asList(ids));
    }


    @Override
    @Transactional
    public void updateExpirationBatch(List<UserVacationPayload> payloadList) {
        for (UserVacationPayload payload : payloadList) {
            userVacationDAO.updateExpirationDate(payload);
        }
    }

    @Override
    public List<UserVacationVO> getAvailableVacation(Long userId, Long vacationId) {
        List<UserVacationVO> availableVacationVOList = userVacationDAO.getAvailableVacation(userId);
        calculateAvailableDay(availableVacationVOList);
        availableVacationVOList = udcUtil.translateList(availableVacationVOList);
        if (vacationId != null) {
            availableVacationVOList.removeIf(userVacationVO -> (userVacationVO.getAvailableDays().compareTo(new BigDecimal(0)) < 1 || userVacationVO.getExpirationDate().isBefore(LocalDate.now())) && !Objects.equals(userVacationVO.getId(), vacationId));
        } else {
            availableVacationVOList.removeIf(userVacationVO -> userVacationVO.getAvailableDays().compareTo(new BigDecimal(0)) < 1 || userVacationVO.getExpirationDate().isBefore(LocalDate.now()));
        }
        return availableVacationVOList;
    }

    @Override
    @Transactional
    public void deleteAllByUserId(Long userId) {
        userVacationDAO.deleteAllByUserId(userId);
    }

    @Override
    public void tmpDownload(HttpServletResponse response) {
        Workbook workbook = getVol();
        String fileName = "假期导入模板-" + LocalDate.now();
        ExcelUtil.writeResponse(response, fileName, workbook);
    }

    @Override
    @Transactional
    public String importBatch(MultipartFile file, Boolean force) {
        if (file == null) {
            throw TwException.error("", "上传文件异常");
        }
        Workbook workbook = null;
        try {
            workbook = WorkbookFactory.create(file.getInputStream());
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
            throw TwException.error("", "文件解析异常");
        }
        Sheet sheet = workbook.getSheet("假期管理");
        if (sheet == null) {
            throw TwException.error("", "表结构错误");
        }
        int dataStartRow = 1;
        List<UserVacationPayload> payloadList = new ArrayList<>();
        for (int i = dataStartRow; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);

            UserVacationPayload userVacationPayload = new UserVacationPayload();
            //工号
            String userName = ExcelUtil.getCellFormatValue(row.getCell(1));
            Long userId = cacheUtil.getUserIdByName(userName);
            if (userId == null) {
                throw TwException.error("", "第" + (i + 1) + "行员工未查询到，请确认");
            }
            //
            userVacationPayload.setUserId(userId);
            // 年度
            userVacationPayload.setVacationYear(Long.valueOf(ExcelUtil.getCellFormatValue(row.getCell(2))));
            // 假期类型
            String vacationType = ExcelUtil.getCellFormatValue(row.getCell(3));

            if (StringUtils.hasText(vacationType)) {
                PrdSystemSelectionVO prdSystemSelectionVisitGradeVO = cacheUtil.transferSystemObjByName("org:vacation_type", vacationType);
                if (prdSystemSelectionVisitGradeVO != null) {
                    userVacationPayload.setVacationType(prdSystemSelectionVisitGradeVO.getSelectionValue());
                }
            }
            // 开始日期
            String startDate = ExcelUtil.getCellFormatValue(row.getCell(4));
            if (StringUtils.hasText(startDate)) {
                userVacationPayload.setStartDate(LocalDate.parse(startDate));
            }
            //结束日期
            String endDate = ExcelUtil.getCellFormatValue(row.getCell(5));
            if (StringUtils.hasText(endDate)) {
                userVacationPayload.setEndDate(LocalDate.parse(endDate));
            }
            //有效期
            String expirationDate = ExcelUtil.getCellFormatValue(row.getCell(6));
            if (StringUtils.hasText(expirationDate)) {
                userVacationPayload.setExpirationDate(LocalDate.parse(expirationDate));
            }
            //总天数
            String totalDays = ExcelUtil.getCellFormatValue(row.getCell(7));
            if (StringUtils.hasText(totalDays)) {
                userVacationPayload.setTotalDays(new BigDecimal(totalDays));
            }
            // 已用天数
            String usedDays = ExcelUtil.getCellFormatValue(row.getCell(8));
            if (StringUtils.hasText(usedDays)) {
                userVacationPayload.setUsedDays(new BigDecimal(usedDays));
            }
            // 备注
            userVacationPayload.setRemark(ExcelUtil.getCellFormatValue(row.getCell(9)));
            payloadList.add(userVacationPayload);

        }
        //保存数据
        if (!CollectionUtils.isEmpty(payloadList)) {
            List<UserVacationDO> userVacationDOS = UserVacationConvert.INSTANCE.payloadListToDoList(payloadList);
            repo.saveAll(userVacationDOS);
        }
        return "";
    }

    public Workbook getVol() {
        ClassPathResource classPathResource = new ClassPathResource("template/userVacationBatch.xlsx");
        try {
            InputStream inputStream = classPathResource.getInputStream();
            Workbook workbook = WorkbookFactory.create(inputStream);
            return workbook;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public void saveRewardLeave(UserVacationPayload payload) {
        UserVacationDO vacationDO = repo.findByReasonIdAndReasonTypeAndUserId(payload.getReasonId(), payload.getReasonType(), payload.getUserId());
        if (!ObjectUtils.isEmpty(vacationDO)) {
            BigDecimal totalDays = payload.getTotalDays().add(vacationDO.getTotalDays());
            vacationDO.setTotalDays(totalDays);
            userVacationDAO.save(vacationDO);
        } else {
            userVacationDAO.save(UserVacationConvert.INSTANCE.payloadToDo(payload));
        }
        
    }
}
