package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.BuProjectPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.BuProjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.BuProjectVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.BuProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QBuProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QBuProjectMembersDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.BuProjectRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * BU项目管理
 *
 * @author carl
 * @date 2023-11-29
 */
@Repository
@RequiredArgsConstructor
public class BuProjectDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BuProjectRepo repo;
    private final QBuProjectDO qdo = QBuProjectDO.buProjectDO;
    private final QBuProjectMembersDO qdoMembers = QBuProjectMembersDO.buProjectMembersDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BuProjectVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BuProjectVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // BU_ID
                qdo.buId,
                // 科目模板id
                qdo.projYear,
                // 项目编号
                qdo.projNo,
                // 项目名称
                qdo.projName,
                // 项目状态
                qdo.projStatus,
                // 项目模板id
                qdo.projTempId,
                // 科目模板id
                qdo.subjectTempId,
                // 预计开始日期
                qdo.planStartDate,
                // 预计结束日期
                qdo.planEndDate,
                // 项目经理资源id
                qdo.pmResId,
                // 项目经理当量系数
                qdo.pmEqvaRatio,
                // 主交付BU_ID
                qdo.deliBuId,
                // 交付用户id
                qdo.deliUserId,
                // 预算总人天
                qdo.totalDays,
                // 总当量
                qdo.totalEqva,
                // 费用总预算
                qdo.totalReimbursement,
                // 当量预估单价
                qdo.eqvaPrice,
                // 费用总成本（当量成本+费用总预算）
                qdo.totalCost,
                // 关闭原因
                qdo.closeReason,
                // 项目进度状态
                qdo.projProcessStatus,
                // 自动汇报标识
                qdo.autoReportFlag,
                // 扩展字段1
                qdo.ext1,
                // 扩展字段2
                qdo.ext2,
                // 扩展字段3
                qdo.ext3,
                // 扩展字段4
                qdo.ext4,
                // 扩展字段5
                qdo.ext5
        )).from(qdo).leftJoin(qdoMembers).on(qdoMembers.projId.eq(qdo.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BuProjectVO> getJpaQueryWhere(BuProjectQuery query) {
        JPAQuery<BuProjectVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));

        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        //分组去重
        jpaQuery.groupBy(qdo.id);
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BuProjectQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qdoMembers).on(qdoMembers.projId.eq(qdo.id));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        //分组去重
        jpaQuery.groupBy(qdo.id);
        List<Long> fetch = jpaQuery.fetch();
        return fetch.size();
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BuProjectQuery query) {
        List<Predicate> list = whereCommon(query);
        //权限判断
        if (query.getPermissionFlag()) {
            list.add(qdoMembers.deleteFlag.eq(0));
            list.add(qdoMembers.resId.eq(query.getLoginUserId()));
            BooleanExpression orExpression = qdo.pmResId.eq(query.getLoginUserId()).or(qdo.deliUserId.eq(query.getLoginUserId()));
            list.add(orExpression);
        }
        //权限判断
        if (query.getPermissionFlagForTimesheet()) {
            list.add(qdoMembers.deleteFlag.eq(0));
            list.add(qdoMembers.resId.eq(query.getLoginUserId()));
        }
        return ExpressionUtils.allOf(list);
    }

    @NotNull
    private List<Predicate> whereCommon(BuProjectQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        if (!ObjectUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 主交付BU_ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBuId())) {
            list.add(qdo.buId.eq(query.getBuId()));
        }
        /** 科目模板id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjYear())) {
            list.add(qdo.projYear.eq(query.getProjYear()));
        }
        /** 项目编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjNo())) {
            list.add(qdo.projNo.like(SqlUtil.toSqlLikeString(query.getProjNo())));
        }
        /** 项目名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjName())) {
            list.add(qdo.projName.like(SqlUtil.toSqlLikeString(query.getProjName())));
        }
        /** 项目状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProjStatus())) {
            list.add(qdo.projStatus.eq(query.getProjStatus()));
        }
        /** 项目模板id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjTempId())) {
            list.add(qdo.projTempId.eq(query.getProjTempId()));
        }
        /** 科目模板id 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectTempId())) {
            list.add(qdo.subjectTempId.eq(query.getSubjectTempId()));
        }
        /** 预计开始日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanStartDate())) {
            list.add(qdo.planStartDate.eq(query.getPlanStartDate()));
        }
        /** 预计结束日期 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanEndDate())) {
            list.add(qdo.planEndDate.eq(query.getPlanEndDate()));
        }
        /** 项目经理资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getPmResId())) {
            list.add(qdo.pmResId.eq(query.getPmResId()));
        }
        /** 项目经理当量系数 精确 */
        if (!ObjectUtils.isEmpty(query.getPmEqvaRatio())) {
            list.add(qdo.pmEqvaRatio.eq(query.getPmEqvaRatio()));
        }
        /** 主交付BU_ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliBuId())) {
            list.add(qdo.deliBuId.eq(query.getDeliBuId()));
        }
        /** 交付用户id 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliUserId())) {
            list.add(qdo.deliUserId.eq(query.getDeliUserId()));
        }
        /** 预算总人天 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalDays())) {
            list.add(qdo.totalDays.eq(query.getTotalDays()));
        }
        /** 总当量 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalEqva())) {
            list.add(qdo.totalEqva.eq(query.getTotalEqva()));
        }
        /** 费用总预算 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalReimbursement())) {
            list.add(qdo.totalReimbursement.eq(query.getTotalReimbursement()));
        }
        /** 当量预估单价 精确 */
        if (!ObjectUtils.isEmpty(query.getEqvaPrice())) {
            list.add(qdo.eqvaPrice.eq(query.getEqvaPrice()));
        }
        /** 费用总成本（当量成本+费用总预算） 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalCost())) {
            list.add(qdo.totalCost.eq(query.getTotalCost()));
        }
        /** 关闭原因 精确 */
        if (!ObjectUtils.isEmpty(query.getCloseReason())) {
            list.add(qdo.closeReason.eq(query.getCloseReason()));
        }
        /** 项目进度状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProjProcessStatus())) {
            list.add(qdo.projProcessStatus.eq(query.getProjProcessStatus()));
        }
        /** 自动汇报标识 精确 */
        if (!ObjectUtils.isEmpty(query.getAutoReportFlag())) {
            list.add(qdo.autoReportFlag.eq(query.getAutoReportFlag()));
        }
        /** 扩展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 扩展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 扩展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 扩展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 扩展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return list;
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BuProjectVO queryByKey(Long id) {
        JPAQuery<BuProjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BuProjectVO> queryListDynamic(BuProjectQuery query) {
        JPAQuery<BuProjectVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BuProjectVO> queryPaging(BuProjectQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BuProjectVO> jpaQuery = getJpaQueryWhere(query);
        List<BuProjectVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BuProjectVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BuProjectDO save(BuProjectDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BuProjectDO> saveAll(List<BuProjectDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BuProjectPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 主交付BU_ID
        if (payload.getBuId() != null) {
            update.set(qdo.buId, payload.getBuId());
        }
        // 科目模板id
        if (payload.getProjYear() != null) {
            update.set(qdo.projYear, payload.getProjYear());
        }
        // 项目编号
        if (payload.getProjNo() != null) {
            update.set(qdo.projNo, payload.getProjNo());
        }
        // 项目名称
        if (payload.getProjName() != null) {
            update.set(qdo.projName, payload.getProjName());
        }
        // 项目状态
        if (payload.getProjStatus() != null) {
            update.set(qdo.projStatus, payload.getProjStatus());
        }
        // 项目模板id
        if (payload.getProjTempId() != null) {
            update.set(qdo.projTempId, payload.getProjTempId());
        }
        // 科目模板id
        if (payload.getSubjectTempId() != null) {
            update.set(qdo.subjectTempId, payload.getSubjectTempId());
        }
        // 预计开始日期
        if (payload.getPlanStartDate() != null) {
            update.set(qdo.planStartDate, payload.getPlanStartDate());
        }
        // 预计结束日期
        if (payload.getPlanEndDate() != null) {
            update.set(qdo.planEndDate, payload.getPlanEndDate());
        }
        // 项目经理资源id
        if (payload.getPmResId() != null) {
            update.set(qdo.pmResId, payload.getPmResId());
        }
        // 项目经理当量系数
        if (payload.getPmEqvaRatio() != null) {
            update.set(qdo.pmEqvaRatio, payload.getPmEqvaRatio());
        }
        // 主交付BU_ID
        if (payload.getDeliBuId() != null) {
            update.set(qdo.deliBuId, payload.getDeliBuId());
        }
        // 交付用户id
        if (payload.getDeliUserId() != null) {
            update.set(qdo.deliUserId, payload.getDeliUserId());
        }
        // 预算总人天
        if (payload.getTotalDays() != null) {
            update.set(qdo.totalDays, payload.getTotalDays());
        }
        // 总当量
        if (payload.getTotalEqva() != null) {
            update.set(qdo.totalEqva, payload.getTotalEqva());
        }
        // 费用总预算
        if (payload.getTotalReimbursement() != null) {
            update.set(qdo.totalReimbursement, payload.getTotalReimbursement());
        }
        // 当量预估单价
        if (payload.getEqvaPrice() != null) {
            update.set(qdo.eqvaPrice, payload.getEqvaPrice());
        }
        // 费用总成本（当量成本+费用总预算）
        if (payload.getTotalCost() != null) {
            update.set(qdo.totalCost, payload.getTotalCost());
        }
        // 关闭原因
        if (payload.getCloseReason() != null) {
            update.set(qdo.closeReason, payload.getCloseReason());
        }
        // 项目进度状态
        if (payload.getProjProcessStatus() != null) {
            update.set(qdo.projProcessStatus, payload.getProjProcessStatus());
        }
        // 自动汇报标识
        if (payload.getAutoReportFlag() != null) {
            update.set(qdo.autoReportFlag, payload.getAutoReportFlag());
        }
        // 扩展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 扩展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 扩展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 扩展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 扩展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 主交付BU_ID
            if (nullFields.contains("buId")) {
                update.setNull(qdo.buId);
            }
            // 科目模板id
            if (nullFields.contains("projYear")) {
                update.setNull(qdo.projYear);
            }
            // 项目编号
            if (nullFields.contains("projNo")) {
                update.setNull(qdo.projNo);
            }
            // 项目名称
            if (nullFields.contains("projName")) {
                update.setNull(qdo.projName);
            }
            // 项目状态
            if (nullFields.contains("projStatus")) {
                update.setNull(qdo.projStatus);
            }
            // 项目模板id
            if (nullFields.contains("projTempId")) {
                update.setNull(qdo.projTempId);
            }
            // 科目模板id
            if (nullFields.contains("subjectTempId")) {
                update.setNull(qdo.subjectTempId);
            }
            // 预计开始日期
            if (nullFields.contains("planStartDate")) {
                update.setNull(qdo.planStartDate);
            }
            // 预计结束日期
            if (nullFields.contains("planEndDate")) {
                update.setNull(qdo.planEndDate);
            }
            // 项目经理资源id
            if (nullFields.contains("pmResId")) {
                update.setNull(qdo.pmResId);
            }
            // 项目经理当量系数
            if (nullFields.contains("pmEqvaRatio")) {
                update.setNull(qdo.pmEqvaRatio);
            }
            // 主交付BU_ID
            if (nullFields.contains("deliBuId")) {
                update.setNull(qdo.deliBuId);
            }
            // 交付用户id
            if (nullFields.contains("deliUserId")) {
                update.setNull(qdo.deliUserId);
            }
            // 预算总人天
            if (nullFields.contains("totalDays")) {
                update.setNull(qdo.totalDays);
            }
            // 总当量
            if (nullFields.contains("totalEqva")) {
                update.setNull(qdo.totalEqva);
            }
            // 费用总预算
            if (nullFields.contains("totalReimbursement")) {
                update.setNull(qdo.totalReimbursement);
            }
            // 当量预估单价
            if (nullFields.contains("eqvaPrice")) {
                update.setNull(qdo.eqvaPrice);
            }
            // 费用总成本（当量成本+费用总预算）
            if (nullFields.contains("totalCost")) {
                update.setNull(qdo.totalCost);
            }
            // 关闭原因
            if (nullFields.contains("closeReason")) {
                update.setNull(qdo.closeReason);
            }
            // 项目进度状态
            if (nullFields.contains("projProcessStatus")) {
                update.setNull(qdo.projProcessStatus);
            }
            // 自动汇报标识
            if (nullFields.contains("autoReportFlag")) {
                update.setNull(qdo.autoReportFlag);
            }
            // 扩展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 扩展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 扩展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 扩展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 扩展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据BU和年度查询
     *
     * @param buId bu
     * @param year 年度
     * @return 结果
     */
    public BuProjectVO queryByBuIdAndYear(Long buId, Integer year) {
        JPAQuery<BuProjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.buId.eq(buId));
        jpaQuery.where(qdo.projYear.eq(year));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public List<BuProjectVO> queryListDynamicSimple(BuProjectQuery query) {
        JPAQuery<BuProjectVO> jpaQuery = jpaQueryFactory.select(Projections.bean(BuProjectVO.class,
            qdo.id,
            qdo.remark,
            qdo.createUserId,
            qdo.creator,
            qdo.createTime,
            //qdo.modifyUserId,
            //qdo.updater,
            //qdo.modifyTime,
            // BU_ID
            qdo.buId,
            // 科目模板id
            qdo.projYear,
            // 项目编号
            qdo.projNo,
            // 项目名称
            qdo.projName,
            // 项目状态
            qdo.projStatus,
            // 项目模板id
            qdo.projTempId,
            // 科目模板id
            qdo.subjectTempId,
            // 预计开始日期
            qdo.planStartDate,
            // 预计结束日期
            qdo.planEndDate,
            // 项目经理资源id
            qdo.pmResId,
            // 项目经理当量系数
            qdo.pmEqvaRatio,
            // 主交付BU_ID
            qdo.deliBuId,
            // 交付用户id
            qdo.deliUserId,
            // 预算总人天
            qdo.totalDays,
            // 总当量
            qdo.totalEqva,
            // 费用总预算
            qdo.totalReimbursement,
            // 当量预估单价
            qdo.eqvaPrice,
            // 费用总成本（当量成本+费用总预算）
            qdo.totalCost,
            // 关闭原因
            qdo.closeReason,
            // 项目进度状态
            qdo.projProcessStatus,
            // 自动汇报标识
            qdo.autoReportFlag,
            // 扩展字段1
            qdo.ext1,
            // 扩展字段2
            qdo.ext2,
            // 扩展字段3
            qdo.ext3,
            // 扩展字段4
            qdo.ext4,
            // 扩展字段5
            qdo.ext5
        )).from(qdo);
        List<Predicate> predicates = whereCommon(query);
        Predicate predicate = ExpressionUtils.allOf(predicates);
        jpaQuery.where(predicate);
        return jpaQuery.fetch();
    }
}

